package com.qiho.center.biz.runnable;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import com.qiho.center.api.dto.order.DeliveryRecordDto;
import com.qiho.center.api.enums.ErpStatusEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderBatchDeliveryEnum;
import com.qiho.center.api.enums.YTOLogisticsStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.OrderFileDeliveryParam;
import com.qiho.center.api.util.BizLog;
import com.qiho.center.biz.bo.ExpressBo;
import com.qiho.center.biz.bo.KuaiDi100Bo;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.job.OrderFileDeliveryJob;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.biz.service.order.OrderFileDeliveryService;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.daoh.qiho.order.DeliveryRecordMapper;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.entityd.qiho.order.DeliveryRecordEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.SequenceNoBuilder;

import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * 批量上传发物流单号-批量发货
 *
 * @Created by zhangshun on 2018-03-10 15:20:17
 */
@Component
@Scope("prototype")
public class OrderFileConfirmDeliveryRunnable implements Runnable {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrderFileConfirmDeliveryRunnable.class);

    private static final String NUM_AND_LETTER_REG_EX = "^[A-Za-z0-9]{1,40}$";

    /**
     * 任务ID
     */
    private final Long recordId;

    /**
     * 商家ID
     */
    private final Long merchantId;

    /**
     * 总条数
     */
    private final Integer totalCount;

    /**
     * 物流编码
     */
    private final String logisticsCode;

    /**
     * 物流名称
     */
    private final String logisticsName;

    /**
     * 工作统计 CountDownLatch
     */
    private final CountDownLatch countDownLatch;

    /**
     * 物流记录
     */
    private final List<OrderFileDeliveryParam> deliveryBeanList;

    private final OrderFileDeliveryJob orderFileDeliveryJob;

    @Autowired
    private SequenceNoBuilder builder;

    @Autowired
    private KuaiDi100Bo kuaiDi100Bo;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private ErpOrderService erpOrderService;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    @Autowired
    private LogisticsOrderService logisticsOrderService;

    @Resource
    private ExpressBo expressBo;

    @Resource
    private OrderFileDeliveryService orderFileDeliveryService;

    @Resource
    private DeliveryRecordMapper deliveryRecordMapper;

    /**
     * 重新订阅快递单号
     */
    private final ThreadLocal<List<LogisticsOrderEntity>> logisticsOrderEntityListLocal = new ThreadLocal() {
        @Override
        protected List<LogisticsOrderEntity> initialValue() {
            return new ArrayList<>();
        }
    };

    public OrderFileConfirmDeliveryRunnable(OrderFileDeliveryJob job, DeliveryRecordDto recordDto, String logisticsName,
        CountDownLatch countDownLatch, List<OrderFileDeliveryParam> deliveryBeanList) {
        this.orderFileDeliveryJob = job;
        this.countDownLatch = countDownLatch;

        this.recordId = recordDto.getId();
        this.merchantId = recordDto.getMerchantId();
        this.totalCount = recordDto.getTotalCount();
        this.logisticsCode = recordDto.getLogisticsCode();
        this.logisticsName = logisticsName;
        this.deliveryBeanList = deliveryBeanList;
    }

    @Override
    public void run() {

        // 检查参数
        if (!checkCommonParam()) {
            return;
        }

        int failedCount = 0;
        int updatedCount = 0;
        int succeedCount = 0;

        try {

            for (OrderFileDeliveryParam bean : deliveryBeanList) {
                try {

                    //检验业务参数
                    boolean checkFlag = this.checkParam(bean);
                    if (!checkFlag) {
                        failedCount++;
                        orderFileDeliveryJob.setConfirmFailId(bean.getRowNumber());
                        continue;
                    }

                    // 主逻辑
                    OrderBatchDeliveryEnum result = this.auditOrder(bean);

                    // 更新成功
                    if (OrderBatchDeliveryEnum.UPDATED == result) {
                        updatedCount++;
                    }
                    // 确认成功
                    else if (OrderBatchDeliveryEnum.SUCCEED == result) {
                        succeedCount++;
                    }
                    // 更新失败
                    else if (OrderBatchDeliveryEnum.FAILED == result) {
                        failedCount++;
                        orderFileDeliveryJob.setConfirmFailId(bean.getRowNumber());
                    }

                    BizLog.log("[确认发货] {} recordId:{}, merchantId:{}, result:{}, {}", Thread.currentThread().getName(),
                        recordId, merchantId, result.getName(), bean);

                } catch (Exception e) {
                    failedCount++;
                    orderFileDeliveryJob.setConfirmFailId(bean.getRowNumber());
                    LOGGER.error("[确认发货]错误。{}, recordId:{}, merchantId:{}, {}", Thread.currentThread().getName(),
                        recordId, merchantId, bean, e);
                }
            }

        } finally {

            orderFileDeliveryJob.setConfirmFailedCount(failedCount);
            orderFileDeliveryJob.setConfirmSucceedCount(succeedCount);
            orderFileDeliveryJob.setConfirmUpdatedCount(updatedCount);

            completeHandler(failedCount, succeedCount, updatedCount);

            //重新订阅快递100
            try {
                expressBo.subscribeExpress(BeanUtils.copyList(logisticsOrderEntityListLocal.get(),LogisticsOrderDo.class));
            } catch (Exception e) {
                LOGGER.error("[确认发货]订阅快递100接口错误", e);
            }

            countDownLatch.countDown();
        }
    }

    /**
     * 检查参数
     *
     * @return
     */
    public boolean checkCommonParam() {

        if (CollectionUtils.isEmpty(deliveryBeanList) || Objects.equals(null, recordId) || Objects
            .equals(null, merchantId)) {
            BizLog.log("[确认发货]参数错误，数据、任务ID、商家ID为空");
            return false;
        }

        if (StringUtils.isBlank(logisticsCode) || StringUtils.isBlank(logisticsName) || Objects.equals(null, totalCount)
            || totalCount.equals(0)) {
            BizLog.log("[确认发货]参数错误，物流编码、物流名称、总条数为空");
            return false;
        }

        return true;
    }

    /**
     * 检查参数
     *
     * @param bean
     * @return
     */
    public boolean checkParam(OrderFileDeliveryParam bean) {

        //格式不完整 不处理 返回失败
        if (StringUtils.isBlank(bean.getPostId()) || StringUtils.isBlank(bean.getErpId()) || Objects
            .isNull(bean.getRowNumber())) {
            return false;
        }

        // 忽略大小写的写法
        Pattern pattern = Pattern.compile(NUM_AND_LETTER_REG_EX, Pattern.CASE_INSENSITIVE);
        Matcher matcher = pattern.matcher(bean.getPostId().trim());
        // 字符串是否与正则表达式相匹配
        if (!matcher.matches()) {//数字和字母的检验
            return false;
        }

        return true;
    }

    /**
     * 单个同步物流信息
     *
     * @param bean
     * @return 说明：
     * 0 = 更新成功
     * 1 = 失败
     * 2 = 确认成功
     */
    @Transactional(rollbackFor = Exception.class)
    public OrderBatchDeliveryEnum auditOrder(OrderFileDeliveryParam bean) {

        // 查询ERP订单信息
        final QihoErpOrderEntity erpOrderEntity = qihoErpOrderDAO.findByErpId(bean.getErpId());
        if (null == erpOrderEntity) {
            BizLog.log("[确认发货]查询ERP订单信息, erpId:{}", bean.getErpId());
            return OrderBatchDeliveryEnum.FAILED;
        }

        if (!isMerchantByOrderId(erpOrderEntity.getOrderId(), String.valueOf(merchantId))) {
            BizLog.log("[确认发货]订单与商家不对应, orderId:{}, merchantId:{}", erpOrderEntity.getOrderId(), merchantId);
            return OrderBatchDeliveryEnum.FAILED;
        }

        LogisticsOrderEntity logisticsOrderEntity = qihoLogisticsOrderDAO.findByErpId(bean.getErpId());

        //物流子订单号已经存在  则更新物流单号
        if (logisticsOrderEntity != null) {

            LogisticsOrderEntity entity = new LogisticsOrderEntity();
            entity.setLogisticsCode(logisticsCode);
            entity.setLogisticsName(logisticsName);
            entity.setPostId(bean.getPostId());
            entity.setErpId(bean.getErpId());

            //更新物流表和快照表的信息
            if (0 == qihoLogisticsOrderDAO.updateByErpId(entity)) {
                throw new QihoException("更新物流表和快照表的信息失败");
            }

            QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
            orderSnapshot.setLogisticsCode(logisticsCode);
            orderSnapshot.setLogisticsName(logisticsName);
            orderSnapshot.setPostId(bean.getPostId());
            orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());

            // 更新订单快照信息
            if (0 == qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot)) {
                throw new QihoException("更新订单快照信息");
            }

            // 重新订阅快递100
            LogisticsOrderEntity logisticsOrderEntity1 = new LogisticsOrderEntity();
            logisticsOrderEntity1.setLogisticsCode(logisticsCode);
            logisticsOrderEntity1.setPostId(bean.getPostId());

            logisticsOrderEntityListLocal.get().add(logisticsOrderEntity1);

            // 返回更新成功
            return OrderBatchDeliveryEnum.UPDATED;
        } else {
            final LogisticsOrderEntity entity = this.getLogisticsEntity(bean, erpOrderEntity);

            //订单置为返回成功
            try {
                erpOrderService
                    .updateErpStatus(entity.getErpId(), ErpStatusEnum.SUCCESS, ErpStatusEnum.TO_NOTIFY, null);

                logisticsOrderService.createLogisticsOrderBatchDelivery(entity);

                LogisticsOrderEntity logisticsOrderEntity1 = new LogisticsOrderEntity();
                logisticsOrderEntity1.setLogisticsCode(logisticsCode);
                logisticsOrderEntity1.setPostId(bean.getPostId());

                logisticsOrderEntityListLocal.get().add(logisticsOrderEntity1);
            } catch (Exception e) {
                throw new QihoException("[确认发货]处理错误");
            }

            // 返回确认成功
            return OrderBatchDeliveryEnum.SUCCEED;
        }
    }

    /**
     * 生成物流子订单实体类
     *
     * @param bean
     * @param erpOrder
     * @return
     */
    private LogisticsOrderEntity getLogisticsEntity(OrderFileDeliveryParam bean, QihoErpOrderEntity erpOrder) {

        LogisticsOrderEntity entity = new LogisticsOrderEntity();

        entity.setErpId(bean.getErpId());
        String postId = bean.getPostId();

        int num = postId.indexOf(',');
        if (num > -1) {
            bean.setPostId(postId.substring(0, num).trim());
        } else {
            entity.setPostId(postId.trim());
        }

        entity.setLogisticsCode(logisticsCode);
        entity.setLogisticsName(logisticsName);

        entity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getDesc());
        // 物流子订单初始化状态为接单
        entity.setOrderStatus(LogisticsOrderStatusEnum.ACCEPT.getCode());
        entity.setConsumerName(erpOrder.getConsumerName());
        entity.setItemId(erpOrder.getItemId());
        entity.setItemName(erpOrder.getItemName());
        entity.setItemNo(erpOrder.getItemNo());
        entity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, erpOrder.getItemId()));
        entity.setMobile(erpOrder.getMobile());
        entity.setOrderId(erpOrder.getOrderId());
        entity.setSkuNo(erpOrder.getSkuNo());
        entity.setProvince(erpOrder.getProvince());

        return entity;
    }

    /**
     * 判断订单是否这个商家
     *
     * @param orderId
     * @return
     */
    private boolean isMerchantByOrderId(String orderId, String merchantId) {
        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if (null == entity) {
            return Boolean.FALSE;
        }

        if (!StringUtils.equals(String.valueOf(entity.getMerchantId()), merchantId)) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    private void completeHandler(int failedCount, int succeedCount, int updatedCount) {

        DeliveryRecordEntity recordEntity = this.getRecordEntity(recordId);

        DeliveryRecordEntity entity = new DeliveryRecordEntity();
        entity.setId(recordId);

        if (null == recordEntity) {

            entity.setFailedCount(failedCount);
            entity.setSucceedCount(succeedCount);
            entity.setUpdatedCount(updatedCount);

        } else {
            entity.setFailedCount(failedCount + recordEntity.getFailedCount());
            entity.setSucceedCount(succeedCount + recordEntity.getSucceedCount());
            entity.setUpdatedCount(updatedCount + recordEntity.getUpdatedCount());
        }

        // 更新数据库
        if (!orderFileDeliveryService.updateByIdSelective(entity)) {
            LOGGER.error("[确认发货]更新失败, id:{}", entity.getId());
        }
    }

    private DeliveryRecordEntity getRecordEntity(Long id) {
        return deliveryRecordMapper.selectById(id);
    }

}
