/**
 * Project Name:qiho-center-biz File Name:ItemSkuServiceImpl.java Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月20日下午6:01:54 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl;

import cn.com.duiba.stock.service.api.dto.NewStockReqDto;
import cn.com.duiba.stock.service.api.dto.NewStockRespDto;
import cn.com.duiba.stock.service.api.dto.UpdateStockReqDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ItemSkuDto;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.biz.event.WarningEvent;
import com.qiho.center.biz.service.ItemSkuService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.daoh.qiho.QihoItemSkuMapper;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.support.BizEventBus;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * ClassName:ItemSkuServiceImpl <br/>
 * Date: 2017年5月20日 下午6:01:54 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class ItemSkuServiceImpl implements ItemSkuService {

    @Autowired
    private QihoItemSkuDAO            qihoItemSkuDAO;

    @Autowired
    private RemoteStockService        remoteStockService;

    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    @Autowired
    private QihoItemSkuMapper qihoItemSkuMapper;

    @Autowired
    private BizEventBus eventBus;

    /**
     * @see com.qiho.center.biz.service.ItemSkuService#getSkuByItemId(java.lang.Long)
     */
    @Override
    public List<ItemSkuDto> getSkuByItemId(Long itemId) {
        List<QihoItemSkuEntity> skuEntityList = qihoItemSkuDAO.getSkuByItemIds(Lists.newArrayList(itemId));
        List<ItemSkuDto> skuDtoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(skuEntityList)) {
            return skuDtoList;
        }
        List<Long> stockIds = Lists.newArrayList();
        for (QihoItemSkuEntity skuEntity : skuEntityList) {
            stockIds.add(skuEntity.getStockId());
        }
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
        for (QihoItemSkuEntity skuEntity : skuEntityList) {
            ItemSkuDto skuDto = BeanUtils.copy(skuEntity, ItemSkuDto.class);
            skuDto.setStock(stockMap.get(skuEntity.getStockId()));
            skuDto.setStockId(skuEntity.getStockId());
            skuDtoList.add(skuDto);
        }
        return skuDtoList;
    }

    /**
     * @see com.qiho.center.biz.service.ItemSkuService#updateItemSku(java.util.List, java.lang.Long)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateItemSku(List<ItemSkuDto> skuList, Long itemId) {
        if (CollectionUtils.isEmpty(skuList) || itemId == null) {
            return;
        }
        //判断是否存在相同的商品编号
        List<String> sameList = Lists.newArrayList();

        List<Long> skuIds = Lists.newArrayList();
        // 要创建的sku列表
        List<QihoItemSkuEntity> createSkuList = Lists.newArrayList();
        // 要更新的sku列表
        List<QihoItemSkuEntity> modifySkuList = Lists.newArrayList();
        for (ItemSkuDto skuDto : skuList) {
            if(sameList.contains(skuDto.getSkuNo())){
                throw new QihoException("不允许存在相同编号");
            }
            sameList.add(skuDto.getSkuNo());
            QihoItemSkuEntity skuEntity = BeanUtils.copy(skuDto, QihoItemSkuEntity.class);
            if (skuDto.getId() != null) {
                skuIds.add(skuDto.getId());
                modifySkuList.add(skuEntity);
            } else {
                skuEntity.setItemId(itemId);
                skuEntity.setDeleted(false);
                createSkuList.add(skuEntity);
            }
        }
        if (CollectionUtils.isNotEmpty(skuIds)) {
            // 删除sku
            qihoItemSkuDAO.deleteNotIn(skuIds, itemId);
        }else{
            qihoItemSkuDAO.deleteByItemId(itemId);
        }
        // 新建sku
        createSku(createSkuList);
        // 更新sku
        modifySku(modifySkuList);
    }

    /**
     * @author zhanglihui
     * @param modifySkuList
     */
    private void modifySku(List<QihoItemSkuEntity> modifySkuList) {
        if(CollectionUtils.isEmpty(modifySkuList)){
            return;
        }
        // 批量更新sku
        List<UpdateStockReqDto> decreaseList = Lists.newArrayList();
        List<UpdateStockReqDto> increaseList = Lists.newArrayList();

        List<String> skuNos = getSkuNo(modifySkuList);
        Map<String, com.qiho.center.common.entityd.qiho.QihoItemSkuEntity> skuNoStockMaps =  qihoItemSkuMapper.selectSkuNoStockIds(skuNos);

        // 批量创建库存，并把库存ID回写到SKU表
        List<NewStockReqDto> stockReqList = Lists.newArrayList();
        Map<Long, QihoItemSkuEntity> skuMap = Maps.newHashMap();

        executeStock(modifySkuList, decreaseList, increaseList, skuNoStockMaps, stockReqList, skuMap);
        //商品编码如果是相同的 库存ID相同
        if(CollectionUtils.isNotEmpty(stockReqList)){
            List<NewStockRespDto> stockRespList = remoteStockBackendService.newStockBatch(stockReqList).getResult();
            for (NewStockRespDto stockResp : stockRespList) {
                QihoItemSkuEntity skuEntity = skuMap.get(stockResp.getGid());
                skuEntity.setStockId(stockResp.getStockId());
            }
        }

        qihoItemSkuDAO.updateBatch(modifySkuList);


        // 批量更新库存
        if(CollectionUtils.isNotEmpty(decreaseList)){
            remoteStockBackendService.decreaseItemStockBatch(decreaseList);
        }
        if(CollectionUtils.isNotEmpty(increaseList)){
            remoteStockBackendService.increaseItemStockBatch(increaseList);
        }

        for (QihoItemSkuEntity qihoItemSkuEntity : modifySkuList) {
            WarningEvent warningEvent = new WarningEvent();
            warningEvent.setSkuId(qihoItemSkuEntity.getId());
            eventBus.post(warningEvent);
        }
    }

    private void executeStock(List<QihoItemSkuEntity> modifySkuList, List<UpdateStockReqDto> decreaseList, List<UpdateStockReqDto> increaseList, Map<String, com.qiho.center.common.entityd.qiho.QihoItemSkuEntity> skuNoStockMaps, List<NewStockReqDto> stockReqList, Map<Long, QihoItemSkuEntity> skuMap) {
        for (QihoItemSkuEntity skuEntity : modifySkuList) {
            if(skuNoStockMaps.containsKey(skuEntity.getSkuNo())){
                //已经存在的修改 存在两种情况
                // 1、商品编码发生了修改 需要修改stockId
                long stockId = skuNoStockMaps.get(skuEntity.getSkuNo()).getStockId();
                skuEntity.setStockId(stockId);
                DubboResult<Long> stockNum = remoteStockBackendService.find(stockId);
                long changeStock = skuEntity.getStock()-stockNum.getResult();
                if(changeStock == 0) continue;
                UpdateStockReqDto updateStockReqDto = new UpdateStockReqDto();
                updateStockReqDto.setStockId(stockId);
                updateStockReqDto.setStock(Math.abs(changeStock));
                if(changeStock>0){
                    increaseList.add(updateStockReqDto);
                }else{
                    decreaseList.add(updateStockReqDto);
                }
            }else {
                skuMap.put(skuEntity.getId(), skuEntity);
                NewStockReqDto stockReq = new NewStockReqDto();
                //根据商品编号获取数字商品编号
                stockReq.setGid(skuEntity.getId());
                stockReq.setStock(skuEntity.getStock());
                stockReqList.add(stockReq);
            }
        }
    }

    /**
     * @author zhanglihui
     * @param createSkuList
     */
    private void createSku(List<QihoItemSkuEntity> createSkuList) {
        if (CollectionUtils.isEmpty(createSkuList)) {
            return;
        }
        qihoItemSkuDAO.insertBatch(createSkuList);
        // 批量创建库存，并把库存ID回写到SKU表
        List<NewStockReqDto> stockReqList = Lists.newArrayList();
        Map<Long, QihoItemSkuEntity> skuMap = Maps.newHashMap();

        List<String> skuNos = getSkuNo(createSkuList);
        //新增的sku  存在两种情况 如果已经存在 获取stockid 修改库存
        //如果sku不存在  则调用接口进行新增
        Map<String, com.qiho.center.common.entityd.qiho.QihoItemSkuEntity> skuNoStockMaps =  qihoItemSkuMapper.selectSkuNoStockIds(skuNos);
        List<UpdateStockReqDto> decreaseList = Lists.newArrayList();
        List<UpdateStockReqDto> increaseList = Lists.newArrayList();

        updateHandlerSku(createSkuList, stockReqList, skuMap, skuNoStockMaps, decreaseList, increaseList);
        //商品编码如果是相同的 库存ID相同
        if(CollectionUtils.isNotEmpty(stockReqList)){
            List<NewStockRespDto> stockRespList = remoteStockBackendService.newStockBatch(stockReqList).getResult();
            for (NewStockRespDto stockResp : stockRespList) {
                QihoItemSkuEntity skuEntity = skuMap.get(stockResp.getGid());
                skuEntity.setStockId(stockResp.getStockId());
            }
        }
        // 批量更新库存
        if(CollectionUtils.isNotEmpty(decreaseList)){
            remoteStockBackendService.decreaseItemStockBatch(decreaseList);
        }
        if(CollectionUtils.isNotEmpty(increaseList)){
            remoteStockBackendService.increaseItemStockBatch(increaseList);
        }
        qihoItemSkuDAO.updateBatch(createSkuList);
        for (QihoItemSkuEntity qihoItemSkuEntity : createSkuList) {
            WarningEvent warningEvent = new WarningEvent();
            warningEvent.setSkuId(qihoItemSkuEntity.getId());
            eventBus.post(warningEvent);
        }
    }

    private void updateHandlerSku(List<QihoItemSkuEntity> createSkuList, List<NewStockReqDto> stockReqList, Map<Long, QihoItemSkuEntity> skuMap, Map<String, com.qiho.center.common.entityd.qiho.QihoItemSkuEntity> skuNoStockMaps, List<UpdateStockReqDto> decreaseList, List<UpdateStockReqDto> increaseList) {
        for (QihoItemSkuEntity skuEntity : createSkuList) {
            if(skuNoStockMaps.containsKey(skuEntity.getSkuNo())){
                long stockId = skuNoStockMaps.get(skuEntity.getSkuNo()).getStockId();
                skuEntity.setStockId(stockId);
                //修改库存
                DubboResult<Long> stockNum = remoteStockBackendService.find(stockId);
                long changeStock = skuEntity.getStock()-stockNum.getResult();
                if(changeStock!=0){
                    UpdateStockReqDto  updateStockReqDto = new UpdateStockReqDto();
                    updateStockReqDto.setStockId(stockId);
                    updateStockReqDto.setStock(Math.abs(changeStock));
                    if(changeStock>0){
                        increaseList.add(updateStockReqDto);
                    }else{
                        decreaseList.add(updateStockReqDto);
                    }
                }
                continue;
            }
            skuMap.put(skuEntity.getId(), skuEntity);
            NewStockReqDto stockReq = new NewStockReqDto();
            //根据商品编号获取数字商品编号
            stockReq.setGid(skuEntity.getId());
            stockReq.setStock(skuEntity.getStock());
            stockReqList.add(stockReq);
        }
    }

    private List<String> getSkuNo(List<QihoItemSkuEntity> createSkuList) {
        List<String> result= new ArrayList<>();
        for (QihoItemSkuEntity qihoItemSkuEntity : createSkuList) {
            result.add(qihoItemSkuEntity.getSkuNo());
        }
        return result;
    }


    /**
     * @see com.qiho.center.biz.service.ItemSkuService#querySku(java.lang.Long)
     */
    @Override
    public ItemSkuDto querySku(Long skuId) {
        QihoItemSkuEntity entity = qihoItemSkuDAO.findById(skuId);
        if(entity == null){
            return null;
        }
        ItemSkuDto sku = BeanUtils.copy(entity, ItemSkuDto.class);
        DubboResult<Long> result = remoteStockService.find(entity.getStockId());
        sku.setStock(result.getResult());
        return sku;
    }

}
