package com.qiho.center.biz.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncAckDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.QueryLogisticsOrderDto;
import com.qiho.center.api.enums.ErpErrorCodeReturnEnum;
import com.qiho.center.api.enums.ErpTypeEnum;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.LogisticsNameEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.enums.YTOLogisticsStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.bo.DeliverData2TuiaBo;
import com.qiho.center.biz.bo.SmsItemFilterBo;
import com.qiho.center.biz.bo.domain.LogisticsOrderDo;
import com.qiho.center.biz.event.DeliverData2TuiaEvent;
import com.qiho.center.biz.event.KuaiDi100Event;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoChannelInfoDAO;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoChannelInfoEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.params.ErpLogisticsOrderParams;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result.WaybillCode;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.YTOClientUtil;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import static com.qiho.center.api.enums.LogisticsOrderStatusEnum.getRemarkByLogisticsStatus;

@Service
public class LogisticsOrderServiceImpl implements LogisticsOrderService {

    private static final Logger LOG = LoggerFactory.getLogger(LogisticsOrderServiceImpl.class);

    private static final ExecutorService EXECUTOR_TUIA = new ThreadPoolExecutor(20, 30, 0L,
            TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(1000));

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;

    @Autowired
    private SequenceNoBuilder builder;

    @Autowired
    private BizEventBus bizEventBus;

    @Autowired
    private QihoErpOrderDAO qihoErpOrderDAO;

    @Autowired
    private QihoOrderDAO qihoOrderDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoFundOrderDAO qihoFundOrderDAO;

    @Autowired
    private FundOrderService fundOrderService;

    @Autowired
    private ErpOrderService erpOrderService;

    @Autowired
    private QihoChannelInfoDAO qihoChannelInfoDAO;

    @Autowired
    private DeliverData2TuiaBo deliverData2TuiaBo;

    @Autowired
    private SmsItemFilterBo smsItemFilterBo;

    @Override
    public void initLogisticsOrder(ErpOrderDto erp, ResolveDeliverParams params) {
        // 构建物流同步对象
        ErpLogisticsSyncDto erpLogisticsSyncDto = new ErpLogisticsSyncDto();
        erpLogisticsSyncDto.setLogisticsCode(params.getBaiqiLogisticsCode());
        erpLogisticsSyncDto.setLogisticsName(params.getBaiqiLogisticsName());
        erpLogisticsSyncDto.setLogisticsNo(params.getExpressCode());
        erpLogisticsSyncDto.setTid(erp.getErpId());
        LogisticsOrderEntity entity = convertToLogisticsOrderEntity(erpLogisticsSyncDto,
            BeanUtils.copy(erp, QihoErpOrderEntity.class));
        createLogisticsOrder(entity);
    }

    @Override
    @Transactional(value = DsConstants.DATABASE_QIHO, rollbackFor = Exception.class)
    public void createLogisticsOrder(LogisticsOrderEntity logisticsOrderEntity) {
        qihoLogisticsOrderDAO.insert(logisticsOrderEntity);
        qihoOrderDAO.updateOrderStatus(logisticsOrderEntity.getOrderId(), OrderStatusEnum.SEND_OUT);
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());
        orderSnapshot.setPostId(logisticsOrderEntity.getPostId());
        orderSnapshot.setOrderStatus(OrderStatusEnum.SEND_OUT.getCode());
        orderSnapshot.setLogisticsCode(logisticsOrderEntity.getLogisticsCode());
        orderSnapshot.setLogisticsName(logisticsOrderEntity.getLogisticsName());
        orderSnapshot.setLogisticsStatus(logisticsOrderEntity.getLogisticsStatus());
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);

        // 针对商品进行发送短信
        if (smsItemFilterBo.validItem(logisticsOrderEntity.getItemId())) {
            // 发布订单物流信息更新事件
            LogisticsInfoUpdateEvent event = BeanUtils.copy(logisticsOrderEntity, LogisticsInfoUpdateEvent.class);
            QihoOrderEntity entity = qihoOrderDAO.findByOrderId(logisticsOrderEntity.getOrderId());
            event.setConsumerName(entity.getConsumerName());
            bizEventBus.post(event);

        }
        // 订阅物流信息
        List<LogisticsOrderDo> logisticsOrderDos = Lists.newArrayList();
        logisticsOrderDos.add(BeanUtils.copy(logisticsOrderEntity,LogisticsOrderDo.class));
        KuaiDi100Event expressEvent = new KuaiDi100Event();
        expressEvent.setLogisticsOrderDos(logisticsOrderDos);
        bizEventBus.post(expressEvent);
    }

    @Transactional(DsConstants.DATABASE_QIHO)
    @Override
    public void createLogisticsOrderBatchDelivery(LogisticsOrderEntity entity) {

        if (0 == qihoLogisticsOrderDAO.insert(entity)) {
            throw new QihoException("保存物流订单失败");
        }

        if (0 == qihoOrderDAO.updateOrderStatus(entity.getOrderId(), OrderStatusEnum.SEND_OUT)) {
            throw new QihoException("更新订单状态失败");
        }

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(entity.getOrderId());
        orderSnapshot.setPostId(entity.getPostId());
        orderSnapshot.setOrderStatus(OrderStatusEnum.SEND_OUT.getCode());
        orderSnapshot.setLogisticsCode(entity.getLogisticsCode());
        orderSnapshot.setLogisticsName(entity.getLogisticsName());
        orderSnapshot.setLogisticsStatus(entity.getLogisticsStatus());
        if (0 == qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot)) {
            throw new QihoException("更新订单快照失败");
        }

    }

    @Override
    public PagenationDto<LogisticsOrderDto> queryLogisticsOrderPage(QueryLogisticsOrderDto queryItem, Integer pageNum,
        Integer pageSize) {
        PagenationDto<LogisticsOrderDto> pagenation = new PagenationDto<>();
        LogisticsOrderEntity param = BeanUtils.copy(queryItem, LogisticsOrderEntity.class);
        int total = qihoLogisticsOrderDAO
            .countLogisticsOrderByParam(param, queryItem.getStartTime(), queryItem.getEndTime());
        List<LogisticsOrderEntity> list = null;
        if (total > 0) {
            list = qihoLogisticsOrderDAO
                .queryLogisticsOrderByParam(param, pageNum, pageSize, queryItem.getStartTime(), queryItem.getEndTime());
        }
        pagenation.setList(convertToLogisticsOrderDto(list));
        pagenation.setTotal(total);
        return pagenation;
    }

    /**
     * LogisticsOrderEntity转LogisticsOrderDto
     *
     * @param list
     *
     * @return
     *
     * @author chencheng
     */
    private List<LogisticsOrderDto> convertToLogisticsOrderDto(List<LogisticsOrderEntity> list) {
        List<LogisticsOrderDto> logisticsOrderDtoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return logisticsOrderDtoList;
        }
        for (LogisticsOrderEntity entity : list) {
            LogisticsOrderDto dto = BeanUtils.copy(entity, LogisticsOrderDto.class);
            logisticsOrderDtoList.add(dto);
        }
        return logisticsOrderDtoList;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer updateBylogisticsId(String logisticsId, String orderStatus) {
        LogisticsOrderDto dto = findByLogisticsId(logisticsId);
        // 查询不到，或者已签收
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            AppLogUtil.info(LOG, "物流子订单已签收，或查询不到。无法修改订单状态,logisticsId={}", logisticsId);
            return 0;
        }
        String logisticsStatus = StringUtils.equals(orderStatus, LogisticsOrderStatusEnum.SUCCESS.getCode()) ?
            YTOLogisticsStatusEnum.SIGNED.getDesc() :
            YTOLogisticsStatusEnum.FAILED.getDesc();
        int success = qihoLogisticsOrderDAO.updateStatusByLogisticsId(dto.getOrderId(), logisticsId, orderStatus, logisticsStatus);
        updateOrderInfo(dto, logisticsStatus, orderStatus);
        return success;
    }

    /**
     * @param logisticsOrderDto 物流子订单
     * @param logisticsStatus   物流状态，由快递平台推送过来
     * @param orderStatus       根据物流状态要更新的物流子订单状态
     *
     * @author zhanglihui
     */
    private void updateOrderInfo(LogisticsOrderDto logisticsOrderDto, String logisticsStatus, String orderStatus) {
        LogisticsOrderStatusEnum logisticsOrderStatusEnum = LogisticsOrderStatusEnum.getByCode(orderStatus);
        switch (logisticsOrderStatusEnum) {
        case SUCCESS:
            updateToSuccess(logisticsOrderDto.getOrderId(), logisticsStatus);
            // 异步推送签收数据回传推啊 9-签收/10-拒签
            EXECUTOR_TUIA.execute(() -> deliverData2Tuia(logisticsOrderDto.getOrderId(),9));
            break;
        case RETURN:
            updateToFailed(logisticsOrderDto.getOrderId(), logisticsStatus, orderStatus);
            // 异步推送签收数据回传推啊 9-签收/10-拒签
            //EXECUTOR_TUIA.execute(() -> deliverData2Tuia(logisticsOrderDto.getOrderId(),10));
            break;
        case SEND_BACK:
        case REJECTED:
            updateToFailed(logisticsOrderDto.getOrderId(), logisticsStatus, orderStatus);
            break;
        default:
            updateDefault(logisticsOrderDto.getOrderId(),logisticsStatus, logisticsOrderStatusEnum);
            break;
        }
        if (StringUtils.equals(orderStatus, LogisticsOrderStatusEnum.RETURN.getCode())) {
            // 在线支付支付退件成功自动退款
            returnAmount(logisticsOrderDto);
            // 退货成功不需要发送短信
            return;
        }

//        if (LogisticsNameEnum.JD.getCode().equals(logisticsOrderDto.getLogisticsCode()))
//            return; //京邦达暂时不需要发送短信
        // 发布物流更新事件
        LogisticsUpdateEvent event = new LogisticsUpdateEvent();
        event.setLogisticsStatus(YTOLogisticsStatusEnum.getByName(logisticsStatus));
        event.setOrderId(logisticsOrderDto.getOrderId());
        event.setLogisticsOrderStatusEnum(logisticsOrderStatusEnum);
        if (LogisticsNameEnum.YTO.getCode().equals(logisticsOrderDto.getLogisticsCode()) && StringUtils
            .equals(YTOLogisticsStatusEnum.SENT_SCAN.getDesc(), logisticsStatus)) {
            event.setDeliveryMsg(logisticsOrderDto.getDeliveryMsg());
        }
        if (!LogisticsNameEnum.YTO.getCode().equals(logisticsOrderDto.getLogisticsCode())) {
            Map<String, String> deliveryMsg = new HashMap<>();
            //物流公司名
            deliveryMsg.put("logisticsCom", logisticsOrderDto.getLogisticsName());
            event.setDeliveryMsg(deliveryMsg);
        }
        event.setLogisticsCode(logisticsOrderDto.getLogisticsCode());
        bizEventBus.post(event);
    }

    /**
     * 更新默认的数据
     * 在途、接单、派件时更新
     * 只需要更新订单的物流状态和备注信息
     * @param orderId
     * @param logisticsStatus
     * @param logisticsOrderStatusEnum
     */
    private void updateDefault(String orderId, String logisticsStatus, LogisticsOrderStatusEnum logisticsOrderStatusEnum){
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);

        QihoOrderEntity qihoOrderEntity = new QihoOrderEntity();
        qihoOrderEntity.setOrderId(orderId);
        qihoOrderEntity.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));
        qihoOrderDAO.updateByOrderId(qihoOrderEntity);
    }

    /**
     * 发送签收回传推啊事件
     *
     * @param orderId
     * @param subType
     */
    @Deprecated
    private void deliverData2TuiaEvent(String orderId, Integer subType) {
        DeliverData2TuiaEvent event = new DeliverData2TuiaEvent();
        event.setOrderId(orderId);
        event.setSubType(subType);
        bizEventBus.post(event);
    }


    /**
     * 发送签收回传推啊
     * @param orderId
     * @param subType
     */
    private void deliverData2Tuia(String orderId,Integer subType){


        DBTimeProfile.enter("签收数据回传推啊事件处理, orderId= " + orderId + ", subType=" + subType);

        // 临时排查问题使用info日志
        LOG.info("签收数据回传推啊 , order {}",orderId);
        DBTimeProfile.enter("orderId查询渠道信息");
        QihoChannelInfoEntity channelInfoEntity = qihoChannelInfoDAO.findByOrderId(orderId);
        if (null == channelInfoEntity) {
            DBTimeProfile.release();
            DBTimeProfile.release();
            LOG.info("根据orderId获取channel对象失败, orderId={}, subType=", orderId, subType);
            return;
        }
        DBTimeProfile.release();

        String url = channelInfoEntity.getUrl();
        String callbackData = null;
        if (StringUtils.isNotBlank(url)) {
            callbackData = deliverData2TuiaBo.deliverData(url, subType);
        }

        if (StringUtils.isBlank(callbackData)) {
            LOG.warn("回传推啊失败, orderId={}, url={} subType={}", orderId, url, subType);
        }

        DBTimeProfile.release();
    }


    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#findByOrderId(java.lang.String)
     */
    @Override
    public List<LogisticsOrderDto> findByOrderId(String orderId) {
        return BeanUtils.copyList(qihoLogisticsOrderDAO.findByOrderId(orderId), LogisticsOrderDto.class);
    }

    private List<ErpLogisticsSyncAckDto> convertToListReurn(Map<String, ErpLogisticsOrderParams> errorCodeMap) {
        List<ErpLogisticsSyncAckDto> listReurn = new ArrayList<>();
        for (Entry<String, ErpLogisticsOrderParams> errorMap : errorCodeMap.entrySet()) {
            ErpLogisticsOrderParams erpLogisticsOrderParams = errorMap.getValue();
            listReurn.add(new ErpLogisticsSyncAckDto(erpLogisticsOrderParams.getRecId(),
                erpLogisticsOrderParams.getErpErrorCodeReturnEnum()));
        }
        return listReurn;
    }

    private LogisticsOrderEntity convertToLogisticsOrderEntity(ErpLogisticsSyncDto erpLogisticsSyncDto,
        QihoErpOrderEntity erpOrder) {
        LogisticsOrderEntity entity = new LogisticsOrderEntity();
        entity.setErpId(erpLogisticsSyncDto.getTid());
        entity.setPostId(erpLogisticsSyncDto.getLogisticsNo());
        entity.setLogisticsCode(erpLogisticsSyncDto.getLogisticsCode());
        entity.setLogisticsName(erpLogisticsSyncDto.getLogisticsName());
        entity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getDesc());
        // 物流子订单初始化状态为接单
        entity.setOrderStatus(LogisticsOrderStatusEnum.ACCEPT.getCode());
        entity.setConsumerName(erpOrder.getConsumerName());
        entity.setItemId(erpOrder.getItemId());
        entity.setItemName(erpOrder.getItemName());
        entity.setItemNo(erpOrder.getItemNo());
        entity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, erpOrder.getItemId()));
        entity.setMobile(erpOrder.getMobile());
        entity.setOrderId(erpOrder.getOrderId());
        entity.setSkuNo(erpOrder.getSkuNo());
        entity.setProvince(erpOrder.getProvince());
        return entity;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer update(LogisticsOrderDto logisticsOrderDto) {
        LogisticsOrderDto dto = findByPostIdCode(logisticsOrderDto.getPostId(), logisticsOrderDto.getLogisticsCode());
        int success = 0;
        // 查询不到，或者已签收(客户签收或仓库签收)
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            AppLogUtil.info(LOG, "物流子订单已签收，或查询不到。无法修改订单状态,postId={}", logisticsOrderDto.getPostId());
            return success;
        }

        QihoOrderEntity orderEntity = qihoOrderDAO.findByOrderId(dto.getOrderId());
        if (StringUtils.equalsIgnoreCase(orderEntity.getOrderStatus(), OrderStatusEnum.CLOSED.getCode())) {
            AppLogUtil.info(LOG, "主订单已关闭，无法修改订单状态，orderId={}", dto.getOrderId());
            return success;
        }

        if (StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.FAILED.getCode())
                && !StringUtils.equals(logisticsOrderDto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())
                && !StringUtils.equals(logisticsOrderDto.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())
                && !StringUtils.equals(logisticsOrderDto.getOrderStatus(), LogisticsOrderStatusEnum.SEND_BACK.getCode())
                && !StringUtils.equals(logisticsOrderDto.getOrderStatus(), LogisticsOrderStatusEnum.REJECTED.getCode())) {
            // 物流子订单状态为签收失败，只能更新为签收成功或退件成功或拒签或退回中
            return success;
        }
        LogisticsOrderEntity param = BeanUtils.copy(logisticsOrderDto, LogisticsOrderEntity.class);
        success = qihoLogisticsOrderDAO.updateStatusByPostIdAndCode(param);
        // 派件状态
        if (StringUtils.equals(YTOLogisticsStatusEnum.SENT_SCAN.getDesc(), logisticsOrderDto.getLogisticsStatus())) {
            dto.setDeliveryMsg(logisticsOrderDto.getDeliveryMsg());
        }

        updateOrderInfo(dto, logisticsOrderDto.getLogisticsStatus(), logisticsOrderDto.getOrderStatus());

        return success;
    }

    /**
     * 退件成功，ERP子订单为发货类型&发货状态时，交易金额自动退还
     */
    private void returnAmount(LogisticsOrderDto logisticsOrderDto) {
        ErpOrderDto erpOrderDto = erpOrderService.findErpOrderByErpId(logisticsOrderDto.getErpId());
        if (ErpTypeEnum.SEND.getCode().equals(erpOrderDto.getErpType())) {
            String orderId = logisticsOrderDto.getOrderId();
            List<FundOrderDto> fundOrderDtoList = fundOrderService
                .findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
            // 一个主订单只会有一笔付款资金流水，对应一次发货类型ERP订单b
            FundOrderDto fundOrder = fundOrderDtoList.get(0);
            // 付款成功的在线支付订单需要发起退款
            if (!StringUtils.equals(PayTypeEnum.COD.getCode(), fundOrder.getPayType()) && StringUtils
                .equals(FundStatusEnum.SUCCESS.getCode(), fundOrder.getFundStatus())) {
                fundOrderService.refund(orderId, fundOrder.getPayType(), fundOrder.getAmt());
            }
        }
    }

    /**
     * @param orderId
     * @param logisticsStatus
     *
     * @author zhanglihui
     */
    private void updateToSuccess(String orderId, String logisticsStatus) {
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.SUCCESS.getCode());
        updateOrder.setRemark(getRemarkByLogisticsStatus(LogisticsOrderStatusEnum.SUCCESS));
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setOrderStatus(OrderStatusEnum.SUCCESS.getCode());
        orderSnapshot.setFundStatus(FundStatusEnum.SUCCESS.getCode());
        orderSnapshot.setRemark(getRemarkByLogisticsStatus(LogisticsOrderStatusEnum.SUCCESS));
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
        List<QihoFundOrderEntity> list = qihoFundOrderDAO
            .findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fundOrder = list.get(0);
            if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
                qihoFundOrderDAO.updateFundOrderStatus(fundOrder.getFundId(), FundStatusEnum.SUCCESS.getCode());
            }
        }
    }

    private void updateToSend(String orderId, String logisticsStatus, String logisticsOrderStatus) {
        LogisticsOrderStatusEnum logisticsOrderStatusEnum = LogisticsOrderStatusEnum.getByCode(logisticsOrderStatus);
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.SEND_OUT.getCode());
        updateOrder.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setOrderStatus(OrderStatusEnum.SEND_OUT.getCode());
        orderSnapshot.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    /**
     * @author zhanglihui
     */
    private void updateToFailed(String orderId, String logisticsStatus, String logisticsOrderStatus) {
        LogisticsOrderStatusEnum logisticsOrderStatusEnum = LogisticsOrderStatusEnum.getByCode(logisticsOrderStatus);
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        updateOrder.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        orderSnapshot.setRemark(getRemarkByLogisticsStatus(logisticsOrderStatusEnum));

        List<QihoFundOrderEntity> list = qihoFundOrderDAO
            .findByOrderIdAndBizType(orderId, FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fundOrder = list.get(0);
            if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
                qihoFundOrderDAO.updateFundOrderStatus(fundOrder.getFundId(), FundStatusEnum.FAILED.getCode());
                orderSnapshot.setFundStatus(FundStatusEnum.FAILED.getCode());
            }
        }
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);

    }


    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#createLogisticsOrderBatch(java.util.List)
     */
    @Override
    public List<ErpLogisticsSyncAckDto> createLogisticsOrderBatch(List<ErpLogisticsSyncDto> list) {
        List<ErpLogisticsSyncAckDto> returnList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return returnList;
        }
        // 订单状态map
        Map<String, ErpLogisticsOrderParams> errorCodeMap = new HashMap<>(list.size());
        Map<String, ErpLogisticsSyncDto> syncMap = Maps.newHashMap();
        List<String> erpIds = Lists.newArrayList();
        for (ErpLogisticsSyncDto erpLogisticsSyncDto : list) {
            // 先置为订单不存在状态，如果数据已存在或更新成功会更改
            errorCodeMap.put(erpLogisticsSyncDto.getTid(), new ErpLogisticsOrderParams(erpLogisticsSyncDto.getRecId(),
                ErpErrorCodeReturnEnum.ORDER_CODE_NON_ERROR));
            erpIds.add(erpLogisticsSyncDto.getTid());
            syncMap.put(erpLogisticsSyncDto.getTid(), erpLogisticsSyncDto);
        }

        List<QihoErpOrderEntity> erpOrderList = qihoErpOrderDAO.findByErpIds(erpIds);
        if (CollectionUtils.isEmpty(erpOrderList)) {
            return convertToListReurn(errorCodeMap);
        }
        // 查询已同步的物流子订单
        List<LogisticsOrderEntity> existList = qihoLogisticsOrderDAO.findByErpIds(erpIds);
        List<String> syncErpIds = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(existList)) {
            for (LogisticsOrderEntity logisticsOrder : existList) {
                syncErpIds.add(logisticsOrder.getErpId());
            }
        }
        for (QihoErpOrderEntity erpOrder : erpOrderList) {
            errorCodeMap.get(erpOrder.getErpId()).setErpErrorCodeReturnEnum(ErpErrorCodeReturnEnum.SUCCESS);
            if (syncErpIds.contains(erpOrder.getErpId())) {
                // 已经同步的不再创建物流子订单
                continue;
            }
            LogisticsOrderEntity logisticsOrderEntity = convertToLogisticsOrderEntity(syncMap.get(erpOrder.getErpId()),
                erpOrder);
            // 创建物流子订单、创建付款子订单、更新订单快照、发送发货短信
            createLogisticsOrder(logisticsOrderEntity);
        }
        return convertToListReurn(errorCodeMap);
    }

    @Override
    public LogisticsOrderDto findByPostIdCode(String postId, String logisticsCode) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByPostIdCode(postId, logisticsCode), LogisticsOrderDto.class);
    }

    @Override
    public LogisticsOrderDto findByLogisticsId(String logisticsId) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByLogisticsId(logisticsId), LogisticsOrderDto.class);
    }

    @Override
    public LogisticsOrderDto findByErpId(String erpId) {
        List<String> erpids = Lists.newArrayList();
        erpids.add(erpId);
        List<LogisticsOrderEntity> list = qihoLogisticsOrderDAO.findByErpIds(erpids);
        return CollectionUtils.isNotEmpty(list) ? BeanUtils.copy(list.get(0), LogisticsOrderDto.class) : null;
    }

    /**
     * @param logisticsOrderStatus
     *
     * @author zhanglihui
     */
    @Override
    public void syncLogisticsStatus(String logisticsOrderStatus) {
        LogisticsOrderEntity param = new LogisticsOrderEntity();
        param.setOrderStatus(logisticsOrderStatus);
        int pageSize = 100;
        int pageNum = 1;
        while (true) {
            List<LogisticsOrderEntity> list = qihoLogisticsOrderDAO
                .queryLogisticsOrderByParam(param, pageNum, pageSize, null, null);
            if (CollectionUtils.isEmpty(list)) {
                break;
            }
            doSync(list);
            pageNum++;
        }
    }

    /**
     * @param list
     *
     * @author zhanglihui
     */
    @Override
    public void doSync(List<LogisticsOrderEntity> list) {
        for (LogisticsOrderEntity entity : list) {
            if (StringUtils.equals(entity.getLogisticsCode(), LogisticsNameEnum.YTO.getCode())) {
                QueryYTOLogisticProcessParams queryXml = new QueryYTOLogisticProcessParams(
                    new Result(new WaybillCode(entity.getPostId())));
                // 请求查询走件流程
                String logisticsString = YTOClientUtil.execute(YTOClientUtil.getQueryLogisticsUrl(), queryXml);
                String[] refuseRemark = { "签收人: 退件", "签收人: 退回" };
                if (StringUtils.indexOfAny(logisticsString, refuseRemark) > 0 && StringUtils
                    .contains(logisticsString, "【浙江省杭州市富阳市公司】 派件人")) {
                    entity.setLogisticsStatus(YTOLogisticsStatusEnum.FAILED.getDesc());
                    entity.setOrderStatus(LogisticsOrderStatusEnum.RETURN.getCode());
                } else if (StringUtils.contains(logisticsString, "已签收 感谢使用圆通速递，期待再次为您服务")) {
                    entity.setLogisticsStatus(YTOLogisticsStatusEnum.SIGNED.getDesc());
                    entity.setOrderStatus(LogisticsOrderStatusEnum.SUCCESS.getCode());
                } else {
                    continue;
                }
                LogisticsOrderDto dto = BeanUtils.copy(entity, LogisticsOrderDto.class);
                update(dto);
            }
        }
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#queryListToFix(int, int, long)
     */
    @Override
    public List<LogisticsOrderEntity> queryListToFix(int pageNum, int pageSize, long minId) {
        return qihoLogisticsOrderDAO.queryListToFix(pageNum, pageSize, minId);
    }

    @Override
    public int queryDistributeOverTimeNum() {
        return qihoLogisticsOrderDAO.queryDistributeOverTimeNum();
    }

    @Override
    public List<LogisticsOrderDto> queryDistributeOverTimePage(Integer pageNum, Integer pageSize) {
        List<LogisticsOrderEntity> list = qihoLogisticsOrderDAO.queryDistributeOverTimePage(pageNum, pageSize);
        return BeanUtils.copyList(list, LogisticsOrderDto.class);
    }

    @Override
    public Integer updateLogisticsOrderRemark(LogisticsOrderDto param) {
        return qihoLogisticsOrderDAO.updateLogisticsOrderRemark(param);
    }

    @Override
    public PagenationDto<LogisticsOrderDto> queryServiceLogisticsOrder(QueryLogisticsOrderDto queryItem,
        Integer pageNum, Integer pageSize) {
        PagenationDto<LogisticsOrderDto> pagenation = new PagenationDto<>();
        LogisticsOrderEntity param = BeanUtils.copy(queryItem, LogisticsOrderEntity.class);
        int total = qihoLogisticsOrderDAO
            .countServiceLogisticsOrder(param, queryItem.getOrderStartTime(), queryItem.getOrderEndTime(),
                queryItem.getOrderProcessStatus());
        List<LogisticsOrderEntity> list = null;
        if (total > 0) {
            list = qihoLogisticsOrderDAO
                .queryServiceLogisticsOrder(param, pageNum, pageSize, queryItem.getOrderStartTime(),
                    queryItem.getOrderEndTime(), queryItem.getOrderProcessStatus());
        }
        pagenation.setList(convertToLogisticsOrderDto(list));
        pagenation.setTotal(total);
        return pagenation;
    }

    @Override
    public Long queryMonitorLogisticsOrderCount(String orderStatus, String logisticsCode, Date startTime, Date endTime,
        List<String> provinceList) {
        return qihoLogisticsOrderDAO
            .queryMonitorLogisticsOrderCount(orderStatus, logisticsCode, startTime, endTime, provinceList);
    }

    @Override
    public String findDeliveryByPostId(String postId) {
        List<String> list = qihoOrderSnapshotDAO.findDeliveryByPostId(postId);
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        }
        return null;
    }


    @Override
    @Transactional(value = DsConstants.DATABASE_QIHO , rollbackFor = Exception.class)
    public void updateStatus(LogisticsOrderEntity logisticsOrderEntity) {


        qihoLogisticsOrderDAO.updateStatusByPostIdAndCode(logisticsOrderEntity);
        LogisticsOrderStatusEnum logisticsOrderStatusEnum = LogisticsOrderStatusEnum.getByCode(logisticsOrderEntity.getOrderStatus());
        switch (logisticsOrderStatusEnum) {
            case SUCCESS:
                updateToSuccess(logisticsOrderEntity.getOrderId(), logisticsOrderEntity.getLogisticsStatus());
                break;
            case RETURN:
            case SEND_BACK:
            case FAILED:
            case REJECTED:
                updateToFailed(logisticsOrderEntity.getOrderId(), logisticsOrderEntity.getLogisticsStatus(), logisticsOrderEntity.getOrderStatus());
                break;
            default:
                QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
                orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());
                orderSnapshot.setLogisticsStatus(logisticsOrderEntity.getLogisticsStatus());
                qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
                updateToSend(logisticsOrderEntity.getOrderId(), logisticsOrderEntity.getLogisticsStatus(), logisticsOrderEntity.getOrderStatus());
        }
    }

}
