package com.qiho.center.biz.service.impl.advert;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.advert.BaiqiItemAdvertDto;
import com.qiho.center.api.params.advert.ItemAdvertQueryParams;
import com.qiho.center.biz.service.advert.ItemAdvertService;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.daoh.qiho.advert.BaiqiItemAdvertMapper;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entityd.qiho.advert.BaiqiItemAdvertEntity;

import cn.com.duiba.wolf.utils.BeanUtils;

/**
 *
 *               ii.                                         ;9ABH,
 *              SA391,                                    .r9GG35&G
 *              &#ii13Gh;                               i3X31i;:,rB1
 *              iMs,:,i5895,                         .5G91:,:;:s1:8A
 *               33::::,,;5G5,                     ,58Si,,:::,sHX;iH1
 *                Sr.,:;rs13BBX35hh11511h5Shhh5S3GAXS:.,,::,,1AG3i,GG
 *                .G51S511sr;;iiiishS8G89Shsrrsh59S;.,,,,,..5A85Si,h8
 *               :SB9s:,............................,,,.,,,SASh53h,1G.
 *            .r18S;..,,,,,,,,,,,,,,,,,,,,,,,,,,,,,....,,.1H315199,rX,
 *          ;S89s,..,,,,,,,,,,,,,,,,,,,,,,,....,,.......,,,;r1ShS8,;Xi
 *        i55s:.........,,,,,,,,,,,,,,,,.,,,......,.....,,....r9&5.:X1
 *       59;.....,.     .,,,,,,,,,,,...        .............,..:1;.:&s
 *      s8,..;53S5S3s.   .,,,,,,,.,..      i15S5h1:.........,,,..,,:99
 *      93.:39s:rSGB@A;  ..,,,,.....    .SG3hhh9G&BGi..,,,,,,,,,,,,.,83
 *      G5.G8  9#@@@@@X. .,,,,,,.....  iA9,.S&B###@@Mr...,,,,,,,,..,.;Xh
 *      Gs.X8 S@@@@@@@B:..,,,,,,,,,,. rA1 ,A@@@@@@@@@H:........,,,,,,.iX:
 *     ;9. ,8A#@@@@@@#5,.,,,,,,,,,... 9A. 8@@@@@@@@@@M;    ....,,,,,,,,S8
 *     X3    iS8XAHH8s.,,,,,,,,,,...,..58hH@@@@@@@@@Hs       ...,,,,,,,:Gs
 *    r8,        ,,,...,,,,,,,,,,.....  ,h8XABMMHX3r.          .,,,,,,,.rX:
 *   :9, .    .:,..,:;;;::,.,,,,,..          .,,.               ..,,,,,,.59
 *  .Si      ,:.i8HBMMMMMB&5,....                    .            .,,,,,.sMr
 *  SS       :: h@@@@@@@@@@#; .                     ...  .         ..,,,,iM5
 *  91  .    ;:.,1&@@@@@@MXs.                            .          .,,:,:&S
 *  hS ....  .:;,,,i3MMS1;..,..... .  .     ...                     ..,:,.99
 *  ,8; ..... .,:,..,8Ms:;,,,...                                     .,::.83
 *   s&: ....  .sS553B@@HX3s;,.    .,;13h.                            .:::&1
 *    SXr  .  ...;s3G99XA&X88Shss11155hi.                             ,;:h&,
 *     iH8:  . ..   ,;iiii;,::,,,,,.                                 .;irHA
 *      ,8X5;   .     .......                                       ,;iihS8Gi
 *         1831,                                                 .,;irrrrrs&@
 *           ;5A8r.                                            .:;iiiiirrss1H
 *             :X@H3s.......                                .,:;iii;iiiiirsrh
 *              r#h:;,...,,.. .,,:;;;;;:::,...              .:;;;;;;iiiirrss1
 *             ,M8 ..,....,.....,,::::::,,...         .     .,;;;iiiiiirss11h
 *             8B;.,,,,,,,.,.....          .           ..   .:;;;;iirrsss111h
 *            i@5,:::,,,,,,,,.... .                   . .:::;;;;;irrrss111111
 *            9Bi,:,,,,......                        ..r91;;;;;iirrsss1ss1111
 *
 *                               神兽保佑，永无bug
 *
 * @author chensong
 * @create 2018-08-17
 */
@Service
public class ItemAdvertServiceImpl implements ItemAdvertService {


    private static final Logger LOGGER = LoggerFactory.getLogger(ItemAdvertServiceImpl.class);

    @Autowired
    private BaiqiItemAdvertMapper baiqiItemAdvertMapper;

    @Autowired
    private QihoItemDAO qihoItemDAO;


    @Override
    public ResultDto<Long> insertItemAdvert(BaiqiItemAdvertDto itemAdvertDto) {

        // 校验商品id是否存在
        QihoItemEntity itemEntity = qihoItemDAO.findSimpleById(itemAdvertDto.getItemId());
        if (itemEntity == null) {
            return ResultDto.failResult("请输入有效的商品ID");
        }

        // 检验商品是否存在记录
        BaiqiItemAdvertEntity itemAdvertEntity = baiqiItemAdvertMapper.findByItemId(itemAdvertDto.getItemId());
        if (itemAdvertEntity != null) {
            return ResultDto.failResult("添加失败，该商品已有配置");
        }

        itemAdvertEntity = BeanUtils.copy(itemAdvertDto, BaiqiItemAdvertEntity.class);
        baiqiItemAdvertMapper.insert(itemAdvertEntity);

        return ResultDto.successResult(itemAdvertEntity.getId());
    }

    @Override
    public PagenationDto<BaiqiItemAdvertDto> queryPage(ItemAdvertQueryParams params) {

        PagenationDto<BaiqiItemAdvertDto> page = new PagenationDto<>();

        int count = baiqiItemAdvertMapper.coutQueryPage(params);
        if (count == 0) {
            return page.emptyPage();
        }
        page.setTotal(count - 1);

        List<BaiqiItemAdvertEntity> list = baiqiItemAdvertMapper.listQueryPage(params);

        if (params.getOffset() == 0 && !list.stream().anyMatch(e -> e.getItemId() == 0)) {
            // 如果当前页是第一页查询 且不包含itemId==0的默认广告位 那就加上默认广告位
            BaiqiItemAdvertEntity defaultAdvert = baiqiItemAdvertMapper.findByItemId(0L);
            list.add(defaultAdvert);

        } else if (params.getOffset() > 0 && list.stream().anyMatch(e -> e.getItemId() == 0L)) {
            // 如果当前页不是第一页 且包含了itemId==0的默认广告位  那就移除默认广告位配置
            list = list.stream().filter(e -> e.getItemId() != 0L).collect(Collectors.toList());
        }

        page.setList(convertVOList(list));

        return page;
    }

    /**
     * 类型转换
     * 如果有默认的数据即itemId==0L的数据就要放在第一条
     * @param advertEntities
     * @return
     */
    private List<BaiqiItemAdvertDto> convertVOList(List<BaiqiItemAdvertEntity> advertEntities){

        List<BaiqiItemAdvertDto> list = Lists.newArrayList();
        if (CollectionUtils.isEmpty(advertEntities)) {
            return list;
        }

        if (advertEntities.stream().anyMatch(e -> e.getItemId() == 0L)) {
            // 有默认数据要放在第一条
            BaiqiItemAdvertEntity firstEntity = advertEntities.stream()
                    .filter(e -> e.getItemId() == 0L).findFirst().orElse(null);
            list.add(BeanUtils.copy(firstEntity, BaiqiItemAdvertDto.class));

            for (BaiqiItemAdvertEntity entity: advertEntities) {
                if (entity.getItemId() == 0L) {
                    continue;
                }
                list.add(BeanUtils.copy(entity, BaiqiItemAdvertDto.class));
            }

        } else {
            for (BaiqiItemAdvertEntity entity: advertEntities) {
                list.add(BeanUtils.copy(entity, BaiqiItemAdvertDto.class));
            }
        }

        return list;
    }



    @Override
    public ResultDto<Boolean> deleteById(Long id) {

        if (id == null) {
            return ResultDto.failResult("id 为null");
        }
        BaiqiItemAdvertEntity entity = baiqiItemAdvertMapper.findById(id);
        if (entity == null) {
            return ResultDto.failResult("数据不存在");
        }
        if (entity.getItemId() == 0) {
            return ResultDto.failResult("默认数据不能删除");
        }

        int row = baiqiItemAdvertMapper.deleteById(id);

        return ResultDto.successResult(row == 1);
    }

    @Override
    public ResultDto<Boolean> updateItemAdvert(BaiqiItemAdvertDto itemAdvertDto) {

        if (itemAdvertDto == null || itemAdvertDto.getId() == null) {
            return ResultDto.failResult("参数不合法");
        }
        BaiqiItemAdvertEntity entity = new BaiqiItemAdvertEntity();
        entity.setId(itemAdvertDto.getId());
        entity.setAdvertId(itemAdvertDto.getAdvertId());
        entity.setOpeartor(itemAdvertDto.getOpeartor());

        try{
            int row = baiqiItemAdvertMapper.updateAdvertById(entity);
            return ResultDto.successResult(row == 1);

        } catch (Exception e){
            LOGGER.error("更新商品广告位出错，data={}", itemAdvertDto);
            return ResultDto.failResult(e.getMessage());
        }
    }

    @Override
    public ResultDto<Boolean> alterAdvertStatus(BaiqiItemAdvertDto itemAdvertDto) {

        if (itemAdvertDto == null || itemAdvertDto.getId() == null
                || itemAdvertDto.getAdvertStatus() == null) {
            return ResultDto.failResult("参数不合法");
        }

        BaiqiItemAdvertEntity itemAdvertEntity = baiqiItemAdvertMapper.findById(itemAdvertDto.getId());
        if (itemAdvertEntity == null) {
            return ResultDto.failResult("id=" +itemAdvertDto.getId()+ "商品广告位不存在");
        }
        if (itemAdvertEntity.getAdvertId() == null && itemAdvertDto.getAdvertStatus() == true) {
            // 设置开启状态必须有advertId
            return ResultDto.failResult("请先设置广告位ID");
        }

        BaiqiItemAdvertEntity entity = new BaiqiItemAdvertEntity();
        entity.setId(itemAdvertDto.getId());
        entity.setAdvertStatus(itemAdvertDto.getAdvertStatus());
        entity.setOpeartor(itemAdvertDto.getOpeartor());

        try{
            int row = baiqiItemAdvertMapper.updateStatusById(entity);
            return ResultDto.successResult(row == 1);

        } catch (Exception e){
            LOGGER.error("更新商品广告位出错，data={}", itemAdvertDto);
            return ResultDto.failResult(e.getMessage());
        }
    }

    @Override
    public BaiqiItemAdvertDto findById(Long id) {

        if (id == null || id < 0L) {
            return null;
        }

        return BeanUtils.copy(baiqiItemAdvertMapper.findById(id), BaiqiItemAdvertDto.class);
    }

    /**
     * 根据商品id获取广告位id配置 如果不需要展示广告位返回null
     * @param itemId 商品id
     * @return null or advertId
     */
    @Override
    public BaiqiItemAdvertDto findByItemConfig(Long itemId) {
        BaiqiItemAdvertEntity itemAdvertEntity = baiqiItemAdvertMapper.findByItemId(itemId);

        if (itemAdvertEntity != null) {
            //该商品配置了广告位 判断广告位是否开启
            return fetchAdvertIfNotNull(itemAdvertEntity);

        } else {
            // 商品未配置广告位 查询itemId为0的默认id
            BaiqiItemAdvertEntity defaultAdvertEntity = baiqiItemAdvertMapper.findByItemId(0L);

            if (defaultAdvertEntity == null) {
                // itemId为0默认的商品广告位不存在 不展示广告位
                return null;

            } else {
                // itemId为0默认的商品广告位存在 要判断状态
                return fetchAdvertIfNotNull(defaultAdvertEntity);
            }
        }
    }

    /**
     * 当广告位配置不为空的时候 获取广告位的配置
     * @param advertEntity
     * @return
     */
    private BaiqiItemAdvertDto fetchAdvertIfNotNull(BaiqiItemAdvertEntity advertEntity){

        if (advertEntity.getAdvertStatus()) {
            // 广告位开启 返回广告位id
            BaiqiItemAdvertDto advertDto = new BaiqiItemAdvertDto();
            advertDto.setAdvertId(advertEntity.getAdvertId());
            return advertDto;

        } else {
            // 广告位未开启 返回null
            return null;
        }
    }


}
