package com.qiho.center.biz.service.impl.blacklist;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.bklist.BaiqiBlackListDto;
import com.qiho.center.api.enums.DealTypeEnum;
import com.qiho.center.api.enums.blacklist.BaiqiBlackListTypeEnum;
import com.qiho.center.api.enums.blacklist.BlackListUseTypeEnum;
import com.qiho.center.api.enums.blacklist.SourceTypeEnum;
import com.qiho.center.api.params.blacklist.BaiqiBlackListQueryParam;
import com.qiho.center.api.params.blacklist.BlackListBatchDealParam;
import com.qiho.center.biz.runnable.BaiqiBlackListRunnable;
import com.qiho.center.biz.service.blacklist.BaiqiBlackListService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.daoh.qiho.blacklist.BaiqiBlackListMapper;
import com.qiho.center.common.entityd.qiho.blacklist.BaiqiBlackListEntity;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

/**
 * @author chensong
 * @create 2018-04-25 12:53
 **/
@Service
public class BaiqiBlackListServiceImpl implements BaiqiBlackListService , ApplicationContextAware {

    @Autowired
    private BaiqiBlackListMapper baiqiBlackListMapper;

    @Autowired
    private MerchantService merchantService;

    @Resource
    private ExecutorService executorService;

    private ApplicationContext context;

    @Override
    public PagenationDto<BaiqiBlackListDto> queryPage(BaiqiBlackListQueryParam param) {
        PagenationDto<BaiqiBlackListDto> page = new PagenationDto<>();
        if (param.getMerchantId().longValue() < 0L){
            param.setMerchantId(null);
        }

        Integer count = baiqiBlackListMapper.countQueryPage(param);
        if (count.intValue() == 0){
            return page.emptyPage();
        }
        List<BaiqiBlackListEntity> blackListEntities = baiqiBlackListMapper.queryPage(param);

        page.setTotal(count);
        page.setList(convertToDtoList(blackListEntities));

        return page;
    }

    @Override
    public ResultDto<Boolean> deleteBlackList(Long id, String name) {
        if (id == null || id.longValue() <= 0L){
            return ResultDto.failResult("id不合法");
        }
        baiqiBlackListMapper.deleteBlackList(id,name);
        return ResultDto.successResult(Boolean.TRUE);
    }

    @Override
    public int queryPageCount(BaiqiBlackListQueryParam param) {
        if (param.getMerchantId().longValue() == -1L) {
            // -1 意味着查所有的记录
            param.setMerchantId(null);
        }

        return baiqiBlackListMapper.countQueryPage(param);
    }

    @Override
    public int batchAddBlackList(String cacheKey, List<BlackListBatchDealParam> list, Long merchantId, String operator) {
        BaiqiBlackListRunnable runnable = context.getBean(BaiqiBlackListRunnable.class);
        runnable.setProgressKey(cacheKey);
        runnable.setParams(list);
        runnable.setMerchantId(merchantId);
        runnable.setOperator(operator);
        int useAbleCount = runnable.enableFilter();
        executorService.submit(runnable);
        return useAbleCount;
    }

    @Override
    public boolean simpleAddBlackList(BaiqiBlackListDto blackListDto) {
        if (blackListDto == null) {
            return false;
        }

        // 做一次数据校验 不用重复添加
        List<String> value = Lists.newArrayList(blackListDto.getBkValue());
        List<String> blacklist = baiqiBlackListMapper.findByValueAndMerchantId(value, blackListDto.getMerchantId());
        if (CollectionUtils.isNotEmpty(blacklist)) {
            // 数据存在返回true
            return true;
        }

        BaiqiBlackListEntity entity = convertToEntity(blackListDto);
        int row = baiqiBlackListMapper.insertSimple(entity);

        return row == 1;
    }

    /**
     * 将黑名单dto转成entity
     * @param blackListDto
     * @return
     */
    private BaiqiBlackListEntity convertToEntity(BaiqiBlackListDto blackListDto){
        BaiqiBlackListEntity entity = BeanUtils.copy(blackListDto, BaiqiBlackListEntity.class);
        entity.setBkType(blackListDto.getBkType().getNum());
        entity.setDealType(blackListDto.getDealType().getNum());
        entity.setSourceType(blackListDto.getSourceType().getNum());
        entity.setUseType(blackListDto.getUseType().getNum());
        return entity;
    }

    /**
     * 将黑名单entity转成dto
     * @param blackListEntities
     * @return
     */
    private List<BaiqiBlackListDto> convertToDtoList(List<BaiqiBlackListEntity> blackListEntities ){
        Map<Long,String> merchantMap = merchantService.fetchNamesWithCache();
        return Lists.transform(blackListEntities, e -> {
            BaiqiBlackListDto dto = BeanUtils.copy(e, BaiqiBlackListDto.class);
            dto.setBkType(BaiqiBlackListTypeEnum.getByNum(e.getBkType()));
            dto.setDealType(DealTypeEnum.getByNum(e.getDealType()));
            dto.setSourceType(SourceTypeEnum.getByNum(e.getSourceType()));
            if (e.getUseType().intValue() == BlackListUseTypeEnum.PART.getNum()){
                dto.setMerchantName(merchantMap.get(e.getMerchantId()));
            } else {
                dto.setMerchantName(BlackListUseTypeEnum.ALL.getDesc());
            }
            return dto;
        });
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }
}
