package com.qiho.center.biz.service.impl.finance;

import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.qiho.center.api.dto.BaseResultDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.account.BaiqiFinanceRechargeDetailDto;
import com.qiho.center.api.dto.account.BaiqiFinanceRechargeDto;
import com.qiho.center.api.enums.finance.*;
import com.qiho.center.api.enums.merchant.BaiqiMerchantLinkTypeEnum;
import com.qiho.center.api.params.finance.BaiqiFinanceRechargeQryParam;
import com.qiho.center.biz.service.finance.AllocateAmountService;
import com.qiho.center.biz.service.finance.BaiqiFinanceRechargeService;
import com.qiho.center.biz.service.impl.finance.bean.PartnerAllocateAmountBean;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoMerchantDAO;
import com.qiho.center.common.daoh.qiho.agent.BaiqiAgentMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceDetailMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceRechargeMapper;
import com.qiho.center.common.daoh.qiho.merchant.BaiqiMerchantLinkMapper;
import com.qiho.center.common.entity.merchant.QihoMerchantEntity;
import com.qiho.center.common.entityd.qiho.agent.BaiqiAgentEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceDetailEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceRechargeEntity;
import com.qiho.center.common.entityd.qiho.merchant.BaiqiMerchantLinkEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 账户充值服务
 * Author: zhanghuifeng
 * Date:   10:02 2017/12/21
 */
@Service("baiqiFinanceRechargeService")
public class BaiqiFinanceRechargeServiceImpl implements BaiqiFinanceRechargeService {

	@Resource
	private BaiqiFinanceRechargeMapper baiqiFinanceRechargeMapper;
	@Resource
	private BaiqiAgentMapper baiqiAgentMapper;
	@Resource
	private QihoMerchantDAO qihoMerchantDAO;
	@Resource(name = "redisTemplate")
	private RedisAtomicClient redisAtomicClient;
	@Resource
	private BaiqiFinanceMapper baiqiFinanceMapper;
	@Resource
	private BaiqiFinanceDetailMapper baiqiFinanceDetailMapper;
	@Resource
	private AllocateAmountService allocateAmountService;
	@Resource
	private BaiqiMerchantLinkMapper baiqiMerchantLinkMapper;

	@Override
	public PagenationDto<BaiqiFinanceRechargeDetailDto> find4Page(BaiqiFinanceRechargeQryParam params) {
		PagenationDto<BaiqiFinanceRechargeDetailDto> page = new PagenationDto<>();
		if (params.getOffset() == null || params.getMax() == null) {
			page.setTotal(0);
			return page;
		}
		Integer count = baiqiFinanceRechargeMapper.count4Page(params);
		page.setTotal(count);
		if (count == 0) {
			return page;
		}

		List<BaiqiFinanceRechargeDetailDto> list = BeanUtils.copyList(baiqiFinanceRechargeMapper.query4Page(params), BaiqiFinanceRechargeDetailDto.class);
		//组装数据
		mergeDto(list);
		page.setList(list);
		return page;
	}

	/**
	 * 拼装数据
	 * @param list 集合
	 */
	private void mergeDto(List<BaiqiFinanceRechargeDetailDto> list){
		//获取商家id和代理商id集合
		List<Long> agentIds = new ArrayList<>();
		List<Long> merchantIds = new ArrayList<>();
		for (BaiqiFinanceRechargeDetailDto dto : list){
			if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.AGENT.getType()){
				agentIds.add(dto.getCommercialTenantId());
			}else if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.MERCHANT.getType()){
				merchantIds.add(dto.getCommercialTenantId());
			}
		}

		//查询商家和代理商信息
		Map<Long, String> agentMap = getAgentMap(agentIds);
		Map<Long, String> merchantMap = getMerchantMap(merchantIds);

		//获取账户名称
		if (MapUtils.isNotEmpty(agentMap) || MapUtils.isNotEmpty(merchantMap)){
			for (BaiqiFinanceRechargeDetailDto dto : list){
				if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.AGENT.getType()){
					dto.setCommercialTenantName(agentMap.get(dto.getCommercialTenantId()));
				}else if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.MERCHANT.getType()){
					dto.setCommercialTenantName(merchantMap.get(dto.getCommercialTenantId()));
				}
			}
		}
	}

	/**
	 * 获取商家map
	 * @param merchantIds
	 * @return
	 */
	private Map<Long, String> getMerchantMap(List<Long> merchantIds){
		Map<Long, String> merchantMap = new HashMap<>();
		if (CollectionUtils.isNotEmpty(merchantIds)){
			List<QihoMerchantEntity> merchantList = qihoMerchantDAO.listByIds(merchantIds);
			if (CollectionUtils.isNotEmpty(merchantList)){
				merchantMap = merchantList.stream().collect(Collectors.toMap(QihoMerchantEntity::getId,QihoMerchantEntity::getMerchantName));
			}
		}
		return merchantMap;
	}

	/**
	 * 代理商map
	 * @param agentIds 代理商id集合
	 * @return
	 */
	private Map<Long, String> getAgentMap(List<Long> agentIds){
		Map<Long, String> agentMap = new HashMap<>();
		if (CollectionUtils.isNotEmpty(agentIds)){
			List<BaiqiAgentEntity> agentList = baiqiAgentMapper.findByIds(agentIds);
			if (CollectionUtils.isNotEmpty(agentList)){
				agentMap = agentList.stream().collect(Collectors.toMap(BaiqiAgentEntity::getId,BaiqiAgentEntity::getAgentName));
			}
		}
		return agentMap;
	}


	@Override
	public Boolean insert(BaiqiFinanceRechargeDto dto) {
		dto.setRechargeNo(getRechargeNo());
		return baiqiFinanceRechargeMapper.insert(BeanUtils.copy(dto, BaiqiFinanceRechargeEntity.class)) > 0;
	}

	/**
	 * 生成充值单号
	 * @return 单号
	 */
	private Long getRechargeNo(){
		DateTimeFormatter dtf = DateTimeFormatter.ofPattern("yyyyMMdd");
		String date = dtf.format(LocalDateTime.now());
		Long value = redisAtomicClient.incrBy(MessageFormat.format(DsConstants.RECHARGE_NO_REDIS_KEY,date),1,1, TimeUnit.DAYS);
		return Long.valueOf(date+String.format("%03d", value));
	}

	@Override
	public BaiqiFinanceRechargeDetailDto findByNo(Long rechargeNo) {
		BaiqiFinanceRechargeDetailDto dto = BeanUtils.copy(baiqiFinanceRechargeMapper.findByRechargeNo(rechargeNo),BaiqiFinanceRechargeDetailDto.class);
		if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.MERCHANT.getType()){
			QihoMerchantEntity merchantEntity = qihoMerchantDAO.getById(dto.getCommercialTenantId());
			if (merchantEntity != null){
				dto.setCommercialTenantName(merchantEntity.getMerchantName());
				dto.setCommercialTenantOwner(merchantEntity.getContactName());
			}
		}else if (dto.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.AGENT.getType()){
			BaiqiAgentEntity agentEntity = baiqiAgentMapper.findById(dto.getCommercialTenantId());
			if (agentEntity != null){
				dto.setCommercialTenantName(agentEntity.getAgentName());
				dto.setCommercialTenantOwner(agentEntity.getContactName());
			}
		}

		return dto;
	}

	@Override
	@Transactional(DsConstants.DATABASE_QIHO)
	public BaseResultDto updateState(Integer state, Long rechargeNo, String operator) {
		//充值成功操作
		if (state == BaiqiFinanceRechargeStateEnum.SUCCEED.getState()){
			return toSuccessState(state, rechargeNo, operator);
		}else {
			return notToSuccessState(state, rechargeNo, operator);
		}
	}

	/**
	 * 更新非成功状态
	 * @param state
	 * @param rechargeNo
	 * @param operator
	 * @return
	 */
	private BaseResultDto notToSuccessState(Integer state, Long rechargeNo, String operator){
		int result = baiqiFinanceRechargeMapper.updateState(state,rechargeNo,operator);
		BaseResultDto dto = new BaseResultDto();
		if(result > 0){
			dto.setResult(true);
		}else {
			dto.setResult(false);
			dto.setErrorMsg("操作失败");
		}
		return dto;
	}

	/**
	 * 更新成功状态
	 * @return
	 */
	private BaseResultDto toSuccessState(Integer state, Long rechargeNo, String operator){
		BaiqiFinanceRechargeEntity rechargeEntity = baiqiFinanceRechargeMapper.findByRechargeNo(rechargeNo);
		if (rechargeEntity.getCommercialTenantType() == BaiqiCommercialTenantTypeEnum.MERCHANT.getType()){
			return merchantRecharge(rechargeEntity,state,operator);
		}else {
			return merchantAndAgentSuccess(rechargeEntity,state,operator);
		}
	}

	private BaseResultDto merchantAndAgentSuccess(BaiqiFinanceRechargeEntity rechargeEntity, Integer state, String operator){
		//充值总金额
		Long incomeTotalAmout = getTotalAmount(rechargeEntity.getCashSum(),rechargeEntity.getRebateSum());
		//财务主表增加金额
		BaiqiFinanceEntity financeEntity = baiqiFinanceMapper.findByRelation(rechargeEntity.getCommercialTenantType(),rechargeEntity.getCommercialTenantId());
		financeEntity.setCashAmount(rechargeEntity.getCashSum() == null?financeEntity.getCashAmount():Long.valueOf(rechargeEntity.getCashSum()) + financeEntity.getCashAmount());
		financeEntity.setRebateAmount(rechargeEntity.getRebateSum() == null?financeEntity.getRebateAmount():Long.valueOf(rechargeEntity.getRebateSum())+financeEntity.getRebateAmount());
		financeEntity.setTotalAmount(financeEntity.getTotalAmount() + incomeTotalAmout);
		baiqiFinanceMapper.updateAmount(financeEntity);

		//增加一条流水信息
		BaiqiFinanceDetailEntity financeDetailEntity = new BaiqiFinanceDetailEntity();
		financeDetailEntity.setFinanceId(financeEntity.getId());
		financeDetailEntity.setRelationId(financeEntity.getRelationId());
		financeDetailEntity.setRelationType(financeEntity.getRelationType());
		financeDetailEntity.setIncomeCash(rechargeEntity.getCashSum() == null?null:Long.valueOf(rechargeEntity.getCashSum()));
		financeDetailEntity.setIncomeRebate(rechargeEntity.getRebateSum() == null?null:Long.valueOf(rechargeEntity.getRebateSum()));
		financeDetailEntity.setState(FinanceDetailStatusEnum.SUCCESS.getNum());
		financeDetailEntity.setOperator(0L);
		financeDetailEntity.setFinanceTradeNo(String.valueOf(rechargeEntity.getRechargeNo()));
		financeDetailEntity.setOperatorType(FinanceOptTypeEnum.OFFLINE_RECHARGE.getNum());
		financeDetailEntity.setIncomeTotal(incomeTotalAmout);
		baiqiFinanceDetailMapper.insert(financeDetailEntity);

		baiqiFinanceRechargeMapper.updateState(state,rechargeEntity.getRechargeNo(),operator);
		BaseResultDto resultDto = new BaseResultDto();
		resultDto.setResult(true);
		return resultDto;
	}

	private BaseResultDto merchantRecharge(BaiqiFinanceRechargeEntity rechargeEntity, Integer state, String operator){
		BaseResultDto resultDto = new BaseResultDto();
		//充值总金额
		Long incomeTotalAmout = getTotalAmount(rechargeEntity.getCashSum(),rechargeEntity.getRebateSum());
		BaiqiMerchantLinkEntity merchantLinkEntity = baiqiMerchantLinkMapper.findByMerchantIdAndRelationType(rechargeEntity.getCommercialTenantId(), BaiqiMerchantLinkTypeEnum.TUI_A.getType());
		if (merchantLinkEntity == null){
			resultDto.setResult(false);
			resultDto.setErrorMsg("商家所关联广告主不存在");
			return resultDto;
		}
		BaiqiFinanceEntity financeEntity = baiqiFinanceMapper.findByRelation(rechargeEntity.getCommercialTenantType(),rechargeEntity.getCommercialTenantId());

		PartnerAllocateAmountBean allocateAmountBean = new PartnerAllocateAmountBean();
		allocateAmountBean.setAccountId(merchantLinkEntity.getRelationId());
		allocateAmountBean.setBaiqiTradeNo(rechargeEntity.getRechargeNo().toString());
		allocateAmountBean.setTradeType(TuiaTradeTypeEnum.RECHARGE);
		allocateAmountBean.setAmount(incomeTotalAmout);
		PartnerResultStateEnum partnerResultStateEnum = allocateAmountService.callPartnerAllocateAmount(allocateAmountBean);
		switch (partnerResultStateEnum){
			case FAILED:
				resultDto.setResult(false);
				resultDto.setErrorMsg("同步推啊失败，请检查推啊账户后重试");
				//增加一条流水信息
				insertFinanceDetail(financeEntity,rechargeEntity,incomeTotalAmout,FinanceDetailStatusEnum.FAILED);
				break;
			case SUCCEED:
				resultDto = merchantAndAgentSuccess(rechargeEntity,state,operator);
				break;
			case EXCEPTION:
				//增加一条流水信息
				insertFinanceDetail(financeEntity,rechargeEntity,incomeTotalAmout,FinanceDetailStatusEnum.ERROR);

				//更新主订单状态
				baiqiFinanceRechargeMapper.updateState(BaiqiFinanceRechargeStateEnum.EXCEPTION.getState(),rechargeEntity.getRechargeNo(),operator);
				resultDto.setResult(false);
				resultDto.setErrorMsg("同步推啊异常");
				break;
			default:
				resultDto.setResult(false);
				break;
		}
		return resultDto;
	}

	private void insertFinanceDetail(BaiqiFinanceEntity financeEntity, BaiqiFinanceRechargeEntity rechargeEntity, Long incomeTotalAmout, FinanceDetailStatusEnum statusEnum){
		//增加一条流水信息
		BaiqiFinanceDetailEntity financeDetailEntity = new BaiqiFinanceDetailEntity();
		financeDetailEntity.setFinanceId(financeEntity.getId());
		financeDetailEntity.setRelationId(financeEntity.getRelationId());
		financeDetailEntity.setRelationType(financeEntity.getRelationType());
		financeDetailEntity.setIncomeCash(rechargeEntity.getCashSum() == null?null:Long.valueOf(rechargeEntity.getCashSum()));
		financeDetailEntity.setIncomeRebate(rechargeEntity.getRebateSum() == null?null:Long.valueOf(rechargeEntity.getRebateSum()));
		financeDetailEntity.setState(statusEnum.getNum());
		financeDetailEntity.setOperator(0L);
		financeDetailEntity.setFinanceTradeNo(String.valueOf(rechargeEntity.getRechargeNo()));
		financeDetailEntity.setOperatorType(FinanceOptTypeEnum.OFFLINE_RECHARGE.getNum());
		financeDetailEntity.setIncomeTotal(incomeTotalAmout);
		baiqiFinanceDetailMapper.insert(financeDetailEntity);
	}

	private Long getTotalAmount(Integer cashSum, Integer rebateSum){
		if (cashSum != null && rebateSum != null){
			return (long) (cashSum + rebateSum);
		}else {
			if (cashSum != null){
				return Long.valueOf(cashSum);
			}
			if (rebateSum != null){
				return Long.valueOf(rebateSum);
			}
		}
		return null;
	}

}
