package com.qiho.center.biz.service.impl.finance;

import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.finance.FinanceDetailDto;
import com.qiho.center.api.dto.finance.FinanceDto;
import com.qiho.center.api.dto.finance.HistoricalBalanceDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.enums.GetParamTimeEnum;
import com.qiho.center.api.enums.finance.FinanceDetailStatusEnum;
import com.qiho.center.api.enums.finance.FinanceOptTypeEnum;
import com.qiho.center.api.enums.finance.FinanceTypeEnum;
import com.qiho.center.api.params.finance.BaiqiMerchantFreetextParam;
import com.qiho.center.api.params.finance.FinanceBasePageParam;
import com.qiho.center.biz.service.agent.BaiqiAgentService;
import com.qiho.center.biz.service.finance.FinanceDetailService;
import com.qiho.center.biz.service.finance.FinanceService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.daoh.qiho.finance.BaiqiFinanceDetailMapper;
import com.qiho.center.common.daoh.qiho.finance.BaiqiHistoricalBalanceMapper;
import com.qiho.center.common.entityd.qiho.agent.BaiqiAgentEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiFinanceDetailEntity;
import com.qiho.center.common.entityd.qiho.finance.BaiqiHistoricalBalanceEntity;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Created by xianhongwei on 2017/12/20.
 */
@Service
public class FinanceDetailServiceImpl implements FinanceDetailService {

    @Autowired
    private BaiqiFinanceDetailMapper financeDetailMapper;

    @Autowired
    private BaiqiHistoricalBalanceMapper historicalBalanceMapper;

    @Resource
    private MerchantService merchantService;

    @Resource
    private BaiqiAgentService baiqiAgentService;

    @Resource
    private FinanceService financeService;

    @Override
    public PagenationDto<FinanceDetailDto> findByParams(BaiqiMerchantFreetextParam params) {
        PagenationDto<FinanceDetailDto> pagenationDto = new PagenationDto<>();
        splitMerchantOrAgent(pagenationDto, params);
        return pagenationDto;
    }

    @Override
    public Long insert(FinanceDetailDto financeDetailDto) {
        BaiqiFinanceDetailEntity entity = BeanUtils.copy(financeDetailDto, BaiqiFinanceDetailEntity.class);
        entity.setOperatorType(financeDetailDto.getOperatorType().getNum());
        entity.setState(financeDetailDto.getState().getNum());
        return financeDetailMapper.insert(entity);
    }

    @Override
    public PagenationDto<FinanceDetailDto> findFinanceDetailByMerchantId(BaiqiMerchantFreetextParam pageParam) {
        List<BaiqiFinanceDetailEntity> list = financeDetailMapper.selectFinanceDetailByMerchantId(pageParam);
        List<FinanceDetailDto> resultList = Lists.newArrayList();
        for (BaiqiFinanceDetailEntity entity : list) {
            FinanceDetailDto financeDetailDto = BeanUtils.copy(entity, FinanceDetailDto.class);
            financeDetailDto.setOperatorType(FinanceOptTypeEnum.getByNum(entity.getOperatorType()));
            financeDetailDto.setState(FinanceDetailStatusEnum.getByNum(entity.getState()));
            resultList.add(financeDetailDto);
        }
        PagenationDto<FinanceDetailDto> pagenationDto = new PagenationDto<>();
        pagenationDto.setList(resultList);
        int total = financeDetailMapper.countFinanceDetailByMerchantId(pageParam);
        pagenationDto.setTotal(total);
        return pagenationDto;
    }

    @Override
    public PagenationDto<FinanceDetailDto> findFinanceDetailByAgentId(BaiqiMerchantFreetextParam pageParam) {
        List<BaiqiFinanceDetailEntity> list = financeDetailMapper.selectFinanceDetailByAgentId(pageParam);
        List<FinanceDetailDto> resultList = Lists.newArrayList();
        for (BaiqiFinanceDetailEntity entity : list) {
            FinanceDetailDto financeDetailDto = BeanUtils.copy(entity, FinanceDetailDto.class);
            financeDetailDto.setOperatorType(FinanceOptTypeEnum.getByNum(entity.getOperatorType()));
            financeDetailDto.setState(FinanceDetailStatusEnum.getByNum(entity.getState()));
            resultList.add(financeDetailDto);
        }
        PagenationDto<FinanceDetailDto> pagenationDto = new PagenationDto<>();
        pagenationDto.setList(resultList);
        int total = financeDetailMapper.countFinanceDetailByAgentId(pageParam);
        pagenationDto.setTotal(total);
        return pagenationDto;
    }

    @Override
    public BaiqiFinanceDetailEntity findFinanceDetailByTradeNo(int relationType, String tradeNo) {
        List<BaiqiFinanceDetailEntity> detailEntityList = financeDetailMapper.selectByTradeNo(tradeNo);
        if (null == detailEntityList || detailEntityList.isEmpty()) {
            return null;
        }
        for (BaiqiFinanceDetailEntity entity : detailEntityList) {

            if (relationType == entity.getRelationType()) {
                return entity;
            }
        }
        return null;
    }

    private void splitMerchantOrAgent(PagenationDto<FinanceDetailDto> pagenationDto, BaiqiMerchantFreetextParam param) {
        if (param.getRelationType() == null || param.getRelationType() == 0) {
            param.setRelationType(null);
        }
        List<BaiqiFinanceDetailEntity> entities = financeDetailMapper.findByParams(param);
        if (CollectionUtils.isEmpty(entities)) {
            pagenationDto.setList(Collections.emptyList());
            pagenationDto.setTotal(0);
            return;
        }
        List<Long> merchantIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();
        setRelationIds(merchantIds, agentIds, entities);
        Map<Long, MerchantDto> merchantDtoMap = new HashMap<>();
        Map<Long, BaiqiAgentEntity> agentMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(merchantIds)) {
            merchantDtoMap = merchantService.findByIdList(merchantIds);
        }
        if (CollectionUtils.isNotEmpty(agentIds)) {
            agentMap = baiqiAgentService.findByIdsMap(agentIds);
        }
        List<FinanceDetailDto> detailDtos = new ArrayList<>();
        for (BaiqiFinanceDetailEntity entity : entities) {
            Long relationId = entity.getRelationId();
            int accountType = entity.getRelationType();
            FinanceDetailDto detailDto = entityToDto(entity);
            if (accountType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                //设置代理商名称
                BaiqiAgentEntity agentEntity = agentMap.get(relationId);
                detailDto.setName(agentEntity == null ? null : agentEntity.getAgentName());
            } else {
                MerchantDto merchantDto = merchantDtoMap.get(relationId);
                detailDto.setName(merchantDto == null ? null : merchantDto.getMerchantName());
            }

            detailDtos.add(detailDto);
        }
        Integer count = financeDetailMapper.count(param);
        pagenationDto.setTotal(count);
        pagenationDto.setList(detailDtos);
    }

    private FinanceDetailDto entityToDto(BaiqiFinanceDetailEntity entity) {
        FinanceDetailDto financeDetailDto = BeanUtils.copy(entity, FinanceDetailDto.class);
        financeDetailDto.setRelationType(FinanceTypeEnum.getByCode(entity.getRelationType()));
        financeDetailDto.setState(FinanceDetailStatusEnum.getByNum(entity.getState()));
        financeDetailDto.setOperatorType(FinanceOptTypeEnum.getByNum(entity.getOperatorType()));
        return financeDetailDto;
    }

    private void setRelationIds(List<Long> merchantIds, List<Long> agentIds, List<BaiqiFinanceDetailEntity> entities) {
        for (BaiqiFinanceDetailEntity entity : entities) {
            int accountType = entity.getRelationType();
            Long relationId = entity.getRelationId();
            if (accountType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                agentIds.add(relationId);
            } else {
                merchantIds.add(relationId);
            }
        }
    }

    @Override
    public PagenationDto<HistoricalBalanceDto> findHistoricalBalanceByPara(BaiqiMerchantFreetextParam params) {
        PagenationDto<HistoricalBalanceDto> pagenationDto = new PagenationDto<>();
        splitMerchantAndAgent(pagenationDto, params);
        return pagenationDto;
    }

    private void splitMerchantAndAgent(PagenationDto<HistoricalBalanceDto> pagenationDto,
        BaiqiMerchantFreetextParam param) {
        if (param.getRelationType() == null || param.getRelationType() == 0) {
            param.setRelationType(null);
        }
        param.setStartTime(DateUtils.getDayDate(param.getStartTime()));
        param.setEndTime(DateUtils.getDayDate(param.getEndTime()));
        List<BaiqiHistoricalBalanceEntity> entities = historicalBalanceMapper.findHistoricalBalanceByPara(param);
        if (CollectionUtils.isEmpty(entities)) {
            pagenationDto.setList(Collections.emptyList());
            pagenationDto.setTotal(0);
            return;
        }
        List<Long> merchantIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();

        for (BaiqiHistoricalBalanceEntity entity : entities) {
            int accountType = entity.getRelationType();
            Long relationId = entity.getRelationId();
            if (accountType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                agentIds.add(relationId);
            } else {
                merchantIds.add(relationId);
            }
        }

        Map<Long, MerchantDto> merchantDtoMap = new HashMap<>();
        Map<Long, BaiqiAgentEntity> agentMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(merchantIds)) {
            merchantDtoMap = merchantService.findByIdList(merchantIds);
        }
        if (CollectionUtils.isNotEmpty(agentIds)) {
            agentMap = baiqiAgentService.findByIdsMap(agentIds);
        }
        List<HistoricalBalanceDto> historicalBalanceDtos = entitiesToDtoList(entities, agentMap, merchantDtoMap);

        Integer count = historicalBalanceMapper.findCountByReportDate(param);
        pagenationDto.setTotal(count);
        pagenationDto.setList(historicalBalanceDtos);
    }

    private List<HistoricalBalanceDto> entitiesToDtoList(List<BaiqiHistoricalBalanceEntity> entities, Map<Long, BaiqiAgentEntity> agentMap, Map<Long, MerchantDto> merchantDtoMap) {
        List<HistoricalBalanceDto> historicalBalanceDtos = new ArrayList<>();
        for (BaiqiHistoricalBalanceEntity entity : entities) {
            Long relationId = entity.getRelationId();
            int accountType = entity.getRelationType();
            HistoricalBalanceDto historicalBalanceDto = BeanUtils.copy(entity, HistoricalBalanceDto.class);
            historicalBalanceDto.setRelationType(FinanceTypeEnum.getByCode(entity.getRelationType()));
            if (accountType == FinanceTypeEnum.AGENT_ACCOUNT.getCode()) {
                //设置代理商名称
                BaiqiAgentEntity agentEntity = agentMap.get(relationId);
                historicalBalanceDto.setName(agentEntity == null ? null : agentEntity.getAgentName());
            } else {
                MerchantDto merchantDto = merchantDtoMap.get(relationId);
                historicalBalanceDto.setName(merchantDto == null ? null : merchantDto.getMerchantName());
            }

            historicalBalanceDtos.add(historicalBalanceDto);
        }
        return historicalBalanceDtos;
    }

    @Override
    public Boolean historicalBalanceSync() {
        BaiqiMerchantFreetextParam params = new BaiqiMerchantFreetextParam();

        FinanceBasePageParam basePageParam = new FinanceBasePageParam();
        // 判断今日是否已经同步数据
        Integer historicalBalanceCount = historicalBalanceMapper.findHistoricalBalanceCount();
        Date dayDate = DateUtils.getDayDate(getParamTime(-1, GetParamTimeEnum.START_TIME));
        if (historicalBalanceCount != 0) {
            params.setMax(historicalBalanceCount);
            List<HistoricalBalanceDto> historicalBalanceDtoList = financeService.findAllHistoricalBalance(params).getList();
            for (HistoricalBalanceDto dto : historicalBalanceDtoList) {
                if (DateUtils.getDayStr(dayDate).equals(DateUtils.getDayStr(dto.getReportDate()))) {
                    return Boolean.FALSE;
                }
            }
        }

        // 查询财务基础信息表中数据
        Integer baseInfoCount = financeService.findBaseInfoCount();
        if (baseInfoCount == 0) {
            return Boolean.FALSE;
        }
        basePageParam.setMax(baseInfoCount);
        List<FinanceDto> financeDtoList = financeService.findAll(basePageParam).getList();
        params.setStartTime(getBeginTime(financeDtoList));
        params.setEndTime(getParamTime(-1, GetParamTimeEnum.END_TIME));
        financeDtoList = financeService.findByParam(params);

        // 同步数据
        List<BaiqiHistoricalBalanceEntity> list = new ArrayList<>();
        for (FinanceDto dto : financeDtoList) {
            BaiqiHistoricalBalanceEntity baiqiHistoricalBalanceEntity = new BaiqiHistoricalBalanceEntity();
            baiqiHistoricalBalanceEntity.setRelationType(dto.getRelationType().getCode());
            baiqiHistoricalBalanceEntity.setRelationId(dto.getRelationId());
            baiqiHistoricalBalanceEntity.setReportDate(dto.getGmtCreate());
            baiqiHistoricalBalanceEntity.setTotalAmount(dto.getTotalAmount());
            baiqiHistoricalBalanceEntity.setCashAmount(dto.getCashAmount());
            baiqiHistoricalBalanceEntity.setRebateAmount(dto.getRebateAmount());
            baiqiHistoricalBalanceEntity.setReportDate(dayDate);
            list.add(baiqiHistoricalBalanceEntity);
        }
        return historicalBalanceMapper.historicalBalanceSync(list) > 0;
    }

    /**
     * 获取指定日期的时间点
     *
     * @param date 提前或延后的天数
     * @return Date 时分秒改为指定日期的时间点
     */
    public static Date getParamTime(int date, GetParamTimeEnum timeType) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DAY_OF_MONTH, date);
        calendar.set(Calendar.HOUR_OF_DAY, timeType.getNum() == 0 ? 00 : 23);
        calendar.set(Calendar.MINUTE, timeType.getNum() == 0 ? 00 : 59);
        calendar.set(Calendar.SECOND, timeType.getNum() == 0 ? 00 : 59);
        return calendar.getTime();
    }

    /**
     * 在list中获取记录最早的时间
     * @param list
     * @return
     */
    private Date getBeginTime(List<FinanceDto> list) {
        Date startTime = list.get(0).getGmtCreate();
        for (FinanceDto dto : list) {
            if (startTime.after(dto.getGmtCreate())) {
                startTime = dto.getGmtCreate();
            }
        }
        return startTime;
    }

    @Override
    public Integer findIncomeInfoCount() {
        return financeDetailMapper.findIncomeInfoCount();
    }

    @Override
    public Integer findHistoricalBalanceCount() {
        return historicalBalanceMapper.findHistoricalBalanceCount();
    }
}
