package com.qiho.center.biz.service.impl.homepage;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.javatuples.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.dto.ItemDto;
import com.qiho.center.api.dto.homepage.TabItemDto;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.enums.TabItemStatusEnum;
import com.qiho.center.api.params.TabItemParams;
import com.qiho.center.api.params.TabItemQueryParam;
import com.qiho.center.biz.service.ItemSaleCountService;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.homepage.TabItemService;
import com.qiho.center.biz.service.order.OrderSnapshotService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.dao.QihoTabItemDAO;
import com.qiho.center.common.entity.homepage.QihoTabItemEntity;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;

import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

/**
 * Created by qianjue on 2017/8/29.
 */
@Service
public class TabItemServiceImpl implements TabItemService{


	private static final Logger LOG           = LoggerFactory.getLogger(TabItemServiceImpl.class);

	@Resource
	private QihoTabItemDAO   qihoTabItemDAO;

	@Resource
	private ItemService itemService;

	@Resource
	private OrderSnapshotService orderSnapshotService;

	@Resource
	private ItemSaleCountService itemSaleCountService;

	@Resource
	private TabItemService tabItemService;

	@Resource
	private QihoItemDAO qihoItemDAO;

	@Resource
	private RemoteStockService remoteStockService;

	@Resource
	private QihoItemSkuDAO qihoItemSkuDAO;

	@Override
	public Integer saveTabItemList(List<TabItemParams> list) {
		if(CollectionUtils.isEmpty(list)){
			return  0;
		}
		List<QihoTabItemEntity> entityList = Lists.newArrayList();
		list.stream().forEach(param -> {
			//根据tabId和itemId查询一下是否已经存在有效记录
			QihoTabItemEntity entity = qihoTabItemDAO.queryTabItemByTabIdAndItemId(param.getTabId(),param.getItemId());
			if(null==entity){
				entity = new QihoTabItemEntity();
				entity.setItemId(param.getItemId());
				entity.setTabId(param.getTabId());
				entity.setDeleted(0);
				entity.setTopPayload(0L);
				entity.setRecentlySaleNum(queryRecentThreeDaysSaleNum(entity.getItemId()));
				entityList.add(entity);
			}
		});
		if(CollectionUtils.isEmpty(entityList)){
			return 0;
		}
		return qihoTabItemDAO.batchInsertQihoTabItem(entityList);
	}


	private Integer queryRecentThreeDaysSaleNum(Long itemId) {
		Date startTime = DateUtils.getStartTime(DateUtils.daysAddOrSub(new Date(),-3));
		Date endTime = DateUtils.getStartTime(new Date());
		return  orderSnapshotService.queryItemSaleCount(itemId,startTime,endTime);
	}

	@Override
	public Integer deleteTabItemByIdList(List<Long> idList) {
		if(CollectionUtils.isEmpty(idList)){
			return 0;
		}
		return  qihoTabItemDAO.deleteQihoTabItem(idList);
	}


	@Transactional(DsConstants.DATABASE_QIHO)
	@Override
	public Integer updateTabItemTopPayloadByList(List<Long> idList, Long tabId, Boolean top) {
		if(CollectionUtils.isEmpty(idList)){
			return 0;
		}
		Long maxValue = null;
		if(top){
			  Long max = qihoTabItemDAO.selectMaxPayload();
			  maxValue = null == max ? idList.size() : max + idList.size();
		}
		for(Long id : idList){
			QihoTabItemEntity entity = new QihoTabItemEntity();
			entity.setId(id);
			entity.setTopPayload(top ? maxValue : 0);
			entity.setTabId(tabId);
			qihoTabItemDAO.updateTabItemTopPayload(entity);
			if(top){
				maxValue --;
			}
		}
		return idList.size();
	}


	@Override
	public List<TabItemDto> findTabItemList(TabItemQueryParam params) {
		List<QihoTabItemEntity> list = qihoTabItemDAO.findQihoTabItemList(params);
		if(CollectionUtils.isEmpty(list)){
			return Lists.newArrayList();
		}
		Map<Long,QihoTabItemEntity> map = Maps.newLinkedHashMap();
		for(QihoTabItemEntity it:list){
			map.put(it.getItemId(),it);
		}
		List<ItemDto> dtoList = itemService.findItemDtoByItemIds(map.keySet());
		Map<Long,ItemDto> itemDtoMap = Maps.newHashMap();
		dtoList.stream().forEach(dto -> itemDtoMap.put(dto.getId(),dto));
		return converToResult(list,itemDtoMap);
	}

	private List<TabItemDto> converToResult(List<QihoTabItemEntity> list, Map<Long, ItemDto> itemDtoMap) {
		List<TabItemDto> resultList = Lists.newArrayList();
		for(QihoTabItemEntity entity:list){
			TabItemDto dto = new TabItemDto();
			dto.setId(entity.getId());
			dto.setItemId(entity.getItemId());
			dto.setTabId(entity.getTabId());
			dto.setRecentlySaleNum(entity.getRecentlySaleNum());
			dto.setTopPayload(entity.getTopPayload());
			dto.setTabItemStatus(entity.getTabItemStatus());
			dto.setStyleConfig(entity.getStyleConfig());
			ItemDto itemDto = itemDtoMap.get(entity.getItemId());
			if(null==itemDto){
				continue;
			}
			dto.setItemNo(itemDto.getItemNo());
			dto.setImage(itemDto.getExtParamValue("image"));
			dto.setItemName(itemDto.getItemName());
			dto.setItemShortName(itemDto.getItemShortName());
			dto.setMinPrice(itemDto.getMinPrice());
			dto.setStock(itemDto.getStock());
			dto.setItemStatus(itemDto.getItemStatus());
			resultList.add(dto);
		}
		return  resultList;
	}

	@Override
	public List<ItemDto> queryTabItemIndex(TabItemQueryParam params) {
		List<QihoTabItemEntity> list = qihoTabItemDAO.findQihoTabItemList(params);
		if(CollectionUtils.isEmpty(list)){
			return Lists.newArrayList();
		}
		Map<Long,QihoTabItemEntity> map = Maps.newLinkedHashMap();
		List<Long> itemList = Lists.newArrayList();
		for(QihoTabItemEntity it:list){
			map.put(it.getItemId(),it);
			itemList.add(it.getItemId());
		}
		List<ItemDto> dtoList = itemService.findItemDtoByItemIds(map.keySet());
		Map<Long,Integer> saleMap = itemSaleCountService.queryItemSaleNum(itemList);
		List<TabItemDto> tabItemDtos = tabItemService.queryTabItemByTabIdAndItemIds(itemList, params.getTabId());
		List<ItemDto> resultList = Lists.newArrayList();
		Map<Long,ItemDto> resultMap = Maps.newHashMap();
		for(ItemDto item:dtoList){
			//如果商品下架或者库存为0,则剔除掉
			if (item == null || StringUtils.equals("OFF", item.getItemStatus())
					|| item.getStock() == null || item.getStock().intValue() <= 0) {
				continue;
			}

			// 设置样式信息
			setStyle(item, tabItemDtos);

			//设置一下库存
			item.setSaleNum(null== saleMap.get(item.getId()) ? 0 :saleMap.get(item.getId()));

			// 设置一下商品的虚拟库存剩余比率
			item.setVirtualRemainRate(setVirRemain(item));

			resultMap.put(item.getId(),item);
		}
		//进行重新排序
		map.entrySet().forEach(e ->{
			if(null!=resultMap.get(e.getKey())){
				resultList.add(resultMap.get(e.getKey()));
			}
		});
		return resultList;
	}

	private void setStyle(ItemDto item, List<TabItemDto> tabItemDtos) {
		for (TabItemDto dto : tabItemDtos) {
			if (item.getId().equals(dto.getItemId())) {
				if (StringUtils.isNotBlank(dto.getStyleConfig())) {
					item.setStyleConfig(dto.getStyleConfig());
				}
				item.setTabItemStatus(dto.getTabItemStatus());
			}
		}
	}

	private Double setVirRemain(ItemDto item){
		Long stock = item.getEnableStock();
		if (stock.longValue() == 0L){
			return 0.00;
		}
		// 当虚拟库存或虚拟库存小于实际时要重新计算
		Long virStock = Long.valueOf(item.getExtParamValue(ItemConstants.ItemExtConstans.VIRTUALSTOCK) == null ?
				"0" : item.getExtParamValue(ItemConstants.ItemExtConstans.VIRTUALSTOCK));
		if (virStock.longValue() == 0L || virStock < stock ) {
			 virStock = itemService.updateVirtailStock(item.getId());
		}
		BigDecimal paramNs = new BigDecimal(virStock);
		BigDecimal itemStocks = new BigDecimal(stock);
		return itemStocks.divide(paramNs,2,BigDecimal.ROUND_HALF_UP).doubleValue();
	}


	@Override
	public List<QihoTabItemEntity> queryAllItemListByPage(int offSet, int pageSize) {
		return  qihoTabItemDAO.queryAllItemListByPage(offSet,pageSize);
	}


	@Override
	public Integer updateTabItem(QihoTabItemEntity enitiy) {
		return qihoTabItemDAO.updateTabItem(enitiy);
	}

	@Override
	public void updateStatus(List<Long> itemIds, Long tabId, Integer status, Long gmtModifier) {
		qihoTabItemDAO.updateStatus(itemIds, tabId, status, gmtModifier);
	}

	@Override
	public List<TabItemDto> queryTabItemByTabIdAndItemIds(List<Long> itemIdList, Long tabId) {
		List<QihoTabItemEntity> list = qihoTabItemDAO.queryTabItemByTabIdAndItemIds(tabId, itemIdList);
		return BeanUtils.copyList(list, TabItemDto.class);
	}

	@Override
	public Integer saveTabItem(TabItemParams params) {
		//根据tabId和itemId查询一下是否已经存在有效记录
		QihoTabItemEntity entity = qihoTabItemDAO.queryTabItemByTabIdAndItemId(params.getTabId(), params.getItemId());
		if (null == entity) {
			entity = new QihoTabItemEntity();
			entity.setItemId(params.getItemId());
			entity.setTabId(params.getTabId());
			entity.setDeleted(0);
			entity.setTopPayload(0L);
			entity.setRecentlySaleNum(queryRecentThreeDaysSaleNum(entity.getItemId()));
			entity.setStyleConfig(params.getStyleConfig());
			return qihoTabItemDAO.insertQihoTabItem(entity);
		} else {
			entity.setStyleConfig(params.getStyleConfig());
			return qihoTabItemDAO.updateTabItem(entity);
		}
	}

	@Override
	public Integer initTabItemStatus() {


		/**
		 *
		 * 查询商品区内所有商品
		 * 根据查询到的商品id在主商品表内查询上架状态的商品
		 * 根据过滤得到的商品id去查询库存，得到在商品区中、状态为上架、库存不为0的商品
		 * 根据获取到的商品id更新商品在商品区表内的状态
		 */


		// 查询商品区内所有商品的id
		List<QihoTabItemEntity> qihoTabItemList = qihoTabItemDAO.findQihoTabItemList(new TabItemQueryParam());
		if (CollectionUtils.isEmpty(qihoTabItemList)) {
			LOG.info("查询tab商品区内无商品数据");
			return -1;
		}

		List<Long> itemIdList = qihoTabItemList.stream()
												.map(QihoTabItemEntity::getItemId)
												.distinct()
												.collect(Collectors.toList());

		// 查询商品
		List<QihoItemDetailEntity> itemDetailList = qihoItemDAO.findByIds(itemIdList);
		if (CollectionUtils.isEmpty(itemDetailList)) {
			LOG.info("查询所有商品区内无商品, itemIdList: {}", itemIdList);
			return -1;
		}
		// 过滤所有未删除、未下架的商品
		List<Long> statusOnItemIdList = itemDetailList.stream()
													.filter(e -> ItemStatusEnum.STATUS_ON.getCode().equals(e.getItemStatus()))
													.map(QihoItemDetailEntity::getId)
													.collect(Collectors.toList());
		if (CollectionUtils.isEmpty(statusOnItemIdList)) {
			LOG.info("过滤商品后无上架商品");
			return -1;
		}

		// 查询sku
		List<QihoItemSkuEntity> skuList = qihoItemSkuDAO.getSkuByItemIds(statusOnItemIdList);
		if (CollectionUtils.isEmpty(skuList)) {
			LOG.info("查询sku商品为空,  statusOnItemIdList: {}" , statusOnItemIdList);
			return -1;
		}

		List<Long> stockIds= skuList.stream()
											.map(QihoItemSkuEntity::getStockId)
											.collect(Collectors.toList());
		// stockId : 库存 键值对
		// 查询库存
		Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
		Pair<Map, Map> stockPair = getSkuStockInfo(stockMap, skuList);

		// itemId : 库存 键值对
		Map<Long, Long> itemStockMap = stockPair.getValue0();

		// 将库存不为0的商品的商品id提取出来
		List<Long> itemIdsWithStockNot0 = statusOnItemIdList.stream()
													.filter(e -> 0L != itemStockMap.get(e))
													.collect(Collectors.toList());
		if(CollectionUtils.isEmpty(itemIdsWithStockNot0)) {
			LOG.info("过滤库存非0的商品id为空");
			return -1;
		}

		// 初始化状态
		return qihoTabItemDAO.updateStatus(itemIdsWithStockNot0, null, TabItemStatusEnum.STATUS_ON.getValue(), null);
	}

	private Pair<Map, Map> getSkuStockInfo(Map<Long, Long> stockMap, List<QihoItemSkuEntity> skuList) {
		// itemId : 库存 键值对
		Map<Long, Long> itemStockMap = Maps.newHashMap();
		//有效库存
		Map<Long, Long> itemEnableStockMap = Maps.newHashMap();
		for (QihoItemSkuEntity skuEntity : skuList) {
			// sku库存
			Long stock = stockMap.get(skuEntity.getStockId());
			if (stock == null || stock <= 0) {
				stock = 0L;
			}
			Long itemId = skuEntity.getItemId();
			// 商品总库存
			Long itemStock = itemStockMap.get(itemId);
			itemStockMap.put(itemId, itemStock == null ? stock : itemStock + stock);
			if (skuEntity.getSkuEnable()) {
				Long itemEnableStock = itemEnableStockMap.get(itemId);
				itemEnableStockMap.put(itemId, itemEnableStock == null ? stock : itemEnableStock + stock);
			}
		}
		return Pair.with(itemStockMap, itemEnableStockMap);
	}
}
