package com.qiho.center.biz.service.impl.logistics;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.base.Objects;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.LogisticsDto;
import com.qiho.center.api.dto.logistics.LogisticsNicknameDto;
import com.qiho.center.api.params.logitics.LogisticsQueryParam;
import com.qiho.center.biz.service.logistics.LogisticsService;
import com.qiho.center.common.daoh.qiho.logistics.BaiqiLogisticsMapper;
import com.qiho.center.common.daoh.qiho.logistics.BaiqiLogisticsNicknameMapper;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsEntity;
import com.qiho.center.common.entityd.qiho.logistics.BaiqiLogisticsNicknameEntity;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author chensong
 * @create 2017-12-28 16:38
 **/
@Service
public class LogisticsServiceImpl implements LogisticsService {

    @Autowired
    private BaiqiLogisticsMapper baiqiLogisticsMapper;

    @Autowired
    private BaiqiLogisticsNicknameMapper baiqiLogisticsNicknameMapper;

    //存放所有百奇物流公司缓存
    private static final String BAIQILOGISTICSLOCALCACHEKEY = "baiqiLogisticsLocalCacheKey";

    private static final Cache<String, List<BaiqiLogisticsEntity>> baiqiLogisticsLocalCache = CacheBuilder.newBuilder()
        .maximumSize(5).expireAfterWrite(10, TimeUnit.MINUTES).build();

    @Override
    public PagenationDto<LogisticsDto> queryPage(LogisticsQueryParam param) {
        PagenationDto<LogisticsDto> page = new PagenationDto<>();
        Integer total = baiqiLogisticsMapper.countByParam(param);
        page.setTotal(total);
        if (total == 0) {
            page.setList(Lists.newArrayList());
            return page;
        }

        List<BaiqiLogisticsEntity> entityList = baiqiLogisticsMapper.findByParam(param);

        // 获取物流公司id
        List<Long> logisticsIds = entityList.stream().map(BaiqiLogisticsEntity::getId).collect(Collectors.toList());
        List<BaiqiLogisticsNicknameEntity> nicknameList = baiqiLogisticsNicknameMapper.findByList(logisticsIds);

        // 用于存放物流公司id 和 别称组合 别称以“|”分隔
        Map<Long, String> nicknameMap = new HashMap<>();
        for (BaiqiLogisticsNicknameEntity entity : nicknameList) {
            if (nicknameMap.get(entity.getLogisticsId()) == null) {
                nicknameMap.put(entity.getLogisticsId(), entity.getNickname());
            } else {
                nicknameMap.put(entity.getLogisticsId(),
                    nicknameMap.get(entity.getLogisticsId()) + "|" + entity.getNickname());
            }
        }

        // 类型转换
        List<LogisticsDto> list = Lists.transform(entityList, entity -> {
            LogisticsDto dto = BeanUtils.copy(entity, LogisticsDto.class);
            dto.setNickname(nicknameMap.get(entity.getId()));
            return dto;
        });
        page.setList(list);

        return page;
    }

    @Override
    public List<LogisticsNicknameDto> findNickname(Long logisticsId) {
        List<BaiqiLogisticsNicknameEntity> nicknameEntities = baiqiLogisticsNicknameMapper
            .findByLogisticsIdDesc(logisticsId);
        return BeanUtils.copyList(nicknameEntities, LogisticsNicknameDto.class);
    }

    @Override
    public List<BaiqiLogisticsEntity> findAll() {
        return baiqiLogisticsMapper.findAll();
    }

    @Override
    public Boolean addNickname(LogisticsNicknameDto dto) throws BizException {
        // 校验该物流公司的别称是否存在
        BaiqiLogisticsNicknameEntity entity = BeanUtils.copy(dto, BaiqiLogisticsNicknameEntity.class);
        Integer count = baiqiLogisticsNicknameMapper.countByParam(entity);
        if (count > 0) {
            throw new BizException("该外部名称已存在");
        }
        // 插入数据到别称表
        baiqiLogisticsNicknameMapper.insertNickname(entity);
        return Boolean.TRUE;
    }

    @Override
    public Boolean deleteNickname(Long nicknameId) throws BizException {
        BaiqiLogisticsNicknameEntity nicknameEntity = baiqiLogisticsNicknameMapper.findById(nicknameId);
        if (Objects.equal(null, nicknameEntity)) {
            throw new BizException("该外部名称不存在");
        }
        baiqiLogisticsNicknameMapper.deleteNickname(nicknameId);
        return Boolean.TRUE;
    }

    @Override
    public List<BaiqiLogisticsEntity> findAllByCache() {
        List<BaiqiLogisticsEntity> list = baiqiLogisticsLocalCache.getIfPresent(BAIQILOGISTICSLOCALCACHEKEY);
        if (CollectionUtils.isEmpty(list)) {
            list = this.findAll();
            baiqiLogisticsLocalCache.put(BAIQILOGISTICSLOCALCACHEKEY, list);
        }
        return list;
    }

    @Override
    public BaiqiLogisticsEntity findByCode(String logisticsCode) {
        return baiqiLogisticsMapper.findByCode(logisticsCode);
    }

    @Override
    public BaiqiLogisticsEntity findById(Long id) {
        return baiqiLogisticsMapper.findById(id);
    }

    @Override
    public Boolean insertLogistics(LogisticsDto dto) {
        BaiqiLogisticsEntity entity = BeanUtils.copy(dto, BaiqiLogisticsEntity.class);
        return baiqiLogisticsMapper.insertLogistics(entity) == 1;
    }

    @Override
    public Boolean deleteLogistics(Long id) {
        return baiqiLogisticsMapper.deleteLogistics(id) == 1;
    }

}
