package com.qiho.center.biz.service.impl.order;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.qiho.center.api.constant.AfterSaleConstant;
import com.qiho.center.api.dto.AfterSaleOrderDto;
import com.qiho.center.api.dto.AfterSaleStageDto;
import com.qiho.center.api.enums.AfterSaleOrderStageEnum;
import com.qiho.center.api.enums.AfterSaleOrderStatusEnum;
import com.qiho.center.api.enums.AfterSaleOrderTypeEnum;
import com.qiho.center.api.enums.FundBizTypeEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.AfterSaleQueryParams;
import com.qiho.center.api.util.JSONUtil;
import com.qiho.center.biz.paychannel.pay.PayChannelDecider;
import com.qiho.center.biz.paychannel.pay.PayChannelProcessor;
import com.qiho.center.biz.service.order.AfterSaleOrderService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoAfterSaleOrderDAO;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.order.QihoAfterSaleOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;

import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * @author chensong
 * @create 2017-10-11 14:18
 **/
@Service
public class AfterSaleOrderServiceImpl implements AfterSaleOrderService {

    @Autowired
    private QihoAfterSaleOrderDAO  qihoAfterSaleOrderDAO;

    @Autowired
    private QihoOrderDAO           qihoOrderDAO;

    @Autowired
    private SequenceNoBuilder sequenceNoBuilder;

    @Autowired
    private PayChannelDecider payChannelDecider;

    @Autowired
    private QihoFundOrderDAO qihoFundOrderDAO;

    @Resource
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private BizEventBus eventBus;

    @Override
    public List<AfterSaleOrderDto> findByOrderId(String orderId) {
        List<QihoAfterSaleOrderEntity> entitys = qihoAfterSaleOrderDAO.findByOrderId(orderId);
        if(CollectionUtils.isEmpty(entitys))
            return new ArrayList<>();
        return entitys.stream().map(this::convertToDto).collect(Collectors.toList());
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Boolean updateAfterSaleOrder(AfterSaleOrderDto dto) {
        QihoAfterSaleOrderEntity entity = BeanUtils.copy(dto,QihoAfterSaleOrderEntity.class);
        String afterSaleId = dto.getAfterSaleId();
        List<AfterSaleStageDto> needProcessStage = dto.getDetailStage();
        QihoAfterSaleOrderEntity afterSaleOrderEntity = qihoAfterSaleOrderDAO.queryAfterSaleById(afterSaleId);
        if (null == afterSaleOrderEntity)
            throw new QihoException("需要更新的售后子订单不存在");
        //处理扩展字段
        String orderId = afterSaleOrderEntity.getOrderId();
        String newMETAValue = updateMetaValue(afterSaleOrderEntity,needProcessStage);
        entity.setMetaValue(newMETAValue);
        qihoAfterSaleOrderDAO.updateAfterSaleOrder(entity);//更新售后子订单状态以及扩展字段

        //更新物流信息
        if(StringUtils.isBlank(dto.getAfterSaleStatus())){
            return Boolean.TRUE;
        }
        switch (AfterSaleOrderStatusEnum.getByCode(dto.getAfterSaleStatus())){
            case AFTER_SALEING:
                //如果是确认售后,创建资金子订单发起退款流程
                List<QihoFundOrderEntity> fundOrderEntities = createAfterSaleFunder(afterSaleOrderEntity);
                fundOrderEntities.forEach(e ->{
                    String payType = e.getPayType();
                    String bizType = e.getBizType();
                    PayTypeEnum payTypeEnum = PayTypeEnum.getEnumByCode(payType);
                    FundBizTypeEnum fundBizTypeEnum = FundBizTypeEnum.getByCode(bizType);
                    if (fundBizTypeEnum == FundBizTypeEnum.REFREIGHT || payTypeEnum == PayTypeEnum.COD){
                        PayChannelProcessor payChannelProcessor = payChannelDecider.decidePayChannel(PayTypeEnum.ALIPAY);
                        payChannelProcessor.transAccount(e.getAccount(),e.getFundId(),e.getAmt());
                    }else{
                        PayChannelProcessor payChannelProcessor = payChannelDecider.decidePayChannel(payTypeEnum);
                        payChannelProcessor.refund(e.getOrderId(),e.getFundId(),e.getAmt());
                    }
                });
                break;
            case SUCCESS://完成售后,订单关闭
                QihoOrderEntity qihoOrderEntity = new QihoOrderEntity();
                qihoOrderEntity.setOrderId(orderId);
                qihoOrderEntity.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
                qihoOrderEntity.setRemark(AfterSaleConstant.CLOSED_REMARK);
                qihoOrderDAO.updateByOrderId(qihoOrderEntity);
                QihoOrderSnapshotEntity qihoOrderSnapshotEntity = new QihoOrderSnapshotEntity();
                qihoOrderSnapshotEntity.setOrderId(orderId);
                qihoOrderSnapshotEntity.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
                qihoOrderSnapshotEntity.setRemark(AfterSaleConstant.CLOSED_REMARK);
                qihoOrderSnapshotDAO.updateByOrderId(qihoOrderSnapshotEntity);

                break;
            default:
                break;
        }
        return Boolean.TRUE;
    }

    @Override
    public Boolean updateAfterSaleMetaValue(AfterSaleOrderDto dto) {
        QihoAfterSaleOrderEntity entity = BeanUtils.copy(dto,QihoAfterSaleOrderEntity.class);
        JSONObject extjson = new JSONObject();
        String detailStageStr = JSONObject.toJSONString(dto.getDetailStage());
        extjson.put(AfterSaleConstant.DETAIL_STAGE,detailStageStr);
        entity.setMetaValue(extjson.toJSONString());
        return qihoAfterSaleOrderDAO.updateAfterSaleMetaValue(entity) > 0;
    }

    @Override
    public Boolean openAfterSale(String orderId) {
        List<QihoAfterSaleOrderEntity> list = qihoAfterSaleOrderDAO.findByOrderId(orderId);
        return !CollectionUtils.isEmpty(list);
    }

    @Override
    public Boolean sponsorAfterSale(String orderId) {
        // 主订单状态交易成功
        QihoOrderEntity order = qihoOrderDAO.findByOrderId(orderId);
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.SUCCESS.getCode())){
            return Boolean.FALSE;
        }
        // 交易成功不超过31天
        long days = (new Date().getTime() - order.getGmtModified().getTime())/(1000*60*60*24);
        if (days >= 31){
            return Boolean.FALSE;
        }
        // 不存在未完成（待审核|待确认|售后中）的售后子订单
        QihoAfterSaleOrderEntity afterSaleOrder  = qihoAfterSaleOrderDAO.findUnfinished(orderId);
        if (afterSaleOrder != null){
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @Override
    public List<AfterSaleOrderDto> findAll(String orderId) {
        List<QihoAfterSaleOrderEntity> entityList = qihoAfterSaleOrderDAO.findAll(orderId);
        if (CollectionUtils.isEmpty(entityList)){
            return Lists.newArrayList();
        }
        // 类型转换
        return entityList.stream().map(this::convertToDto).collect(Collectors.toList());
    }

    @Override
    public List<AfterSaleOrderDto> findAfterSaleByQuery(AfterSaleQueryParams params) {
        List<QihoAfterSaleOrderEntity> entities = qihoAfterSaleOrderDAO.findAfterSaleByQuery(params);
        if (CollectionUtils.isEmpty(entities)){
            return Lists.newArrayList();
        }
        List<AfterSaleOrderDto> afterSaleOrderList = entities.stream().map(e -> {
            AfterSaleOrderDto afterSaleOrderDto = BeanUtils.copy(e,AfterSaleOrderDto.class);
            afterSaleOrderDto.setRefreightAmt(e.getFreightAmt());
            return afterSaleOrderDto;
        }).collect(Collectors.toList());
        afterSaleOrderList.forEach(e -> e.setDetailStage(JSONUtil.parseJSONStrToList(AfterSaleConstant.DETAIL_STAGE,e.getMetaValue(),AfterSaleStageDto.class)));
        return afterSaleOrderList;
    }

    @Override
    public Integer countAfterSaleByQuery(AfterSaleQueryParams params) {
        return qihoAfterSaleOrderDAO.countAfterSaleByQuery(params);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public String save(AfterSaleOrderDto dto) {
        QihoAfterSaleOrderEntity entity = BeanUtils.copy(dto,QihoAfterSaleOrderEntity.class);
        entity.setFreightAmt(dto.getRefreightAmt());
        String orderId = dto.getOrderId();
        String bizId = StringUtils.substring(orderId, StringUtils.indexOf(orderId, SeqBizTypeEnum.ORDER.getCode())+1);
        String afterSaleId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.AFTER_SALE, Long.valueOf(bizId));
        entity.setAfterSaleId(afterSaleId);
        entity.setAfterSaleStatus(AfterSaleOrderStatusEnum.TO_COMFIRM.getCode());

        //查询付款子订单的支付方式
        List<QihoFundOrderEntity> allFundOrderEntitys = qihoFundOrderDAO.findByOrderIdAndBizType(orderId,FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isEmpty(allFundOrderEntitys))
            throw new QihoException("通过订单id查询不到资金子订单");
        String payType = allFundOrderEntitys.get(0).getPayType();
        entity.setPayType(payType);
        dto.setPayType(payType);
        dto.setAfterSaleId(afterSaleId);
        if (StringUtils.isBlank(dto.getAccount())){
            dto.setAccount(allFundOrderEntitys.get(0).getAccount());
        }

        //构建时间节点处理方式
        List<AfterSaleStageDto> afterSaleStageDtos = dto.getDetailStage();
        JSONObject metaValueJSON = new JSONObject();
        metaValueJSON.put(AfterSaleConstant.DETAIL_STAGE,JSONObject.toJSONString(afterSaleStageDtos));
        entity.setMetaValue(metaValueJSON.toJSONString());
        entity.setFreightAmt(dto.getRefreightAmt());

        qihoAfterSaleOrderDAO.save(entity);

        return afterSaleId;
    }

    @Override
    public AfterSaleOrderDto queryAfterSaleById(String afterSaleId) {
        QihoAfterSaleOrderEntity entity = qihoAfterSaleOrderDAO.queryAfterSaleById(afterSaleId);
        if (null == entity){
            return new AfterSaleOrderDto();
        }
        AfterSaleOrderDto afterSaleOrderDto = BeanUtils.copy(entity,AfterSaleOrderDto.class);
        List<AfterSaleStageDto> detailStage = JSONUtil.parseJSONStrToList(AfterSaleConstant.DETAIL_STAGE,
                afterSaleOrderDto.getMetaValue(), AfterSaleStageDto.class);
        afterSaleOrderDto.setDetailStage(detailStage);
        return afterSaleOrderDto;
    }

    @Override
    public List<AfterSaleOrderDto> queryAfterSaleByIds(List<String> afterSaleIds) {
        List<QihoAfterSaleOrderEntity> entities =  qihoAfterSaleOrderDAO.queryAfterSaleByIds(afterSaleIds);
        if (CollectionUtils.isEmpty(entities))
            return new ArrayList<>();
        List<AfterSaleOrderDto> afterSaleOrderDtos = BeanUtils.copyList(entities,AfterSaleOrderDto.class);
        for (AfterSaleOrderDto dto : afterSaleOrderDtos){
            List<AfterSaleStageDto> detailStage = JSONUtil.parseJSONStrToList(AfterSaleConstant.DETAIL_STAGE,
                    dto.getMetaValue(), AfterSaleStageDto.class);
            dto.setDetailStage(detailStage);
        }
        return  afterSaleOrderDtos;
    }

    @Override
    public Boolean updateLogistics(AfterSaleOrderDto afterSaleOrderDto) {
        AfterSaleOrderDto afterSaleOrder = queryAfterSaleById(afterSaleOrderDto.getAfterSaleId());

        afterSaleOrder.setLogisticsName(afterSaleOrderDto.getLogisticsName());
        afterSaleOrder.setPostId(afterSaleOrderDto.getPostId());
        boolean flag = true;
        AfterSaleStageDto logisticsStage = afterSaleOrderDto.getDetailStage().get(0);
        // 判断是否存在物流提交信息
        for (AfterSaleStageDto stage : afterSaleOrder.getDetailStage()){
            if (StringUtils.equals(stage.getType(),AfterSaleOrderStageEnum.SUBMIT_LOGISTICS.getCode())){
                flag = false;
                break;
            }
        }
        // 新增物流提交状态
        if (flag){
            afterSaleOrder.getDetailStage().add(logisticsStage);
        }
        QihoAfterSaleOrderEntity entity = BeanUtils.copy(afterSaleOrder,QihoAfterSaleOrderEntity.class);
        JSONObject metaJson = new JSONObject();
        metaJson.put(AfterSaleConstant.DETAIL_STAGE,JSON.toJSONString(afterSaleOrder.getDetailStage()));
        entity.setMetaValue(metaJson.toJSONString());

        return qihoAfterSaleOrderDAO.updateAfterSaleOrder(entity) > 0 ;
    }


    private AfterSaleOrderDto convertToDto(QihoAfterSaleOrderEntity entity){
        AfterSaleOrderDto dto = BeanUtils.copy(entity,AfterSaleOrderDto.class);
        dto.setRefreightAmt(entity.getFreightAmt());
        String mateValue = entity.getMetaValue();
        if (StringUtils.isNotBlank(mateValue)){
            dto.setDetailStage(JSONUtil.parseJSONStrToList(AfterSaleConstant.DETAIL_STAGE,mateValue,AfterSaleStageDto.class));
        }
        return dto;
    }

    /**
     * 创建对应的退款子订单和退运费子订单
     * @param afterSaleEntity
     */
    private List<QihoFundOrderEntity> createAfterSaleFunder(QihoAfterSaleOrderEntity afterSaleEntity){
        List<QihoFundOrderEntity> resultFundList = new ArrayList<>();
        if (null == afterSaleEntity)
            return resultFundList;
        //创建退款资金子订单
        String orderId = afterSaleEntity.getOrderId();
        Integer refundAmt = afterSaleEntity.getRefundAmt();//退款金额
        Integer freightAmt = afterSaleEntity.getFreightAmt();//退运费金额
        String payType = afterSaleEntity.getPayType();
        String afterSaleId = afterSaleEntity.getAfterSaleId();
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        String account = afterSaleEntity.getAccount();//退款账号
        entity.setOrderId(orderId);
        String bizId = StringUtils.substring(orderId, StringUtils.indexOf(orderId, SeqBizTypeEnum.ORDER.getCode())+1);
        if (refundAmt >= 10){//当退款金额小于0.1元时不生成退款子订单,不向支付宝发起请求
            QihoFundOrderEntity refundEntity = createFundOrder(orderId,bizId,payType,refundAmt,afterSaleId,FundBizTypeEnum.REFUND,SeqBizTypeEnum.FUND,account);
            resultFundList.add(refundEntity);
        }
        //如果需要退运费,构建退运费资金子订单
        if (AfterSaleOrderTypeEnum.getByCode(afterSaleEntity.getAfterSaleType()) == AfterSaleOrderTypeEnum.RETURN_AND_FREIGHT){
            QihoFundOrderEntity freightFundEntity = createFundOrder(orderId,bizId,PayTypeEnum.ALIPAY.getCode(),freightAmt,afterSaleId,FundBizTypeEnum.REFREIGHT,SeqBizTypeEnum.FUND,account);
            resultFundList.add(freightFundEntity);
        }
        return resultFundList;
    }

    /**
     * 初始化资金子订单,返回资金订单id
     * @param orderId
     * @param bizId
     * @param payType
     * @param amt
     * @param afterSaleId
     * @param bizType
     * @param fundIdType
     * @return
     */
    private QihoFundOrderEntity createFundOrder(String orderId,String bizId,String payType,Integer amt,String afterSaleId
            ,FundBizTypeEnum bizType,SeqBizTypeEnum fundIdType,String account){
        String fundId = sequenceNoBuilder.createSeqNo(fundIdType, Long.valueOf(bizId));
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        entity.setOrderId(orderId);
        entity.setFundId(fundId);
        entity.setPayType(payType);
        entity.setAmt(amt);
        entity.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        entity.setBizType(bizType.getCode());
        entity.setAftersaleId(afterSaleId);
        entity.setAccount(account);
        qihoFundOrderDAO.initFundOrder(entity);
        return entity;
    }

    /**
     * 更新扩展字段
     */
    private String updateMetaValue(QihoAfterSaleOrderEntity afterSaleOrderEntity,List<AfterSaleStageDto> needAfterSaleStageDtos){
        //更新扩展字段
        String metaValueStr = afterSaleOrderEntity.getMetaValue();
        if (StringUtils.isNotBlank(metaValueStr)){
            //更新时间节点信息
            List<AfterSaleStageDto> afterSaleStageDtos = JSONUtil.parseJSONStrToList(AfterSaleConstant.DETAIL_STAGE
                    ,metaValueStr,AfterSaleStageDto.class);
            afterSaleStageDtos.addAll(needAfterSaleStageDtos);
            JSONObject metaValueJson = new JSONObject();
            metaValueJson.put(AfterSaleConstant.DETAIL_STAGE, JSON.toJSONString(afterSaleStageDtos));
            return metaValueJson.toJSONString();
        }
        return new JSONObject().toJSONString();
    }


}
