package com.qiho.center.biz.service.impl.ordertmp;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.constant.UserOrderConstant;
import com.qiho.center.api.dto.ItemDetailDto;
import com.qiho.center.api.dto.ItemSkuDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.abtest.TestPlanStrategyDetailDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.dto.ordertmp.OrderTmpDetailDto;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.enums.ordertmp.OrderTmpRiskCheckFailEnum;
import com.qiho.center.api.enums.ordertmp.OrderTmpStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.ordertmp.OrderTmpPageParam;
import com.qiho.center.biz.event.OrderTmpCreateEvent;
import com.qiho.center.biz.event.OrderTmpToFormalEvent;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.ItemSkuService;
import com.qiho.center.biz.service.abtest.TestPlanService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.biz.service.merchant.MerchantUndeliveryService;
import com.qiho.center.biz.service.order.OrderDiscountService;
import com.qiho.center.biz.service.order.OrderTokenService;
import com.qiho.center.biz.service.ordersms.OrderSmsService;
import com.qiho.center.biz.service.ordertmp.FilterRuleHitService;
import com.qiho.center.biz.service.ordertmp.OrderTmpService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.daoh.qiho.UserOrderMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiFilterRuleHitMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderSmsMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderTmpExtMapper;
import com.qiho.center.common.daoh.qiho.ordertmp.BaiqiOrderTmpMapper;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.entityd.qiho.order.UserOrderEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiFilterRuleHitEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderSmsEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderTmpEntity;
import com.qiho.center.common.entityd.qiho.ordertmp.BaiqiOrderTmpExtEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 临时订单相关Service接口
 *
 * @author chensong
 * @create 2018-05-10 11:29
 **/
@Service
public class OrderTmpServiceImpl implements OrderTmpService {

    private final static Logger LOGGER = LoggerFactory.getLogger(OrderTmpServiceImpl.class);

    @Autowired
    private SequenceNoBuilder sequenceNoBuilder;

    @Autowired
    private BaiqiOrderTmpMapper baiqiOrderTmpMapper;

    @Autowired
    private BaiqiOrderTmpExtMapper baiqiOrderTmpExtMapper;

    @Autowired
    private BizEventBus eventBus;

    @Autowired
    private ItemService itemService;

    @Autowired
    private ItemSkuService itemSkuService;

    @Autowired
    private TestPlanService testPlanService;

    @Autowired
    private MerchantUndeliveryService merchantUndeliveryService;

    @Autowired
    private QihoItemSkuDAO qihoItemSkuDAO;

    @Autowired
    private BaiqiOrderSmsMapper baiqiOrderSmsMapper;

    @Autowired
    private OrderSmsService orderSmsService;

    @Autowired
    private BaiqiFilterRuleHitMapper baiqiFilterRuleHitMapper;

    @Autowired
    private OrderTokenService orderTokenService;

    @Autowired
    private UserOrderMapper userOrderMapper;

    @Autowired
    private OrderDiscountService orderDiscountService;


    private LoadingCache<Long, ItemDetailDto> itemDetailCache = CacheBuilder.newBuilder()
            .refreshAfterWrite(2, TimeUnit.MINUTES).build(

                new CacheLoader<Long, ItemDetailDto>() {
                    @Override
                    public ItemDetailDto load(Long itemId) throws Exception {
                        return itemService.queryItemDetail(itemId);
                    }
                });

    @Override
    @Transactional(value = DsConstants.DATABASE_QIHO, rollbackFor = Exception.class)
    public String submitTmpOrder(OrderTmpDetailDto orderDeatil) {

        // 生成订单编号
        String orderId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.ORDER, orderDeatil.getItemId());

        QihoItemSkuEntity skuEntity = qihoItemSkuDAO.findById(orderDeatil.getSkuId());
        if(skuEntity == null){
            LOGGER.warn("sku实体为空, orderId = {}, skuId = {}", orderId, orderDeatil.getSkuId());
        }else {
            // 记录商品成本价至订单
            orderDeatil.setItemCost(skuEntity.getCostPrice());
        }

        // 数据写入 order_tmp
        baiqiOrderTmpMapper.insertTmpOrder(buildOrderTmpEntity(orderDeatil,orderId));

        // 数据写入 order_tmp_ext
        baiqiOrderTmpExtMapper.insertOrderTmpExt(buildOrderTmpExtEntity(orderDeatil,orderId));

        // 小程序保存相关信息
        if (!Objects.isNull(orderDeatil.getUserId())){
            userOrderMapper.insertUserOrder(buildUserOrderEntity(orderDeatil,orderId));
        }

        // 针对优惠信息进行保存
        orderDiscountService.insertDiscountOrder(orderDeatil.getOrderDiscountDto(), orderId);


        // 事务提交后发送临时订单创建异步事件
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {

            @Override
            public void afterCompletion(int status) {
                if (TransactionSynchronization.STATUS_COMMITTED == status) {
                    postOrderTmpCreateEvent(orderId);
                }
            }
        });

        return orderId;
    }



    @Override
    public OrderTmpDetailDto findByOrderId(String orderId) {
        // 先获取order_tmp 表数据
        BaiqiOrderTmpEntity orderTmpEntity = baiqiOrderTmpMapper.findByOrderId(orderId);
        if (null == orderTmpEntity) {
            return null;
        }
        OrderTmpDetailDto orderTmpDetail = BeanUtils.copy(orderTmpEntity, OrderTmpDetailDto.class);

        // 再获取order_tmp_ext表数据
        BaiqiOrderTmpExtEntity orderTmpExtEntity = baiqiOrderTmpExtMapper.findByOrderId(orderId);
        if (orderTmpExtEntity == null) {
            return null;
        }

        org.springframework.beans.BeanUtils.copyProperties(orderTmpExtEntity, orderTmpDetail);
        orderTmpDetail.setDeliveryEnum(DeliveryEnum.getByCode(orderTmpDetail.getDelivery()));

        return orderTmpDetail;
    }

    @Override
    public OrderTmpDetailDto findDetailByOrderId(String orderId) {
        OrderTmpDetailDto orderTmpDetailDto = this.findByOrderId(orderId);
        if (null != orderTmpDetailDto) {
            // 获取防作弊信息
            List<BaiqiFilterRuleHitEntity> filterRuleHitEntities = baiqiFilterRuleHitMapper.findByOrderId(orderId);
            List<String> anticheatRules = filterRuleHitEntities.stream().map(BaiqiFilterRuleHitEntity::getRule).collect(Collectors.toList());
            orderTmpDetailDto.setAnticheatRules(anticheatRules);
        }

        return orderTmpDetailDto;
    }

    @Override
    public boolean bizValidate(OrderTmpDetailDto orderTmpDetailDto) {

        String orderId = orderTmpDetailDto.getOrderId();

        try{
            ItemDetailDto itemDetailDto = itemDetailCache.getUnchecked(orderTmpDetailDto.getItemId());

            // 校验商品状态
            validateItemStatus(itemDetailDto);

            // 校验SKU
            validateSku(orderTmpDetailDto);

            // 校验商家不配送地区
            validateMerchantUndelivery(itemDetailDto.getMerchantDto(), orderTmpDetailDto.getAreaNum());

            // 商品限购件数校验
            validateItemLimit(itemDetailDto, orderTmpDetailDto.getQuantity());

            // 货到付款校验
            validateSupportCOD(orderTmpDetailDto.getPayType(), itemDetailDto);

            return true;

        } catch (QihoException qe){
            LOGGER.warn("临时订单业务异常，orderId :{}", orderId, qe);
            // 业务校验不通过  订单变更为无效
            this.updateOrderTmpToInvalid(orderId, qe.getMessage(), OrderTmpRiskCheckFailEnum.BIZ_CHECK_FAIL);
            return false;

        } catch (Exception e) {
            // 系统异常也将订单变为无效  后续还可以人工复验
            LOGGER.warn("临时订单业务校验系统异常，orderId：{}", orderId, e);
            this.updateOrderTmpToInvalid(orderId, "临时订单业务校验系统异常", OrderTmpRiskCheckFailEnum.BIZ_CHECK_FAIL);
            return false;
        }
    }

    /**
     * 校验商品状态
     * @param itemDetailDto
     */
    private void validateItemStatus(ItemDetailDto itemDetailDto){
        if (itemDetailDto == null || itemDetailDto.getDeleted()) {
            throw new QihoException("商品已删除");
        }
        if (!StringUtils.equals(itemDetailDto.getItemStatus(), ItemStatusEnum.STATUS_ON.getCode())){
            throw new QihoException("商品未上架");
        }
    }

    /**
     * 校验SKU
     * @param orderTmpDetailDto
     */
    private void validateSku(OrderTmpDetailDto orderTmpDetailDto){
        Long skuId = orderTmpDetailDto.getSkuId();
        Long itemId = orderTmpDetailDto.getItemId();

        ItemSkuDto itemSkuDto = itemSkuService.querySku(skuId);
        if (itemSkuDto == null) {
            throw new QihoException("SKU不存在，skuId=" + skuId + ",itemId=" + itemId);
        }
        if (!itemSkuDto.getSkuEnable()) {
            throw new QihoException("SKU无效，skuId" + skuId + ",itemId=" + itemId);
        }
        /*if (itemSkuDto.getStock() == null || itemSkuDto.getStock() < orderTmpDetailDto.getQuantity()) {
            throw new QihoException("SKU库存不足，skuId=" + skuId + ",itemId=" + itemId);
        }*/

        String planCode = orderTmpDetailDto.getPlanCode();
        // 如果有相关的ABTest计划，更改SKU的价格
        if (StringUtils.isNotBlank(planCode)) {
            TestPlanStrategyDetailDto testPlanStrategyDetailDto = testPlanService.queryStrategyDetailByPlanCode(planCode);
            if (null != testPlanStrategyDetailDto) {
                List<HashMap> priceList = JSON.parseArray(testPlanStrategyDetailDto.getPriceInfo(), HashMap.class);
                Map<String, String> map = Maps.newHashMap();
                priceList.forEach(price -> map.put(price.get("skuId").toString(), price.get("price").toString()));
                String sellPrice = map.get(itemSkuDto.getId().toString());
                if (StringUtils.isNotBlank(sellPrice)) {
                    // 当sku价格作为ABTest实验时，更改校验时sku价格
                    itemSkuDto.setSellingPrice(Integer.valueOf(sellPrice));
                }
            }
        }

        if (!Objects.equals(orderTmpDetailDto.getSellingPrice(), itemSkuDto.getSellingPrice())) {
            throw new QihoException("商品价格错误");
        }
    }

    /**
     * 商家不配送地区校验
     * @param areaNum
     * @param merchantDto
     */
    private void validateMerchantUndelivery(MerchantDto merchantDto, String areaNum){
        if (merchantDto == null) {
            throw new QihoException("没有获取到商家信息");
        }

        // 非ERP商家才需要校验物流不配送地区
        if (!DeliveryEnum.ERP.getCode().equals(merchantDto.getDelivery())) {

            Long merchantId = merchantDto.getId();
            // 通过商家id查询商家的不配送地区
            List<String> undeliveryList = merchantUndeliveryService.findUndelivery(merchantId);

            // 当前配送在不配送范围
            if (CollectionUtils.isNotEmpty(undeliveryList) && undeliveryList.contains(areaNum)) {
                throw new QihoException("用户收货地区在商家不配送地区内");
            }
        }
    }

    /**
     * 校验商品限购件数
     * @param itemDetailDto
     * @param quantity
     */
    private void validateItemLimit(ItemDetailDto itemDetailDto, Integer quantity){
        Integer limitNumber;
        if (itemDetailDto.getExtParam().get(ItemConstants.ItemExtConstans.LIMIT_NUMBER) == null) {
            limitNumber = ItemConstants.DEFAULT_LIMIT_NUMBER;
        } else {
            limitNumber = Integer.valueOf(itemDetailDto.getExtParam().get(ItemConstants.ItemExtConstans.LIMIT_NUMBER));
        }
        if (quantity > limitNumber) {
            throw new QihoException("购买数量超过商品限购件数");
        }
    }

    /**
     * 货到付款校验
     * @param payType
     * @param itemDetailDto
     */
    private void validateSupportCOD(String payType, ItemDetailDto itemDetailDto){
        if (StringUtils.equals(PayTypeEnum.COD.getCode(), payType) && StringUtils
                .equals(StringUtils.trimToEmpty(itemDetailDto.getExtParamValue(ItemConstants.ItemExtConstans.SUPPORT_COD)), "0")) {
            throw new QihoException("商品不支持货到付款");
        }
    }



    @Override
    public boolean updateOrderTmpToInvalid(String orderId, String remark, OrderTmpRiskCheckFailEnum checkFailEnum) {
        BaiqiOrderTmpEntity param = new BaiqiOrderTmpEntity();
        param.setRiskCheck(checkFailEnum.getNum());
        param.setOrderStatus(OrderTmpStatusEnum.INVALID.getNum());
        param.setOrderId(orderId);
        param.setRemark(remark);
        return baiqiOrderTmpMapper.updateOrderStatus(param).intValue() == 1;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void deleteOrderTmpByOrderId(String orderId) {
        baiqiOrderTmpMapper.deleteOrderTmpByOrderId(orderId);
        baiqiOrderTmpExtMapper.deleteOrderTmpExtByOrderId(orderId);
    }

    @Autowired
    private MerchantService merchantService;

    @Autowired
    private FilterRuleHitService filterRuleHitService;

    @Override
    public PagenationDto<OrderTmpDetailDto> queryPage(OrderTmpPageParam pageParam) {
        PagenationDto<OrderTmpDetailDto> page = new PagenationDto<>();
        int count = baiqiOrderTmpMapper.countQueryPage(pageParam);
        page.setTotal(count);
        if (count == 0 ){
            return page.emptyPage();
        }

        List<BaiqiOrderTmpEntity> orderTmpEntities = baiqiOrderTmpMapper.queryPage(pageParam);

        List<String> orderIds = orderTmpEntities.stream().map(BaiqiOrderTmpEntity::getOrderId).collect(Collectors.toList());
        Map<Long,String> merMap = merchantService.fetchNamesWithCache();
        Map<String,List<String>> ruleMap = filterRuleHitService.batchFindFilterRuleHit(orderIds);

        List<BaiqiOrderTmpExtEntity> orderTmpExtEntities = baiqiOrderTmpExtMapper.findBatchByOrderIds(orderIds);
        Map<String, BaiqiOrderTmpExtEntity> extMap = Maps.newHashMap();
        orderTmpExtEntities.stream().forEach(e -> {
            extMap.put(e.getOrderId(),e);
        });

        List<OrderTmpDetailDto> orderTmpDetailDtoList = Lists.transform(orderTmpEntities, e->{
            OrderTmpDetailDto orderTmpDetailDto = BeanUtils.copy(e, OrderTmpDetailDto.class);
            org.springframework.beans.BeanUtils.copyProperties(extMap.get(e.getOrderId()), orderTmpDetailDto);
            orderTmpDetailDto.setDeliveryEnum(DeliveryEnum.getByCode(orderTmpDetailDto.getDelivery()));
            orderTmpDetailDto.setOrderStatusEnum(OrderTmpStatusEnum.getByNum(orderTmpDetailDto.getOrderStatus()));
            orderTmpDetailDto.setMerchantName(merMap.get(e.getMerchantId()));
            orderTmpDetailDto.setOrderStatusEnum(OrderTmpStatusEnum.getByNum(e.getOrderStatus()));
            orderTmpDetailDto.setDeliveryEnum(DeliveryEnum.getByCode(e.getDelivery()));
            orderTmpDetailDto.setAnticheatRules(ruleMap.get(e.getOrderId()) == null ? Lists.newArrayList() : ruleMap.get(e.getOrderId()));
            return orderTmpDetailDto;
        });

        page.setList(orderTmpDetailDtoList);
        return page;
    }

    @Override
    public ResultDto<Boolean> toValid(String orderId) {
        OrderTmpDetailDto orderTmpDetailDto = this.findByOrderId(orderId);
        if (null == orderTmpDetailDto) {
            return ResultDto.failResult("临时订单不存在");
        }

        // 只能变更无效的订单
        if (orderTmpDetailDto.getOrderStatus().intValue() != OrderTmpStatusEnum.INVALID.getNum()) {
            return ResultDto.failResult("临时订单不是无效状态");
        }

        // 如果没发短信就发送一条
        BaiqiOrderSmsEntity orderSmsEntity = baiqiOrderSmsMapper.findByOrderId(orderId);
        if (null == orderSmsEntity) {
            orderSmsService.sendMessage(orderTmpDetailDto);
        }

        // 进行业务校验
        boolean bizCheck = this.bizValidate(orderTmpDetailDto);
        if (!bizCheck) {
            return ResultDto.failResult("业务校验失败");
        }

        // 业务校验成功 将订单变更为正式订单
        OrderTmpToFormalEvent toFormalEvent = new OrderTmpToFormalEvent();
        toFormalEvent.setOrderId(orderId);
        eventBus.post(toFormalEvent);
        LOGGER.info("人工将订单置为有效，orderId： {} ", orderId);

        return ResultDto.successResult();
    }

    @Override
    public int countPageQuery(OrderTmpPageParam pageParam) {
        return baiqiOrderTmpMapper.countQueryPage(pageParam);
    }



    /**
     * 发送临时订单创建事件
     *
     * @param orderId
     */
    private void postOrderTmpCreateEvent(String orderId) {
        OrderTmpCreateEvent event = new OrderTmpCreateEvent();
        event.setOrderId(orderId);
        eventBus.post(event);
    }

    /**
     * 封装临时订单实体
     *
     * @param orderDeatil
     * @return
     */
    private BaiqiOrderTmpEntity buildOrderTmpEntity(OrderTmpDetailDto orderDeatil, String orderId) {
        BaiqiOrderTmpEntity orderTmpEntity = BeanUtils.copy(orderDeatil, BaiqiOrderTmpEntity.class);
        // 默认是待过滤状态
        orderTmpEntity.setOrderStatus(OrderTmpStatusEnum.TO_FILTER.getNum());
        orderTmpEntity.setOrderId(orderId);
        return orderTmpEntity;
    }


    /**
     * 封装临时订单拓展实体
     *
     * @param orderDeatil
     * @return
     */
    private BaiqiOrderTmpExtEntity buildOrderTmpExtEntity(OrderTmpDetailDto orderDeatil, String orderId) {
        BaiqiOrderTmpExtEntity orderTmpExt = BeanUtils.copy(orderDeatil, BaiqiOrderTmpExtEntity.class);

        orderTmpExt.setOrderId(orderId);
        if (StringUtils.isBlank(orderTmpExt.getTuiaCid())) {
            orderTmpExt.setTuiaCid("0");
        }
        return orderTmpExt;
    }

    private UserOrderEntity buildUserOrderEntity(OrderTmpDetailDto orderDeatil, String orderId){
        UserOrderEntity entity = new UserOrderEntity();
        entity.setOrderId(orderId);
        entity.setOrderSource(UserOrderConstant.WECHAT_SOURCE);
        entity.setUserId(orderDeatil.getUserId());
        return entity;
    }


}
