package com.qiho.center.biz.service.impl.tag;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.CategoryDto;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.tag.TagDto;
import com.qiho.center.api.dto.tag.TagTypeDto;
import com.qiho.center.api.params.tag.QueryTagTypeParam;
import com.qiho.center.biz.service.tag.TagTypeService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoCategoryDAO;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.daoh.qiho.tag.QihoItemTagMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoTagMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoTagTypeCategoryMapper;
import com.qiho.center.common.daoh.qiho.tag.QihoTagTypeMapper;
import com.qiho.center.common.entity.item.QihoCategoryEntity;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoTagEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoTagTypeCategoryEntity;
import com.qiho.center.common.entityd.qiho.tag.QihoTagTypeEntity;
import com.qiho.center.common.params.tag.QihoTagTypeQueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author chensong
 * @create 2018-03-23 09:41
 **/
@Service
public class TagTypeServiceImpl implements TagTypeService {

    @Autowired
    private QihoTagMapper qihoTagMapper;

    @Autowired
    private QihoTagTypeMapper qihoTagTypeMapper;

    @Autowired
    private QihoTagTypeCategoryMapper qihoTagTypeCategoryMapper;

    @Autowired
    private QihoCategoryDAO qihoCategoryDAO;

    @Autowired
    private QihoItemTagMapper qihoItemTagMapper;

    @Autowired
    private QihoItemDAO qihoItemDAO;

    @Override
    public ResultDto<Boolean> saveTagType(TagTypeDto tagTypeDto, List<Long> categoryIds) {
        if (CollectionUtils.isNotEmpty(categoryIds)){
            // 过滤掉无效类目id
            List<QihoCategoryEntity> categoryEntities = qihoCategoryDAO.findByCategoryIds(categoryIds);
            if (categoryEntities.size() < categoryIds.size()) {
                categoryIds = categoryEntities.stream().map(QihoCategoryEntity::getId).collect(Collectors.toList());
            }
        }
        if (Objects.equals(null, tagTypeDto.getId())) {
            return createTagType(tagTypeDto, categoryIds);
        } else {
            return updateTagType(tagTypeDto, categoryIds);
        }
    }

    /**
     * 插入标签类型
     * @param categoryIds
     * @return
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> createTagType(TagTypeDto tagTypeDto, List<Long> categoryIds){
        QihoTagTypeEntity qihoTagTypeEntity = qihoTagTypeMapper.findByTypeName(tagTypeDto.getTypeName());
        if (!Objects.equals(null, qihoTagTypeEntity)) {
            return ResultDto.failResult("标签类型名称已存在，请勿重复添加");
        }
        // 保存标签类型
        QihoTagTypeEntity tagTypeEntity = BeanUtils.copy(tagTypeDto, QihoTagTypeEntity.class);
        qihoTagTypeMapper.insert(tagTypeEntity);
        // 批量插入标签类型-类目
        if (CollectionUtils.isNotEmpty(categoryIds)) {
            List<QihoTagTypeCategoryEntity> typeCategoryEntities = categoryIds.stream().map(e -> {
                QihoTagTypeCategoryEntity entity = new QihoTagTypeCategoryEntity();
                entity.setTypeId(tagTypeEntity.getId());
                entity.setCategoryId(e);
                return entity;
            }).collect(Collectors.toList());
            qihoTagTypeCategoryMapper.insertBatch(typeCategoryEntities);
        }
        return ResultDto.successResult(true);
    }

    /**
     * 修改标签类型
     * @param tagTypeDto
     * @param newCategoryIds
     * @return
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public ResultDto<Boolean> updateTagType(TagTypeDto tagTypeDto, List<Long> newCategoryIds){
        QihoTagTypeEntity tagTypeEntity = qihoTagTypeMapper.findByTypeName(tagTypeDto.getTypeName());
        if (!Objects.equals(null, tagTypeEntity) && tagTypeEntity.getId().longValue() != tagTypeDto.getId().longValue()) {
            return ResultDto.failResult("标签类型名称已存在，请勿重复添加");
        }
        // 1.先获取原有的类目id
        List<QihoTagTypeCategoryEntity> typeCategoryEntities = qihoTagTypeCategoryMapper.findByTypeId(tagTypeDto.getId());
        List<Long> oldCategoryIds = typeCategoryEntities.stream().map(QihoTagTypeCategoryEntity::getCategoryId).collect(Collectors.toList());
        // 2.原先的类目ids 和 新的类目ids求交集
        List<Long> intersecIds = newCategoryIds.stream().collect(Collectors.toList());
        intersecIds.retainAll(oldCategoryIds);
        // 3.获得需要新增的类目id和需要删除的类目id
        newCategoryIds.removeAll(intersecIds);
        oldCategoryIds.removeAll(intersecIds);
        QihoTagTypeEntity qihoTagTypeEntity = BeanUtils.copy(tagTypeDto, QihoTagTypeEntity.class);
        qihoTagTypeMapper.update(qihoTagTypeEntity);
        // 4.删除不需要的数据
        if (CollectionUtils.isNotEmpty(oldCategoryIds)) {
            Map<String, Object> param = Maps.newHashMap();
            param.put("typeId", tagTypeDto.getId());
            param.put("categoryIdList", oldCategoryIds);
            qihoTagTypeCategoryMapper.deleteBatch(param);

            // 联动删除商品和标签的关系
            List<QihoItemEntity> itemEntityList = qihoItemDAO.findByCategoryIds(oldCategoryIds);
            List<QihoTagEntity> tagEntityList = qihoTagMapper.findByTypeId(tagTypeDto.getId());
            if (CollectionUtils.isNotEmpty(itemEntityList) &&  CollectionUtils.isNotEmpty(tagEntityList)) {
                List<Long> itemIdList = itemEntityList.stream().map(QihoItemEntity::getId).collect(Collectors.toList());
                List<Long> tagIdList = tagEntityList.stream().map(QihoTagEntity::getId).collect(Collectors.toList());
                Map<String, Object> paramMap = Maps.newHashMap();
                paramMap.put("itemIdList",itemIdList);
                paramMap.put("tagIdList", tagIdList);
                qihoItemTagMapper.deleteByItemTagIds(paramMap);
            }

        }
        // 5.新增类目与绑定关系
        if (CollectionUtils.isNotEmpty(newCategoryIds)) {
            List<QihoTagTypeCategoryEntity> tagTypeCategoryEntities = newCategoryIds.stream().map(e -> {
                QihoTagTypeCategoryEntity entity = new QihoTagTypeCategoryEntity();
                entity.setTypeId(tagTypeDto.getId());
                entity.setCategoryId(e);
                return entity;
            }).collect(Collectors.toList());
            qihoTagTypeCategoryMapper.insertBatch(tagTypeCategoryEntities);
        }
        return ResultDto.successResult(true);
    }

    @Override
    public List<TagTypeDto> queryTagType(QueryTagTypeParam param) {
        QihoTagTypeQueryParam queryParam = new QihoTagTypeQueryParam();
        queryParam.setTypeName(param.getTypeName());
        // 如果有类目id查询条件 先筛选出相应的typeIdList
        if (!Objects.equals(null, param.getCategoryId()) && param.getCategoryId().longValue() > 0L) {
            List<QihoTagTypeCategoryEntity> typeCategorys = qihoTagTypeCategoryMapper.findByCategoryId(param.getCategoryId());
            if (CollectionUtils.isEmpty(typeCategorys)) {
                return Lists.newArrayList();
            }
            List<Long> typeIds = typeCategorys.stream().map(QihoTagTypeCategoryEntity::getTypeId).collect(Collectors.toList());
            queryParam.setTypeIdList(typeIds);
        }
        return BeanUtils.copyList(qihoTagTypeMapper.queryTayType(queryParam), TagTypeDto.class);

    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Boolean deleteTagType(Long typeId) {
        qihoTagTypeMapper.deleteTagType(typeId);
        qihoTagMapper.deleteByTypeId(typeId);
        qihoTagTypeCategoryMapper.deleteBatchByTypeId(typeId);
        return Boolean.TRUE;
    }

    @Override
    public ResultDto<List<TagTypeDto>> findByItemId(Long itemId) {
        if (Objects.equals(null, itemId) || itemId.longValue() <= 0L) {
            return ResultDto.failResult("非法的商品id");
        }
        // 获取商品对应的一级类目
        QihoItemEntity item = qihoItemDAO.findSimpleById(itemId);
        if (Objects.equals(null, item)) {
            return ResultDto.failResult("无效的商品");
        }
        Long categoryId = item.getTopCategoryId();
        // 获取相应的标签类型id
        List<QihoTagTypeCategoryEntity> typeCategorys = qihoTagTypeCategoryMapper.findByCategoryId(categoryId);
        if (CollectionUtils.isEmpty(typeCategorys)){
            return ResultDto.successResult(Lists.newArrayList());
        }
        List<Long> typeIds = typeCategorys.stream().map(QihoTagTypeCategoryEntity::getTypeId).collect(Collectors.toList());
        List<QihoTagTypeEntity> tagTypes = qihoTagTypeMapper.findByTypeIds(typeIds);
        List<TagTypeDto> typeDtos = BeanUtils.copyList(tagTypes, TagTypeDto.class);
        // 获取tagType 下的tag
        List<QihoTagEntity> tagEntities  = qihoTagMapper.findByTypeIds(typeIds);
        Map<Long,List<TagDto>> map = Maps.newHashMap();  // key->标签类型id value->标签集合
        for (QihoTagEntity tagEntity: tagEntities) {
            TagDto tagDto = BeanUtils.copy(tagEntity, TagDto.class);
            Long typeId = tagEntity.getTypeId();
            if (map.get(typeId) == null) {
                map.put(typeId,Lists.newArrayList(tagDto));
            } else {
                List<TagDto> tagDtoList = map.get(typeId);
                tagDtoList.add(tagDto);
                map.put(typeId, tagDtoList);
            }
        }
        typeDtos.stream().peek(e -> e.setTagList(map.get(e.getId())==null?Lists.newArrayList():map.get(e.getId())))
                .collect(Collectors.toList());
        return ResultDto.successResult(typeDtos);
    }

    @Override
    public List<CategoryDto> findCategoryByTypeId(Long typeId) {
        // 获取关联的类目
        List<QihoTagTypeCategoryEntity> typeCategorys = qihoTagTypeCategoryMapper.findByTypeId(typeId);
        if (CollectionUtils.isEmpty(typeCategorys)) {
            return Lists.newArrayList();
        }
        List<Long> categoryIds = typeCategorys.stream().map(QihoTagTypeCategoryEntity::getCategoryId).collect(Collectors.toList());
        List<QihoCategoryEntity> categoryEntities = qihoCategoryDAO.findByCategoryIds(categoryIds);
        return BeanUtils.copyList(categoryEntities, CategoryDto.class);
    }


}
