package com.qiho.center.biz.service.item.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.item.ItemEvaluateDto;
import com.qiho.center.biz.service.item.ItemEvaluateService;
import com.qiho.center.common.daoh.qiho.item.ItemEvaluateMapper;
import com.qiho.center.common.entityd.qiho.item.ItemEvaluateEntity;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;

/**
 * @author jiafeng.geng
 * @date 2018-8-21.
 */
@Service
public class ItemEvaluateServiceImpl implements ItemEvaluateService {

    private final static Logger LOGGER = LoggerFactory.getLogger(ItemEvaluateServiceImpl.class);

    private static final int MAX_SHOW_LEN = 50;

    @Resource
    private ItemEvaluateMapper itemEvaluateMapper;

    @Override
    public List<ItemEvaluateDto> queryItmeEvalList(Long itemId) {
        // 获取当前商品下所有的评价
        List<ItemEvaluateEntity> itemEvaluateEntityList = itemEvaluateMapper.queryItmeEvalListBySort(itemId);
        if (CollectionUtils.isEmpty(itemEvaluateEntityList)) {
            return Lists.newArrayList();
        }

        return converToItemEvalDtoList(itemEvaluateEntityList);
    }

    private List<ItemEvaluateDto> converToItemEvalDtoList(List<ItemEvaluateEntity> itemEvaluateEntityList) {
        List<ItemEvaluateDto> itemEvaluateDtoList = Lists.newArrayList();
        itemEvaluateEntityList.forEach(itemEvaluateEntity -> {
            ItemEvaluateDto itemEvaluateDto = BeanUtils.copy(itemEvaluateEntity, ItemEvaluateDto.class);
            itemEvaluateDto.setGmtCreate(DateUtils.getSecondStr(itemEvaluateEntity.getGmtCreate()));
            itemEvaluateDtoList.add(itemEvaluateDto);
        });
        return itemEvaluateDtoList;
    }

    @Override
    public Integer queryItemEvalCount(Long itemId) {
        return itemEvaluateMapper.queryItemEvalCount(itemId);
    }

    @Override
    public Integer queryMaxSort(Long itemId) {
        return itemEvaluateMapper.queryMaxSort(itemId);
    }

    @Override
    public Integer addItemEval(ItemEvaluateDto itemEvaluateDto) {
        ItemEvaluateEntity itemEvaluateEntity = BeanUtils.copy(itemEvaluateDto, ItemEvaluateEntity.class);
        itemEvaluateEntity.setGmtCreate(DateUtils.getSecondDate(itemEvaluateDto.getGmtCreate()));
        return itemEvaluateMapper.insertItemEval(itemEvaluateEntity);
    }

    @Override
    public Integer updateItemEval(ItemEvaluateDto itemEvaluateDto) {
        ItemEvaluateEntity itemEvaluateEntity = BeanUtils.copy(itemEvaluateDto, ItemEvaluateEntity.class);
        itemEvaluateEntity.setGmtCreate(DateUtils.getSecondDate(itemEvaluateDto.getGmtCreate()));
        return itemEvaluateMapper.updateItemEval(itemEvaluateEntity);
    }

    @Override
    public Integer deletedItemEval(Long itemEvalId) {
        return itemEvaluateMapper.deletedItemEval(itemEvalId);
    }

    @Override
    public Integer sortItemEval(Long itemId, List<Long> itemEvalIds) {
        // 判断排序数量和实际商品总数是否相同
        List<ItemEvaluateEntity> itemEvaluateEntityList = itemEvaluateMapper.queryItmeEvalListBySort(itemId);
        if (itemEvalIds.size() != itemEvaluateEntityList.size()) {
            LOGGER.info("传入id数量和商品评价总数不一致");
            return -1;
        }

        // 类型转换
        ItemEvaluateOrdering itemEvaluateOrdering = new ItemEvaluateOrdering();
        itemEvaluateOrdering.setOrderItemEvalIds(itemEvalIds);
        itemEvaluateEntityList = Lists.transform(itemEvaluateEntityList, itemEvaluateOrdering);

        // 重载排序值
        int count = 0;
        for (ItemEvaluateEntity collItemEntity : itemEvaluateEntityList) {
            collItemEntity.setSort(count++);
        }

        return itemEvaluateMapper.batchUpdateItemEval(itemId, itemEvaluateEntityList);
    }

    private class ItemEvaluateOrdering implements Function<ItemEvaluateEntity, ItemEvaluateEntity> {

        private List<Long> orderItemEvalIds;

        private int maxSort;

        private void setOrderItemEvalIds(List<Long> orderItemEvalIds) {
            this.orderItemEvalIds = orderItemEvalIds;
            this.maxSort = orderItemEvalIds.size() - 1;
        }

        @Override
        public ItemEvaluateEntity apply(ItemEvaluateEntity input) {
            int index = orderItemEvalIds.indexOf(input.getId());
            input.setSort(maxSort - index);
            return input;
        }
    }

    @Override
    public Integer batchInsertItemEval(Long itemId, List<ItemEvaluateDto> itemEvaluateDtoList) {
        if (null == itemId || CollectionUtils.isEmpty(itemEvaluateDtoList)) {
            LOGGER.info("批量导入商品评价失败, itemId={}, itemEvaluateDtoList={}", itemId, itemEvaluateDtoList);
            return 0;
        }

        // 获取保存评价前的最大排序值
        Integer maxSort = itemEvaluateMapper.queryMaxSort(itemEvaluateDtoList.get(0).getItemId());
        if (null == maxSort) {
            maxSort = -1;
        }

        // 获取当前商品有效评价的总数
        Integer count = itemEvaluateMapper.queryItemEvalCount(itemId);

        // 类型转换
        List<ItemEvaluateEntity> itemEvaluateEntityList = Lists.newArrayList();
        for (ItemEvaluateDto itemEvaluateDto : itemEvaluateDtoList) {

            // 最多保存50条有效评价
            if (count >= MAX_SHOW_LEN) {
                LOGGER.info("当前商品的有效商品评价已超过50条, itemId={}", itemId);
                return 0;
            }

            ItemEvaluateEntity itemEvaluateEntity = BeanUtils.copy(itemEvaluateDto, ItemEvaluateEntity.class);
            itemEvaluateEntity.setSort(++maxSort);

            itemEvaluateEntityList.add(itemEvaluateEntity);
        }

        return itemEvaluateMapper.batchInsert(itemEvaluateEntityList);
    }
}
