package com.qiho.center.biz.service.merchant.impl;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.merchant.MerchantAdvertDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.dto.merchant.MerchantSimpleDto;
import com.qiho.center.api.enums.merchant.BaiqiMerchantLinkTypeEnum;
import com.qiho.center.biz.service.merchant.BaiqiMerchantLinkService;
import com.qiho.center.biz.service.merchant.MerchantService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.daoh.qiho.merchant.BaiqiMerchantLinkMapper;
import com.qiho.center.common.entityd.qiho.merchant.BaiqiMerchantLinkEntity;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 商家对外关联service
 *
 * @author peanut.huang
 * @date 2017/12/27.
 */
@Service("baiqiMerchantLinkService")
public class BaiqiMerchantLinkSerivceImpl implements BaiqiMerchantLinkService {

    @Resource
    private BaiqiMerchantLinkMapper merchantLinkMapper;
    @Resource
    private MerchantService merchantService;

    @Override
    @Transactional(rollbackFor = Exception.class, value = DsConstants.DATABASE_QIHO)
    public void batchInsertTuiaLink(List<BaiqiMerchantLinkEntity> insertEntityList) {

        if (CollectionUtils.isEmpty(insertEntityList)) {
            return;
        }

        List<BaiqiMerchantLinkEntity> resultList = insertEntityList;

        //find all tuia exist
        List<BaiqiMerchantLinkEntity> allTuiaList = merchantLinkMapper
            .findByType(BaiqiMerchantLinkTypeEnum.TUI_A.getType());

        if (CollectionUtils.isNotEmpty(allTuiaList)) {

            List<Long> existAdvertIdList = allTuiaList.stream().map(BaiqiMerchantLinkEntity::getRelationId)
                .collect(Collectors.toList());

            //no exist tuia advert
            resultList = insertEntityList.stream().filter(e -> !existAdvertIdList.contains(e.getRelationId()))
                .collect(Collectors.toList());
        }

        if(CollectionUtils.isNotEmpty(resultList)){
            merchantLinkMapper.batchInsert(resultList);
        }
    }

    @Override
    public BaiqiMerchantLinkEntity findByRelation(Integer relationType, Long relationId) {

        Assert.notNull(relationId, "关联id不能为空");

        BaiqiMerchantLinkTypeEnum typeEnum = BaiqiMerchantLinkTypeEnum.findByType(relationType);
        if (typeEnum == null) {
            return null;
        }

        return merchantLinkMapper.findByRelation(relationType, relationId);
    }

    @Override
    public List<BaiqiMerchantLinkEntity> findByType(BaiqiMerchantLinkTypeEnum tuiAEnum) {
        if (tuiAEnum == null) {
            return Collections.emptyList();
        }

        return merchantLinkMapper.findByType(tuiAEnum.getType());
    }

    @Override
    public List<BaiqiMerchantLinkEntity> findByTypeWithLinked(BaiqiMerchantLinkTypeEnum typeEnum) {
        if(typeEnum == null){
            return Collections.emptyList();
        }

        //find by type
        List<BaiqiMerchantLinkEntity> allList = findByType(typeEnum);
        if(CollectionUtils.isEmpty(allList)){
            return Collections.emptyList();
        }

        //filter
        return allList.stream().filter(e -> e.getMerchantId() != null).collect(Collectors.toList());
    }

    @Override
    public List<MerchantAdvertDto> findAllWithUnlinked(Long merchantId) {

        //查询所有广告主
        List<BaiqiMerchantLinkEntity> advertList = this.findByType(BaiqiMerchantLinkTypeEnum.TUI_A);

        //商家id精准查询
        if (merchantId != null) {
            return findWithMerchantId(merchantId, advertList);
        }

        //查询所有商家
        List<MerchantSimpleDto> allMerchantList = merchantService.findAllIdAndNames();
        if (CollectionUtils.isEmpty(allMerchantList)) {
            return Collections.emptyList();
        }

        //构建dto
        List<MerchantAdvertDto> result = Lists.newArrayListWithExpectedSize(allMerchantList.size());
        allMerchantList.forEach(e -> {
            MerchantAdvertDto merchantAdvertDto = new MerchantAdvertDto();

            merchantAdvertDto.setMerchantId(e.getId());
            merchantAdvertDto.setMerchantName(e.getMerchantName());

            //设置广告主信息
            setAdvertInfo(e.getId(), merchantAdvertDto, advertList);

            result.add(merchantAdvertDto);
        });

        // sort ,未关联的商家排前面
        return sortMerchant(result);
    }

    @Override
    public Boolean insert(MerchantAdvertDto merchantAdvertDto) {

        BaiqiMerchantLinkEntity insertEntity = new BaiqiMerchantLinkEntity();
        insertEntity.setRelationType(BaiqiMerchantLinkTypeEnum.TUI_A.getType());
        insertEntity.setRelationId(merchantAdvertDto.getAdvertId());
        insertEntity.setRelationName(merchantAdvertDto.getAdvertName());
        insertEntity.setMerchantId(merchantAdvertDto.getMerchantId());

        return merchantLinkMapper.insert(insertEntity) > 0;
    }

    @Override
    public Long findRelationIdByMerchantIdAndRelationType(Long merchantId, BaiqiMerchantLinkTypeEnum relationType) {
        BaiqiMerchantLinkEntity entity = merchantLinkMapper
            .findByMerchantIdAndRelationType(merchantId, relationType.getType());
        if (null == entity) {
            return 0L;
        }
        return entity.getRelationId();
    }

    /**
     * 排序
     *
     * @param list
     * @return
     */
    private List<MerchantAdvertDto> sortMerchant(List<MerchantAdvertDto> list) {

        //未关联
        List<MerchantAdvertDto> unlinkedList = list.stream()
                                                   .filter(e -> e.getAdvertId() == null)
                                                   .collect(Collectors.toList());

        //已关联
        List<MerchantAdvertDto> linkedList = list.stream()
                                                 .filter(e -> e.getAdvertId() != null)
                                                 .collect(Collectors.toList());

        List<MerchantAdvertDto> result = Lists.newArrayListWithExpectedSize(list.size());

        if (CollectionUtils.isNotEmpty(unlinkedList)) {
            result.addAll(unlinkedList);
        }

        if (CollectionUtils.isNotEmpty(linkedList)) {
            result.addAll(linkedList);
        }

        return result;
    }

    private void setAdvertInfo(Long merchantId, MerchantAdvertDto merchantAdvertDto,
        List<BaiqiMerchantLinkEntity> advertList) {

        if (CollectionUtils.isNotEmpty(advertList)) {

            //相关联的广告主
            BaiqiMerchantLinkEntity linkEntity = advertList.stream()
                                                           .filter(n -> merchantId.equals(n.getMerchantId()))
                                                           .findFirst()
                                                           .orElse(null);
            if (linkEntity != null) {
                merchantAdvertDto.setAdvertId(linkEntity.getRelationId());
                merchantAdvertDto.setAdvertName(linkEntity.getRelationName());
                merchantAdvertDto.setLinked(true);
            }
        }
    }

    /**
     * 商家id精准查询
     *
     * @param merchantId
     * @param advertList
     * @return
     */
    private List<MerchantAdvertDto> findWithMerchantId(Long merchantId, List<BaiqiMerchantLinkEntity> advertList) {

        //
        MerchantDto merchantDto = merchantService.findById(merchantId);
        if (merchantDto == null) {
            return Collections.emptyList();
        }

        MerchantAdvertDto merchantAdvertDto = new MerchantAdvertDto();

        //设置广告主信息
        setAdvertInfo(merchantId, merchantAdvertDto, advertList);

        merchantAdvertDto.setMerchantId(merchantId);
        merchantAdvertDto.setMerchantName(merchantDto.getMerchantName());

        return Lists.newArrayList(merchantAdvertDto);

    }
}
