package com.qiho.center.biz.service.page.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.ItemDetailDto;
import com.qiho.center.api.dto.ItemDto;
import com.qiho.center.api.dto.page.CollItemDto;
import com.qiho.center.api.dto.page.CollItemSaveDto;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.page.CollItemService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.daoh.qiho.page.CollItemMapper;
import com.qiho.center.common.entityd.qiho.page.CollItemEntity;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jiafeng.geng
 * @date 2018-07-24.
 */
@Service
public class CollItemServiceImpl implements CollItemService {

    private final static Logger LOGGER = LoggerFactory.getLogger(CollItemServiceImpl.class);

    @Resource
    private CollItemMapper collItemMapper;

    @Resource
    private ItemService itemService;

    @Override
    public List<CollItemDto> queryItemListBySort(Long collId) {
        // 获取当前集合页下所有的商品
        List<CollItemEntity> collItemEntityList = collItemMapper.queryItemListBySort(collId);
        if (CollectionUtils.isEmpty(collItemEntityList)) {
            LOGGER.info("当前集合页下没有商品, collId={}", collId);
            return Lists.newArrayList();
        }

        return converToCollItemDtoList(collItemEntityList);
    }

    @Override
    public List<CollItemDto> queryItemListBySortWithItemIds(Long collId, List<Long> collItemIds) {
        // 根据商品id集合获取当前集合页下的商品
        List<CollItemEntity> collItemEntityList = collItemMapper.queryItemListBySortWithItemIds(collId, collItemIds);
        if (CollectionUtils.isEmpty(collItemEntityList)) {
            LOGGER.info("当前集合页下没有指定商品, collId={}, collItemIds={}", collId, collItemIds);
            return Lists.newArrayList();
        }

        return converToCollItemDtoList(collItemEntityList);
    }

    private List<CollItemDto> converToCollItemDtoList(List<CollItemEntity> collItemEntityList) {
        List<CollItemDto> collItemDtoList = Lists.newArrayList();

        // 提取所有商品id
        List<Long> itemIds = collItemEntityList.stream().map(CollItemEntity::getItemId).collect(Collectors.toList());

        // 根据itemIds查询商品对象集合
        List<ItemDto> itemDtoList = itemService.findItemDtoByItemIds(new HashSet<>(itemIds));
        if (CollectionUtils.isEmpty(itemDtoList)) {
            LOGGER.info("当根据商品id没有查找到指定商品, itemIds={}", itemIds);
            return collItemDtoList;
        }

        // 类型转换
        collItemEntityList.forEach(collItemEntity ->
                itemDtoList.forEach(itemDto -> {
                            if (itemDto.getId().longValue() == collItemEntity.getItemId().longValue()) {
                                CollItemDto collItemDto = BeanUtils.copy(collItemEntity, CollItemDto.class);
                                collItemDto.setId(itemDto.getId());
                                collItemDto.setItemName(itemDto.getItemName());
                                collItemDto.setImage(itemDto.getExtParamValue("image"));
                                collItemDtoList.add(collItemDto);
                            }
                        }
                )
        );

        return collItemDtoList;
    }

    @Override
    public Integer addCollItemList(List<CollItemSaveDto> collItemSaveDtoList) {
        if (CollectionUtils.isEmpty(collItemSaveDtoList)) {
            LOGGER.info("要新增的商品集合为空, collItemSaveDtoList={}", collItemSaveDtoList);
            return 0;
        }

        List<CollItemEntity> collItemEntityList = Lists.newArrayList();

        for (CollItemSaveDto collItemSaveDto : collItemSaveDtoList) {
            //根据collId和itemId查询一下是否已经存在有效记录
            CollItemEntity collItemEntity = collItemMapper.queryCollItemByCollIdAndItemId(collItemSaveDto.getId(), collItemSaveDto.getItemId());
            if (null == collItemEntity) {
                collItemEntity = BeanUtils.copy(collItemSaveDto, CollItemEntity.class);
                collItemEntity.setCollId(collItemSaveDto.getId());
                collItemEntityList.add(collItemEntity);
            }
        }

        if (CollectionUtils.isEmpty(collItemEntityList)) {
            LOGGER.info("要新增的商品集合均已存在, collItemSaveDtoList={}", collItemSaveDtoList);
            return 0;
        }

        return collItemMapper.batchInsertCollItem(collItemEntityList);

    }

    @Override
    public Integer updateCollItemList(List<CollItemSaveDto> collItemSaveDtoList) {
        if (CollectionUtils.isEmpty(collItemSaveDtoList)) {
            LOGGER.info("要更新的商品集合为空, collItemSaveDtoList={}", collItemSaveDtoList);
            return 0;
        }

        List<CollItemEntity> collItemEntityList = Lists.newArrayList();

        for (CollItemSaveDto collItemSaveDto : collItemSaveDtoList) {
            CollItemEntity collItemEntity = BeanUtils.copy(collItemSaveDto, CollItemEntity.class);
            collItemEntity.setCollId(collItemSaveDto.getId());
            collItemEntity.setStyleConfig(collItemSaveDto.getStyleConfig());
            collItemEntityList.add(collItemEntity);
        }

        return collItemMapper.batchUpdateCollItem(collItemSaveDtoList.get(0).getId(), collItemEntityList);
    }

    @Override
    public Integer findMaxSort(Long collId) {
        return collItemMapper.findMaxSort(collId);
    }

    @Override
    public Integer deleted4Phy(Long collId, Long itemId) {
        return collItemMapper.deleted4Phy(collId, itemId);
    }

    @Override
    public Integer batchDeleted(List<Long> itemIds) {
        return collItemMapper.batchDeleted(itemIds);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer sort(Long collId, List<Long> itemIds) {
        // 判断排序数量和实际商品总数是否相同
        List<CollItemEntity> collItemEntityList = collItemMapper.queryItemListBySort(collId);
        if (itemIds.size() != collItemEntityList.size()) {
            throw new QihoException("传入id和集合页商品总数不一致");
        }

        // 类型转换
        ItemOrdering ordering = new ItemOrdering();
        ordering.setOrderItemIds(itemIds);
        collItemEntityList = Lists.transform(collItemEntityList, ordering);

        // 重载排序值
        int count = 0;
        for (CollItemEntity collItemEntity : collItemEntityList) {
            collItemEntity.setSort(count++);
        }

        return collItemMapper.batchUpdateCollItem(collId, collItemEntityList);
    }

    private class ItemOrdering implements Function<CollItemEntity, CollItemEntity> {

        private List<Long> orderItemIds;

        private int maxSort;

        public void setOrderItemIds(List<Long> orderItemIds) {
            this.orderItemIds = orderItemIds;
            this.maxSort = orderItemIds.size() - 1;
        }

        @Override
        public CollItemEntity apply(CollItemEntity input) {
            int index = orderItemIds.indexOf(input.getItemId());
            input.setSort(maxSort - index);
            return input;
        }
    }

    @Override
    public Integer updateStatusBatch(Long collId, List<Long> itemIds, Integer collItemStatus) {
        return collItemMapper.updateStatusBatch(collId, itemIds, collItemStatus);
    }

    @Override
    public CollItemDto queryItem(Long collId, Long itemId) {
        CollItemEntity collItemEntity = collItemMapper.queryItem(collId, itemId);
        return converToCollItemDto(collItemEntity);
    }

    private CollItemDto converToCollItemDto(CollItemEntity collItemEntity) {

        // 根据itemId查询商品对象
        ItemDetailDto itemDetailDto = itemService.queryItemDetail(collItemEntity.getItemId());
        if (null == itemDetailDto) {
            LOGGER.info("根据itemIds查询商品对象结果为空, collItemIds={}", collItemEntity.getItemId());
            return null;
        }

        // 类型转换
        CollItemDto collItemDto = BeanUtils.copy(itemDetailDto, CollItemDto.class);
        collItemDto.setSort(collItemEntity.getSort());
        collItemDto.setStyleConfig(collItemEntity.getStyleConfig());
        collItemDto.setCollItemStatus(collItemEntity.getCollItemStatus());
        collItemDto.setImage(itemDetailDto.getExtParamValue("image"));

        return collItemDto;
    }
}
