package com.qiho.center.biz.service.sms.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.qiho.center.api.constant.OrderStrategyConstant;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.sms.ChuangLanSmsReportDto;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.ordersms.SmsMobileStatusEnum;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.biz.engine.impl.ShotOrderEngine;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.service.sms.ChuangLanService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.daoh.qiho.ChuangLanSmsCommitMapper;
import com.qiho.center.common.daoh.qiho.ChuangLanSmsReportMapper;
import com.qiho.center.common.daoh.qiho.anticheate.AnticheateMapper;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.entityd.qiho.ChuangLanSmsCommitEntity;
import com.qiho.center.common.entityd.qiho.ChuangLanSmsReportEntity;
import com.qiho.center.common.entityd.qiho.anticheate.AnticheateEntity;
import com.qiho.center.common.util.HttpClientUtil;
import com.qiho.center.common.util.StringUtil;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * Created by weny.cai on 2018/5/2.
 */
@Service
public class ChuangLanServiceImpl implements ChuangLanService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ChuangLanServiceImpl.class);

    /*private static Map<String,String> reportMap = Maps.newHashMap();

    static {
        reportMap.put("bq000", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_ERROR);
        reportMap.put("bq001", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_CLOSE);
        reportMap.put("bq002", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_BLACKLIST);
        reportMap.put("bq003", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_EMPTY);
        reportMap.put("bq999", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_ERROR);
        reportMap.put("bq888", OrderStrategyConstant.SHOTORDER_TIMEOUT_RECEIVED_SUCCESS);
    }*/


    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Resource
    private ShotOrderEngine shotOrderEngine;

    @Autowired
    ChuangLanSmsCommitMapper chuangLanSmsCommitMapper;

    @Autowired
    ChuangLanSmsReportMapper chuangLanSmsReportMapper;

    @Autowired
    AnticheateMapper anticheateMapper;

    @Resource
    private OrderService orderService;

    @Value("${qiho.ssjx.sms.status.push}")
    private String pushUrl;

    @Value("${qiho.order.ssjx.merchantid}")
    private String ssjxMerchartId;

    @Transactional(value = DsConstants.DATABASE_QIHO, rollbackFor = Exception.class)
    @Override
    public void processReport(ChuangLanSmsCommitEntity commitEntity,ChuangLanSmsReportDto dto) {

        DBTimeProfile.enter("invoke ChuangLanService.processReport");

        //判断是否已经做了超时处理
        Map<String,String> map = new HashMap<>();
        String status = dto.getStatus();
        String baiqiCode = getBaiqiCode(status);//获取对应baiqiCode

        LOGGER.info("返回的短信码为:{} ,命中的短信码为:{},orderId: {} ,mobile: {} ,handlerStatus:{}",
                status, baiqiCode, commitEntity.getOrderId(), dto.getMobile(),commitEntity.getHandlerStatus());

        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(commitEntity.getOrderId());

        if (entity != null) {
            // 正式订单不为空

            if(StringUtils.equals(entity.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())
                    && StringUtils.isBlank(entity.getOrderProcessWay())){

                LOGGER.info("接收到创蓝消息启动踢单规则， orderId： {} ", entity.getOrderId());

                // 未处理 启动提单规则
                OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(entity,OrderSnapshotDto.class);
                orderSnapshotDto.setSmsRetCode(baiqiCode);

                DBTimeProfile.enter("ChuangLanServiceImpl.processReport 踢单规则, orderId: " + entity.getOrderId());
                shotOrderEngine.process(orderSnapshotDto);
                DBTimeProfile.release();

            }else {
                // 踢单规则已处理过该订单 进行后续操作
                List<AnticheateEntity> anticheateEntities = anticheateMapper.selectByOrderId(commitEntity.getOrderId());

                if(isOrderToSend(entity, anticheateEntities, baiqiCode)){
                    // 如果返回的状态为正常 订单修改为待发货
                    DBTimeProfile.enter("将待审核的订单变成待发货");
                    LOGGER.info("将待审核的订单变成待发货并修改防作弊信息，orderId： {} ", entity.getOrderId());
                    OrderAuditParams params = new OrderAuditParams();
                    params.setOrderId(commitEntity.getOrderId());
                    params.setAuditResult(OrderAuditParams.AUDIT_SUCCESS);
                    orderService.auditOrder(params);

                    // 修改提单规则信息
                    updateAnticheatInfo(baiqiCode, commitEntity);
                    DBTimeProfile.release();

                } else if (isUpdateAnticheat(entity, anticheateEntities, baiqiCode)) {

                    LOGGER.info("订单修改防作弊细信息，orderId： {} ", entity.getOrderId());

                    // 修改防作弊信息
                    updateAnticheatInfo(baiqiCode, commitEntity);

                }

                map.put("update","baiqi");
            }

            // 推送手机号状态给盛世嘉祥
            pushMobileToSSJX(entity, map, dto);
        }

        // 处理创蓝报告数据
        dealChuangLanReportData(dto, commitEntity);

        DBTimeProfile.release();
    }

    /**
     * 推送手机号状态给盛世嘉祥
     * @param entity
     * @param map
     * @param dto
     */
    private void pushMobileToSSJX(QihoOrderSnapshotEntity entity, Map<String,String> map, ChuangLanSmsReportDto dto){
        List<Long> merchantIds = StringUtil.stringToListOrderByIds(ssjxMerchartId);
        if(merchantIds.contains(entity.getMerchantId())) {
            DBTimeProfile.enter("推送盛世嘉祥");
            try{
                //推送给盛世家祥
                map.put("phone", dto.getMobile());
                map.put("msgBacKcode", dto.getStatus());
                map.put("sendTime", Long.toString(System.currentTimeMillis()));
                HttpClientUtil.sendPost(pushUrl, map);
            } catch (Exception e) {
                LOGGER.error("手机号状态推送盛世嘉祥异常， orderId：{}，mobile：{} ", entity.getOrderId(), dto.getMobile(), e);
            }
            DBTimeProfile.release();
        }
    }

    /**
     * 处理创蓝数据报告
     * @param dto
     * @param commitEntity
     */
    private void dealChuangLanReportData(ChuangLanSmsReportDto dto, ChuangLanSmsCommitEntity commitEntity){
        DBTimeProfile.enter("处理创蓝报告数据");
        //短信报告存入数据库
        ChuangLanSmsReportEntity reportEntity = BeanUtils.copy(dto,ChuangLanSmsReportEntity.class);
        reportEntity.setReportStatus(dto.getStatus());
        reportEntity.setReportStatusDesc(dto.getStatusDesc());
        reportEntity.setOrderId(commitEntity.getOrderId());
        chuangLanSmsReportMapper.insert(reportEntity);

        //删除处理过的
        chuangLanSmsCommitMapper.deleteByPrimaryKey(commitEntity.getId());
        DBTimeProfile.release();
    }

    /**
     * 判断是否能变更为待发货状态
     *  订单是待审核状态 并且 处理方式不为空 且 只命中超时的踢单规则 时return true
     * @param entity
     * @param anticheateEntities
     * @param baiqiCode
     * @return true-可以
     */
    private boolean isOrderToSend(QihoOrderSnapshotEntity entity, List<AnticheateEntity> anticheateEntities, String baiqiCode){

        return  StringUtils.equals(entity.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())
                && StringUtils.isNotBlank(entity.getOrderProcessWay())
                && anticheateEntities.size() == 1
                && StringUtils.equals(anticheateEntities.get(0).getRule(), OrderStrategyConstant.SHOTORDER_TIMEOUT_NAME)
                && baiqiCode.equals("bq888");
    }

    /**
     * 判断是否可以更新命中防作弊状态
     * 当订单是待审核状态且orderProcessWay字段不为空 命中的防作弊规则包含超时的 return true
     * @param entity
     * @param anticheateEntities
     * @param baiqiCode
     * @return true-可以
     */
    private boolean isUpdateAnticheat(QihoOrderSnapshotEntity entity, List<AnticheateEntity> anticheateEntities, String baiqiCode) {
        if (!(StringUtils.equals(entity.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())
            && StringUtils.isNotBlank(entity.getOrderProcessWay()))) {
            return false;
        }

        boolean flag = false;
        for (AnticheateEntity an : anticheateEntities) {
            if (StringUtils.equals(an.getRule(), OrderStrategyConstant.SHOTORDER_TIMEOUT_NAME)) {
                flag = true;
                break;
            }
        }

        return flag;
    }

    /**
     * 根据短信服务商的返回码，后置更新防作弊信息
     * @param baiqiCode
     * @param commitEntity
     */
    private void updateAnticheatInfo(String baiqiCode, ChuangLanSmsCommitEntity commitEntity){
        AnticheateEntity record = new AnticheateEntity();
        record.setOrderId(commitEntity.getOrderId());
        record.setRule(OrderStrategyConstant.SHOTORDER_TIMEOUT_NAME);
        record.setNewRule(SmsMobileStatusEnum.getUpdateTextByCode(baiqiCode));
        anticheateMapper.updateByOrderId(record);
    }


    /**
     * 通过映射规则
     * 获取baiqiCode
     * @param status
     * @return
     */
    @NotNull
    private String getBaiqiCode(String status) {
        String baiqiCode = "";
        if(status.equals("ERRNUM")||status.equals("SGIP:1")||status.equals("MN:0001")
            ||status.equals("MK:0001")||status.equals("CB:0001")||status.equals("6640")
            ||status.equals("0006")||status.equals("XL:169")||status.equals("1")||
            status.equals("10")||status.equals("67")||status.equals("61")||status.equals("72")||
            status.equals("101")||status.equals("-1")||status.equals("RP:1")||
            status.equals("Deliver")||status.equals("LT:0001")){
            baiqiCode = "bq003";
        }else if(status.equals("MBBLACK")){
            baiqiCode = "bq002";
        }else if(status.startsWith("MK")||status.startsWith("MI")||status.startsWith("MN")||
            status.startsWith("SGIP")||status.startsWith("UNDELIV")||status.startsWith("CU")||
            status.startsWith("RP:")||
            status.equals("0005")||status.equals("IB:0064")||status.equals("4") ||status.equals("54")||
            status.equals("51")||status.equals("-37")||status.equals("53")||
            status.equals("4716")||status.equals("0007")||status.equals("5")||status.equals("50")||
            status.equals("45")||status.equals("36")||status.equals("56") ||status.equals("213")||
            status.equals("ID:0102")||status.equals("SPMSERR:102")){
            baiqiCode = "bq001";
        }else if(status.equals("DELIVRD")){
            baiqiCode = "bq888";
        }else {
            baiqiCode = "bq000";
        }
        return baiqiCode;
    }









}
