package com.qiho.center.biz.task;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.order.BatchOrderCancelDto;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.biz.service.OSSFileService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.biz.task.bo.OrderCancelDo;
import com.qiho.center.biz.task.to.OrderCancelTo;
import com.qiho.center.common.daoh.qiho.OrderConfirmTaskDetailMapper;
import com.qiho.center.common.daoh.qiho.OrderConfirmTaskMapper;
import com.qiho.center.common.daoh.qiho.OrderSnapshotMapper;
import com.qiho.center.common.entityd.qiho.OrderConfirmTaskDetailEntity;
import com.qiho.center.common.entityd.qiho.OrderConfirmTaskEntity;
import com.qiho.center.common.entityd.qiho.OrderSnapshotEntity;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * 取消批量发货
 * 任务线程
 * Created by weny.cai on 2018/3/19.
 */
@Component
@Scope("prototype")
public class BatchOrderCancelTask implements Runnable{

    private Logger logger = LoggerFactory.getLogger(BatchOrderCancelTask.class);

    @Autowired
    private OSSFileService ossFileService;

    private BatchOrderCancelDto dto;

    @Autowired
    StringRedisTemplate redisTemplate;

    private BoundHashOperations<String, String, String> ops;

    private OrderConfirmTaskEntity taskEntity;

    private OrderConfirmTaskDetailEntity detailEntity;

    @Autowired
    OrderConfirmTaskMapper orderConfirmTaskMapper;

    @Autowired
    OrderConfirmTaskDetailMapper orderConfirmTaskDetailMapper;

    @Autowired
    OrderService orderService;

    @Autowired
    OrderSnapshotMapper orderSnapshotMapper;

    private long startTime;

    private int failNum;

    private String failLines = "";

    List<OrderCancelDo> dos;

    List<OrderConfirmTaskDetailEntity> detailEntities = Lists.newArrayList();

    Map<String, OrderSnapshotEntity> entityMap;

    /** 订单关闭时的备注 */
    private String remark;

    /**
     * 初始化参数
     * @param batchOrderCancelDto
     */
    public void init(BatchOrderCancelDto batchOrderCancelDto,OrderConfirmTaskEntity taskEntity) {
        this.dto = batchOrderCancelDto;
        ops = redisTemplate.boundHashOps(dto.getTaskId());
        this.startTime = System.currentTimeMillis();
        this.taskEntity = taskEntity;
    }

    public void init(BatchOrderCancelDto batchOrderCancelDto,OrderConfirmTaskEntity taskEntity,List<OrderCancelDo> dos){
        this.dos = dos;
        this.dto = batchOrderCancelDto;
        ops = redisTemplate.boundHashOps(dto.getTaskId());
        this.startTime = System.currentTimeMillis();
        this.taskEntity = taskEntity;
    }

    @Override
    public void run() {
        //获取集合
        if(dto.getTaskType()<=2){
            this.getOrderDosByExcel();
        }
        //初始化task 插入数据库
        initTask();

        //校验参数  重复的订单列为失败
        checkDos();

        if(!dos.isEmpty()){
            //获取订单map 用于处理
            getStringOrderSnapshotEntityMap();
            //批量处理 订单数据
            handlerBos();
        }

        //最后处理的结果 存入数据库
        handlerResult();
    }

    private void checkDos() {
        List<OrderCancelDo> list = Lists.newArrayList();
        List<String> maps = Lists.newArrayList();
        for (OrderCancelDo aDo : dos) {
            //如果已经包含订单ID 列为重复 并不加入处理集合
            if(StringUtils.isNotBlank(aDo.getOrderId())&&maps.contains(aDo.getOrderId())){
                OrderConfirmTaskDetailEntity detailEntity = new OrderConfirmTaskDetailEntity();
                detailEntity.setOrderId(aDo.getOrderId());
                detailEntity.setOrderConfirmTaskId(taskEntity.getId());
                detailEntity.setLineNum(aDo.getLineNum());
                detailEntity.setFailMsg("订单ID重复");
                detailEntities.add(detailEntity);
                failNum++;
                failLines = failLines + aDo.getLineNum() + ",";
                ops.increment("handleCount", 1);
                continue;
            }
            maps.add(aDo.getOrderId());
            list.add(aDo);
        }
        this.dos = list;
    }

    private void handlerResult() {
        //批量插入处理结果
        if(!detailEntities.isEmpty()){
            orderConfirmTaskDetailMapper.batchInsert(detailEntities);
        }
        ops.put("status","over");
        ops.put("failLines",failLines);
        ops.put("failNum",String.valueOf(failNum));
        this.taskEntity.setConsumeTime(System.currentTimeMillis()-this.startTime);
        this.taskEntity.setFailNum(this.failNum);
        this.taskEntity.setTaskStatus(1);
        orderConfirmTaskMapper.updateByPrimaryKeySelective(taskEntity);
    }

    public void handlerBos() {
        //处理集合
        for (OrderCancelDo bo : dos) {
            Integer lineNum = bo.getLineNum();
            try{
                OrderSnapshotEntity orderSnapshotEntity = entityMap.get(bo.getOrderId());
                boolean checkFlag = this.checkParam(bo,orderSnapshotEntity,lineNum);
                if (!checkFlag){
                    detailEntities.add(detailEntity);
                    this.failNum++;
                    failLines = failLines + lineNum + ",";
                    continue;
                }
                orderService.orderCancel(bo.getOrderId(), this.getRemark());
            }catch (Exception e){  //数据库处理失败
                this.failNum++;
                failLines = failLines + lineNum + ",";
                detailEntity.setFailMsg("数据库操作异常!"+e.getMessage());
                detailEntities.add(detailEntity);
            }finally {
                this.ops.increment("handleCount", 1);
            }
        }
    }

    /**
     * 初始化task 插入数据库
     */
    private void initTask() {
        ops.put("totalCount", String.valueOf(dos.size()));
        ops.put("status","handling");
        taskEntity.setAllNum(dos.size());
        if(dos.size()==0){
            taskEntity.setTaskStatus(1);
        }else{
            taskEntity.setTaskStatus(0);
        }
        orderConfirmTaskMapper.updateByPrimaryKeySelective(taskEntity);
    }

    /**
     * 校验参数
     * @param orderCancelDo
     * @param lineNum
     * @return
     */
    private boolean checkParam(OrderCancelDo orderCancelDo,OrderSnapshotEntity orderSnapshotEntity, int lineNum) {
        this.detailEntity = new OrderConfirmTaskDetailEntity();
        detailEntity.setOrderId(orderCancelDo.getOrderId());
        detailEntity.setOrderConfirmTaskId(taskEntity.getId());
        detailEntity.setLineNum(lineNum);

        if("ERROR_CELL".equals(orderCancelDo.getOrderId())){
            detailEntity.setOrderId("");
            detailEntity.setFailMsg("错误的单元格");
            return false;
        }

        if(StringUtils.isBlank(orderCancelDo.getOrderId())){
            detailEntity.setFailMsg("订单ID为空!");
            return false;
        }

        if(orderSnapshotEntity.getDelivery().equals("ERP")){
            detailEntity.setFailMsg("ERP订单操作失败!");
            return false;
        }

        if(orderSnapshotEntity == null){
            detailEntity.setFailMsg("订单不存在!");
            return false;
        }

        //订单必须处于待发货状态
        if(!(OrderStatusEnum.TO_SEND.getCode().equals(orderSnapshotEntity.getOrderStatus())||
                OrderStatusEnum.TO_APPROVE.getCode().equals(orderSnapshotEntity.getOrderStatus()))){
            detailEntity.setFailMsg("订单非待发货或待审核状态!");

            // 判断如果订单处于交易关闭状态，则打印应用日志
            if (orderSnapshotEntity.getOrderStatus().equals(OrderStatusEnum.CLOSED.getCode())) {
                logger.info("订单号id: " + orderSnapshotEntity.getOrderId() + " 上传物流单号失败,失败原因:该订单已经关闭");
            }

            return false;
        }

        //-1奇货后台传的数据 不进行商家校验
        if(orderCancelDo.getMerchantId()==-1){
            return true;
        }

        //对商家进行校验 操作权限
        if (!orderCancelDo.getMerchantId().equals(orderSnapshotEntity.getMerchantId())) {
            detailEntity.setFailMsg("不属于导入的商家!");
            return false;
        }

        return true;
    }

    /**
     * 读取文件
     * 转化为集合
     * @return
     */
    private void getOrderDosByExcel() {
        InputStream input = ossFileService.getOssFileInputStream(dto.getFileUrl());
        ImportParams params = new ImportParams();
        String[] importFields = {"订单ID"};
        params.setImportFields(importFields);
        params.setKeyIndex(null);
        List<OrderCancelTo> list = Lists.newArrayList();
        try {
            list = ExcelImportUtil.importExcel(input, OrderCancelTo.class, params);
        } catch (Exception e) {
            logger.warn("导入文件的表头格式不正确!");
        }
        List<OrderCancelDo> dos = Lists.newArrayList();
        for (OrderCancelTo to : list) {
            if(StringUtils.isBlank(to.getOrderId())){
                continue;
            }
            OrderCancelDo bo = new OrderCancelDo();
            bo.setLineNum(to.getRowNum());
            bo.setOrderId(to.getOrderId().trim());
            bo.setMerchantId(dto.getMerchantId());
            dos.add(bo);
        }
        this.dos = dos;
    }

    /**
     * 从数据库中
     * 获取存在的订单
     * @return
     */
    private void getStringOrderSnapshotEntityMap() {
        List<String> orderIds = Lists.newArrayList();
        for (OrderCancelDo orderCancelBo : dos) {
            orderIds.add(orderCancelBo.getOrderId());
        }
        //数据库中查询所有订单数据  转化为map
        List<OrderSnapshotEntity> entities = orderSnapshotMapper.selectByOrderIds(orderIds);
        Map<String,OrderSnapshotEntity> entityMap = Maps.newHashMap();
        for (OrderSnapshotEntity entity : entities) {
            entityMap.put(entity.getOrderId(),entity);
        }
        this.entityMap = entityMap;
    }


    public String getRemark() {
        if (remark == null) {
            return "";
        }
        return remark;
    }

    public void setRemark(String remark) {
        this.remark = remark;
    }
}
