package com.mediamain.android.base.download;

import android.app.Application;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;

import com.liulishuo.okdownload.DownloadListener;
import com.liulishuo.okdownload.DownloadTask;
import com.liulishuo.okdownload.OkDownload;
import com.liulishuo.okdownload.core.breakpoint.BreakpointInfo;
import com.liulishuo.okdownload.core.cause.EndCause;
import com.liulishuo.okdownload.core.cause.ResumeFailedCause;
import com.liulishuo.okdownload.core.listener.DownloadListener1;
import com.liulishuo.okdownload.core.listener.assist.Listener1Assist;
import com.mediamain.android.R;
import com.mediamain.android.base.config.Constants;
import com.mediamain.android.base.config.FoxBaseConstants;
import com.mediamain.android.base.download.broadcast.NotificationBroadCast;
import com.mediamain.android.base.util.FoxBaseCommonUtils;
import com.mediamain.android.base.util.FoxBaseEncryptUtils;
import com.mediamain.android.base.util.FoxBaseLogUtils;
import com.mediamain.android.base.util.FoxBaseSPUtils;
import com.mediamain.android.base.util.FoxBaseThreadUtils;
import com.mediamain.android.base.util.FoxBaseUtils;
import com.mediamain.android.base.util.crash.FoxBaseCrashUtils;
import com.mediamain.android.view.imageloader.FoxImageWorker;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * <p> File description: <p>
 * <p> Creator: Adroll   <p>
 * <p> Created date: 11/27/20 <p>
 * * * * * * * * * * * * * * * * * * * * * *
 * Thinking is more important than coding *
 * * * * * * * * * * * * * * * * * * * * * *
 */
public class DownloadController {

    private static volatile DownloadController INSTANCE;

    private static boolean isSupportDownload;

    private final HashMap<String, Bitmap> bitmapMap = new HashMap<>(10);
    private final HashMap<String, Future<Bitmap>> futureMap = new HashMap<>(10);

    private final String parentFileDir = FoxBaseCommonUtils.getDownLoadPath(Constants.CACHE_NAME);

    /**
     * 当前下载状态
     */
    public enum DownloadStatus {
        // 未下载
        UN_DOWNLOAD,
        // 正在下载
        DOWNLOADING,
        // 已下载未安装
        UN_INSTALL,
        // 已安装
        INSTALLED,
        // 暂停下载
        STOP_DOWNLOAD,
        // 下载失败
        DOWNLOAD_FAIL,
    }

    public static DownloadController getInstance(){
        if (INSTANCE == null){
            synchronized (DownloadController.class){
                if (INSTANCE == null){
                    INSTANCE = new DownloadController();
                    isSupportDownload = FoxBaseSPUtils.getInstance().getBoolean(FoxBaseConstants.KEY_TUIA_SDK_ISSUPPORTDOWNLOAD, true);
                }
            }
        }
        return INSTANCE;
    }

    public DownloadStatus getStatus(String url, String appPackageName, boolean isCancelTask) {
        return statusCheck(url, appPackageName, isCancelTask);
    }

    public NotificationManagerCompat showNotification(String url, String packageName, String icon, int progress,
                                                      boolean isCancelTask, String tuiaId, boolean isShowNotification){
        return createNotification(url, packageName, icon, progress, isCancelTask, tuiaId, isShowNotification);
    }

    public void removeBitmapFuture(String icon){
        if (futureMap.containsKey(icon)){
            futureMap.get(icon).cancel(true);
            futureMap.remove(icon);
        }
    }

    public DownloadTask download(final String url, final String fileName, final String icon, boolean isReDownload, int priority,
                                 final DownloadCallback downloadCallback){
        if (!isSupportDownload){
            FoxBaseLogUtils.d("Current setting not support download");
            return null;
        }

        if (TextUtils.isEmpty(url) || TextUtils.isEmpty(fileName)){
            return null;
        }

        DownloadTask task = new DownloadTask.Builder(url, parentFileDir, fileName)
                .setMinIntervalMillisCallbackProcess(50)
                .setPassIfAlreadyCompleted(isReDownload)
                .setPriority(priority)
                .build();

        task.enqueue(new DownloadListener1() {
            @Override
            public void taskStart(@NonNull DownloadTask task, @NonNull Listener1Assist.Listener1Model model) {
                if (downloadCallback != null){
                    downloadCallback.onTaskStart();
                }
            }

            @Override
            public void retry(@NonNull DownloadTask task, @NonNull ResumeFailedCause cause) {

            }

            @Override
            public void connected(@NonNull DownloadTask task, int blockCount, long currentOffset, long totalLength) {

            }

            @Override
            public void progress(@NonNull DownloadTask task, long currentOffset, long totalLength) {
                if (downloadCallback == null){
                    return;
                }

                final float p = (float) currentOffset * 100 / totalLength;
                if (downloadCallback.cancelTask()){
                    task.cancel();
                    // 防止暂停任务时偶现通知栏状态不更新，手动进行延时更新通知栏
                    FoxBaseThreadUtils.executeBySingleWithDelay(new FoxBaseThreadUtils.Task<Float>() {

                        @Override
                        public Float doInBackground() throws Throwable {
                            createNotification(url, "", icon, (int)p, true, "", true);
                            return null;
                        }

                        @Override
                        public void onSuccess(Float result) {

                        }

                        @Override
                        public void onCancel() {

                        }

                        @Override
                        public void onFail(Throwable t) {

                        }
                    }, 500, TimeUnit.MILLISECONDS);
                    return;
                }

                downloadCallback.updateProgress(p);
            }

            @Override
            public void taskEnd(@NonNull DownloadTask task, @NonNull EndCause cause, @Nullable Exception realCause, @NonNull Listener1Assist.Listener1Model model) {
                if (downloadCallback != null){
                    downloadCallback.onTaskEnd();
                }
            }
        });

        return task;
    }

    /**
     * 断点下载，继续下载任务
     * @param type：
     *            1 -> 异步下载
     *            2 -> 同步下载
     */
    public void taskContinue(@NonNull DownloadTask task, DownloadListener listener, int type){
        if (type == 1){
            task.enqueue(listener);
        }else if (type == 2){
            task.execute(listener);
        }
    }

    private DownloadStatus statusCheck(String url, String appPackageName, boolean isCancelTask){
        try {
            Application context = FoxBaseUtils.getApp();
            if (context == null){
                return DownloadStatus.DOWNLOAD_FAIL;
            }

            // 检查是否安装应用
            if (checkAppInstalled(context, appPackageName)){
                return DownloadStatus.INSTALLED;
            }

            if (!FoxBaseSPUtils.getInstance().containsKey(url)){
                File file = FoxBaseCommonUtils.checkFileExit(Constants.CACHE_NAME, FoxBaseEncryptUtils.encryptMD5ToString(url) + "tm.apk");
                // 检查是否已下载未安装
                if (file != null && file.exists()){
                    return DownloadStatus.UN_INSTALL;
                }
                // 未安装
                else {
                    return DownloadStatus.UN_DOWNLOAD;
                }
            } else {
                // 下载中杀死应用时，重新将状态置于未下载
                String temp = FoxBaseSPUtils.getInstance().getString(url, "");
                if (!TextUtils.isEmpty(temp)){
                    String[] tempArray = temp.split(",");
                    if (tempArray != null && tempArray.length > 1){
                        BreakpointInfo info = OkDownload.with().breakpointStore().get(Integer.parseInt(tempArray[1]));
                        if (info == null){
                            return DownloadStatus.UN_DOWNLOAD;
                        }
                    }
                }
            }

            // 检查是否暂停下载
            if (isCancelTask){
                return DownloadStatus.STOP_DOWNLOAD;
            }
            // 下载中
            else {
                return DownloadStatus.DOWNLOADING;
            }
        }catch (Exception e){
            return DownloadStatus.DOWNLOAD_FAIL;
        }
    }

    private NotificationManagerCompat createNotification(String url, String packageName, String icon, int progress,
                                                         boolean isCancelTask, String tuiaId, boolean isShowNotification) {
        try {
            String channelId = "channelId";
            CharSequence channelName = "channelName";
            String channelDescription = "channelDescription";

            int channelImportance = NotificationManager.IMPORTANCE_DEFAULT;
            int iconTemp = com.mediamain.android.R.drawable.fox_notification_download;

            Context context = FoxBaseUtils.getApp();

            // 创建通知栏广播
            Intent intent = new Intent(context, NotificationBroadCast.class);
            intent.putExtra("url", url);
            intent.putExtra("packageName", packageName);
            intent.putExtra("icon", icon);
            intent.putExtra("tuiaId", tuiaId);
            PendingIntent pendingIntent = PendingIntent.getBroadcast(context.getApplicationContext(),
                    url.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT);

            // 获取通知栏网络图片
            createBitmap(icon, context);
            Bitmap bitmapTemp = null;
            if (bitmapMap.containsKey(icon)){
                bitmapTemp = bitmapMap.get(icon);
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, channelImportance);
                // 设置描述 最长30字符
                notificationChannel.setDescription(channelDescription);
                // 该渠道的通知是否使用震动
                notificationChannel.enableLights(false);
                notificationChannel.enableVibration(false);
                notificationChannel.setVibrationPattern(new long[]{0});
                notificationChannel.setSound(null, null);
                // 设置显示模式
                notificationChannel.setLockscreenVisibility(NotificationCompat.VISIBILITY_PUBLIC);

                NotificationManager notificationManager = (NotificationManager) FoxBaseUtils.getApp().getSystemService(Context.NOTIFICATION_SERVICE);
                notificationManager.createNotificationChannel(notificationChannel);
            }

            NotificationCompat.Builder notificationCompatBuilder = new NotificationCompat.Builder(context.getApplicationContext(), channelId);
            Notification notification;
            if (progress < 100) {
                notification = notificationCompatBuilder
                        // Title for API <16 (4.0 and below) devices.
                        .setContentTitle(isCancelTask ? "暂停下载" : "正在下载")
                        .setSmallIcon(iconTemp)
                        .setLargeIcon((bitmapTemp != null) ? bitmapTemp : BitmapFactory.decodeResource(context.getResources(), iconTemp))
                        .setDefaults(NotificationCompat.DEFAULT_ALL)
                        .setColor(ContextCompat.getColor(context.getApplicationContext(), R.color.colorPrimary))
                        .setCategory(Notification.CATEGORY_REMINDER)
                        .setPriority(NotificationCompat.PRIORITY_MAX)
                        .setContentIntent(pendingIntent)
                        .setVibrate(new long[]{0})
                        .setSound(null)
                        .setProgress(100, progress, false)
                        .build();
            } else if (progress == 100){
                notification = notificationCompatBuilder
                        // Title for API <16 (4.0 and below) devices.
                        .setContentTitle("立即安装")
                        // Content for API <24 (7.0 and below) devices.
                        .setContentText("点击领取福利")
                        .setSmallIcon(iconTemp)
                        .setLargeIcon((bitmapTemp != null) ? bitmapTemp : BitmapFactory.decodeResource(context.getResources(), iconTemp))
                        .setDefaults(NotificationCompat.DEFAULT_ALL)
                        .setColor(ContextCompat.getColor(context.getApplicationContext(), R.color.colorPrimary))
                        .setCategory(Notification.CATEGORY_REMINDER)
                        .setPriority(NotificationCompat.PRIORITY_MAX)
                        .setContentIntent(pendingIntent)
                        .setVibrate(new long[]{0})
                        .setSound(null)
                        .build();
            } else {
                notification = notificationCompatBuilder
                        // Title for API <16 (4.0 and below) devices.
                        .setContentTitle("立即打开")
                        // Content for API <24 (7.0 and below) devices.
                        .setContentText("点击领取福利")
                        .setSmallIcon(iconTemp)
                        .setLargeIcon((bitmapTemp != null) ? bitmapTemp : BitmapFactory.decodeResource(context.getResources(), iconTemp))
                        .setDefaults(NotificationCompat.DEFAULT_ALL)
                        .setColor(ContextCompat.getColor(context.getApplicationContext(), R.color.colorPrimary))
                        .setCategory(Notification.CATEGORY_REMINDER)
                        .setPriority(NotificationCompat.PRIORITY_MAX)
                        .setContentIntent(pendingIntent)
                        .setVibrate(new long[]{0})
                        .setSound(null)
                        .build();
            }

            NotificationManagerCompat notificationManagerCompat = NotificationManagerCompat.from(context.getApplicationContext());

            if (isShowNotification){
                // TODO: 12/7/20 如果开多任务打开注释即可
//            notificationManagerCompat.notify(url.hashCode(), notification);
                notificationManagerCompat.notify(1, notification);
            }else {
//            notificationManagerCompat.cancel(url.hashCode(), notification);
                notificationManagerCompat.cancel(1);
            }

            return notificationManagerCompat;
        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
        }

        return null;
    }

    private void createBitmap(final String iconUrl, Context context){
        if (TextUtils.isEmpty(iconUrl) || context == null || futureMap.containsKey(iconUrl)){
            return;
        }

        Future<Bitmap> future = FoxImageWorker.getInstance().getBitmapFromUrl(context, 0, iconUrl, new FoxImageWorker.LoadImgCallable() {
            @Override
            public void setImage(Bitmap bitmap, String url) {
                bitmapMap.put(iconUrl, bitmap);
            }

            @Override
            public void loadFaild() {
            }
        });

        futureMap.put(iconUrl, future);
    }

    /**
     * 检查应用是否安装
     */
    private boolean checkAppInstalled(Context context, String appPackageName){
        if (TextUtils.isEmpty(appPackageName)){
            return false;
        }

        PackageManager packageManager = context.getPackageManager();
        List<PackageInfo> packageInfos = packageManager.getInstalledPackages(0);
        if (packageInfos != null){
            for (int i=0; i<packageInfos.size(); i++){
                String packageName = packageInfos.get(i).packageName;
                if (appPackageName.equals(packageName)){
                    return true;
                }
            }
        }

        return false;
    }
}
