package com.mediamain.android.base.download;

import android.content.Intent;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.liulishuo.okdownload.OkDownload;
import com.mediamain.android.base.config.Constants;
import com.mediamain.android.base.data.DownloadTaskBean;
import com.mediamain.android.base.download.broadcast.DownloadBroadCast;
import com.mediamain.android.base.util.FoxBaseCommonUtils;
import com.mediamain.android.base.util.FoxBaseEncryptUtils;
import com.mediamain.android.base.util.FoxBaseSPUtils;
import com.mediamain.android.base.util.FoxBaseUtils;

import java.io.File;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p> File description: <p>
 * <p> Creator: Adroll   <p>
 * <p> Created date: 2020/12/27 <p>
 * * * * * * * * * * * * * * * * * * * * * *
 * Thinking is more important than coding *
 * * * * * * * * * * * * * * * * * * * * * *
 */
public class IntegrateH5MultiDownloadManager {

    private static volatile IntegrateH5MultiDownloadManager INSTANCE;

    private static Handler handler = new Handler(Looper.getMainLooper());

    private ConcurrentHashMap<String, DownloadTaskBean> beanHashMap = new ConcurrentHashMap<>();
    private ConcurrentHashMap<String, DownloadMethods> downloadHashMap = new ConcurrentHashMap<>();
    private ConcurrentHashMap<String, DownloadMethods> completedDownloadMap = new ConcurrentHashMap<>();

    private final static int DOWNLOAD_TASK_COUNT_LIMITATION = 2;

    private IntegrateH5MultiDownloadManager(){}

    public static IntegrateH5MultiDownloadManager getInstance() {
        if (INSTANCE == null){
            synchronized (IntegrateH5MultiDownloadManager.class){
                if (INSTANCE == null){
                    INSTANCE = new IntegrateH5MultiDownloadManager();
                }
            }
        }

        return INSTANCE;
    }

    private DownloadMethods preCheckTask(String url, String packageName){
        if (TextUtils.isEmpty(url) || TextUtils.isEmpty(packageName)){
            return null;
        }

        Set<String> packageList = downloadHashMap.keySet();
        for (String s : packageList){
            if (url.equals(s)){
                return downloadHashMap.get(s);
            }
        }

        return null;
    }

    private void dependencyDownloadBroadcast(String url, String packageName, String icon, String tuiaId){
        DownloadBroadCast downloadBroadCast = FoxBaseCommonUtils.registerDownloadBroadCast();
        if(null != downloadBroadCast) {
            downloadBroadCast.setBean(url, packageName, icon, tuiaId);
        }
    }

    private DownloadMethods createDownloadMethods(String url, String packageName, final String icon, String tuiaId){
        final DownloadMethods downloadMethods = new IntegrateH5Download(url, packageName, icon, tuiaId) {
            @Override
            public void downloadCompleted(String url) {
                completedDownloadMap.put(url, downloadHashMap.get(url));
                downloadHashMap.remove(url);
                beanHashMap.remove(url);
                DownloadController.getInstance().removeBitmapFuture(icon);
            }
        };

        downloadHashMap.put(url, downloadMethods);
        DownloadTaskBean bean = new DownloadTaskBean(url, packageName, icon, tuiaId);
        beanHashMap.put(url, bean);

        //****************   适配单任务逻辑，修改回多任务直接删除即可   ****************/
        if (downloadHashMap.size() >= DOWNLOAD_TASK_COUNT_LIMITATION){
            Set<String> urlList = downloadHashMap.keySet();
            for (final String s : urlList){
                if (!s.equals(url)){
                    downloadHashMap.get(s).cancel();
                    beanHashMap.remove(s);
                    downloadHashMap.remove(s);

                    String temp = FoxBaseSPUtils.getInstance().getString(s, "");
                    if (!TextUtils.isEmpty(temp)){
                        String[] tempArray = temp.split(",");
                        if (tempArray != null && tempArray.length > 1){
                            // 移除OkDownload保存的下载缓存
                            OkDownload.with().breakpointStore().remove(Integer.parseInt(tempArray[1]));

                            // 移除未完成的下载文件
                            File file = FoxBaseCommonUtils.checkFileExit(Constants.CACHE_NAME, FoxBaseEncryptUtils.encryptMD5ToString(s) + "tm.apk");
                            if (file != null){
                                file.delete();
                            }
                            // 移除SP保存的任务信息
                            FoxBaseSPUtils.getInstance().remove(s);
                        }
                    }
                }
            }
        }
        //****************   适配单任务逻辑，修改回多任务直接删除即可   ****************/

        return downloadMethods;
    }

    private void sendLimitBroadcast(String url, String packageName){
        Intent intent = new Intent();
        intent.putExtra("url", url);
        intent.putExtra("packageName", packageName);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.setAction("com.mediamain.limited");
        FoxBaseUtils.getApp().sendBroadcast(intent);
    }

    public void defaultDownload(final String url, String packageName, String icon,String tuiaId){
        if (TextUtils.isEmpty(url)){
            return;
        }

        dependencyDownloadBroadcast(url, packageName, icon, tuiaId);

        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            downloadMethods = createDownloadMethods(url, packageName, icon, tuiaId);
        }

        if (downloadHashMap.size() >= DOWNLOAD_TASK_COUNT_LIMITATION){
            beanHashMap.remove(url);
            downloadHashMap.remove(url);
            sendLimitBroadcast(url, packageName);
            return;
        }

        DownloadController.DownloadStatus status = DownloadController.getInstance().getStatus(url, packageName, downloadMethods.isCancel());
        if (status == DownloadController.DownloadStatus.UN_DOWNLOAD || status == DownloadController.DownloadStatus.STOP_DOWNLOAD){
            final DownloadMethods finalDownloadMethods = downloadMethods;
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    finalDownloadMethods.defaultDownload();
                }
            }, 1000);
        }
    }

    public void backgroundDownload(final String url, String packageName,String tuiaId){
        if (TextUtils.isEmpty(url)){
            return;
        }

        dependencyDownloadBroadcast(url, packageName, "", tuiaId);

        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            downloadMethods = createDownloadMethods(url, packageName, "", tuiaId);
        }

        if (downloadHashMap.size() >= DOWNLOAD_TASK_COUNT_LIMITATION){
            beanHashMap.remove(url);
            downloadHashMap.remove(url);
            sendLimitBroadcast(url, packageName);
            return;
        }

        DownloadController.DownloadStatus status = DownloadController.getInstance().getStatus(url, packageName, downloadMethods.isCancel());
        if (status == DownloadController.DownloadStatus.UN_DOWNLOAD || status == DownloadController.DownloadStatus.STOP_DOWNLOAD){
            final DownloadMethods finalDownloadMethods = downloadMethods;
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    finalDownloadMethods.backgroundDownload();
                }
            }, 1000);
        }
    }

    public void cancel(String url, String packageName,String tuiaId){
        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            return;
        }

        downloadMethods.cancel();
    }

    public float getProgress(String url, String packageName,String tuiaId){
        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            return 0;
        }

        return downloadMethods.getProgress();
    }

    public void installApp(String url, String packageName, String icon, int isRail, String tuiaId){
        dependencyDownloadBroadcast(url, packageName, icon, tuiaId);

        if (FoxBaseSPUtils.getInstance().containsKey(url)){
            return;
        }

        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            downloadMethods = new IntegrateH5Download(url, packageName, icon, tuiaId) {
                @Override
                public void downloadCompleted(String url) {

                }
            };
        }

        downloadMethods.installApp(isRail);
    }

    public int getDownloadStatus(String url, String packageName,String tuiaId){
        DownloadController.DownloadStatus status;

        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if ((downloadMethods == null || downloadMethods.isCancel()) && FoxBaseSPUtils.getInstance().containsKey(url)){
            status = DownloadController.getInstance().getStatus(url, packageName, true);
        }else {
            status = DownloadController.getInstance().getStatus(url, packageName, false);
        }

        switch (status){
            case UN_DOWNLOAD:
                return 1;
            case DOWNLOADING:
                return 2;
            case UN_INSTALL:
                return 3;
            case INSTALLED:
                return 4;
            case STOP_DOWNLOAD:
                return 5;
            default:
                return -1;
        }
    }

    public void setShowNotification(String url, String packageName, String icon, boolean showNotification,String tuiaId){
        DownloadMethods downloadMethods = preCheckTask(url, packageName);
        if (downloadMethods == null){
            return;
        }

        downloadMethods.setShowNotification(showNotification, icon);
    }

    public void openApp(String url, String packageName, String icon, int isRail, String tuiaId){
        dependencyDownloadBroadcast(url, packageName, icon, tuiaId);

        DownloadMethods downloadMethods = new IntegrateH5Download(url, packageName, icon, tuiaId) {
            @Override
            public void downloadCompleted(String url) {

            }
        };

        Set<String> urlList = completedDownloadMap.keySet();
        for (String s : urlList){
            if (s.equals(url)){
                completedDownloadMap.remove(s);
                break;
            }
        }

        downloadMethods.openApp(isRail);
    }

    public void webviewDestroy(){
        Set<String> urlList = downloadHashMap.keySet();
        for (final String s : urlList){
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    // 正在下载退出WebView重新显示通知栏
                    if (downloadHashMap.get(s) != null && downloadHashMap.get(s).isCancel()){
                        DownloadController.getInstance().showNotification(s,
                                downloadHashMap.get(s).getPackageName(),
                                downloadHashMap.get(s).getIcon(),
                                (int) downloadHashMap.get(s).getProgress(),
                                downloadHashMap.get(s).isCancel(),
                                downloadHashMap.get(s).getTuiaId(), true);
                    }
                    // 正在下载退出WebView继续下载静默任务
                    else if (TextUtils.isEmpty(beanHashMap.get(s).getIcon())){
                        if (downloadHashMap.get(s) != null){
                            downloadHashMap.get(s).backgroundDownload();
                        }
                    }
                    // 正在下载退出WebView继续下载普通任务
                    else {
                        if (downloadHashMap.get(s) != null){
                            downloadHashMap.get(s).defaultDownload();
                        }
                    }
                }
            }, 1000);
        }

        if (urlList != null && urlList.size() > 0){
            return;
        }

        Set<String> urlCompleteList = completedDownloadMap.keySet();
        for (final String str : urlCompleteList){
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    // 安装完成退出WebView重新显示通知栏
                    if (DownloadController.getInstance().getStatus(str, completedDownloadMap.get(str).getPackageName(),
                            completedDownloadMap.get(str).isCancel()) == DownloadController.DownloadStatus.INSTALLED){
                        DownloadController.getInstance().showNotification(str,
                                completedDownloadMap.get(str).getPackageName(),
                                completedDownloadMap.get(str).getIcon(),
                                200,
                                completedDownloadMap.get(str).isCancel(),
                                completedDownloadMap.get(str).getTuiaId(), true);
                    }
                    // 完成下载退出WebView，且属于普通下载，重新显示通知栏
                    else if (completedDownloadMap.get(str) != null && completedDownloadMap.get(str).getProgress() == 100
                            && completedDownloadMap.get(str).isShowNotification()){
                        DownloadController.getInstance().showNotification(str,
                                completedDownloadMap.get(str).getPackageName(),
                                completedDownloadMap.get(str).getIcon(),
                                (int) completedDownloadMap.get(str).getProgress(),
                                completedDownloadMap.get(str).isCancel(),
                                completedDownloadMap.get(str).getTuiaId(), true);
                    }
                }
            }, 1000);
        }
    }
}
