package com.mediamain.android.base.util;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.telephony.ServiceState;
import android.telephony.SubscriptionManager;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import com.mediamain.android.base.util.crash.FoxBaseCrashUtils;

import java.lang.reflect.Method;

public class NetworkUtil {

    public static final int NETWORK_TYPE_NR = 20;
    public static final int SDK_VERSION_Q = 29;

    /**
     * get the network type
     *
     * @param ctx Context
     * @return networktype
     */
    public static int getNetWorkType(Context ctx) {
        int networkType = TelephonyManager.NETWORK_TYPE_UNKNOWN;
        TelephonyManager tm = (TelephonyManager) ctx.getSystemService(Context.TELEPHONY_SERVICE);
        try {
            int defaultDataSubId = getSubId();
            if (defaultDataSubId == -1) {
                networkType = tm.getNetworkType();
            } else {
                try {
                    Method dataNetworkType = TelephonyManager.class
                            .getDeclaredMethod("getDataNetworkType", new Class[]{int.class});
                    dataNetworkType.setAccessible(true);
                    networkType = (int) dataNetworkType.invoke(tm, defaultDataSubId);
                } catch (Throwable t) {
                    FoxBaseCrashUtils.reportErrorData(t);
                    networkType = tm.getNetworkType();
                }
            }
        } catch (Throwable t) {
            FoxBaseCrashUtils.reportErrorData(t);
        }
        if (networkType == TelephonyManager.NETWORK_TYPE_LTE) {
            networkType = adjustNetworkType(ctx, networkType);
        }
        return networkType;
    }

    /**
     * get the 5G network type
     *
     * @param ctx Context
     * @param networkTypeFromSys this method can be call only when networkTypeFromSys = 13(LET)
     * @return correct network type
     */
    private static int adjustNetworkType(Context ctx, int networkTypeFromSys) {
        int networkType = networkTypeFromSys;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            if (Build.VERSION.SDK_INT >= SDK_VERSION_Q
                    && ctx.checkSelfPermission(Manifest.permission.READ_PHONE_STATE)
                    == PackageManager.PERMISSION_GRANTED) {
                try {
                    TelephonyManager tm = (TelephonyManager) ctx
                            .getSystemService(Context.TELEPHONY_SERVICE);
                    ServiceState ss = null;
                    int defaultDataSubId = getSubId();
                    if (defaultDataSubId == -1) {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                            ss = tm.getServiceState();
                        }
                    } else {
                        try {
                            Class<TelephonyManager> infTm = TelephonyManager.class;
                            Method method = infTm
                                    .getDeclaredMethod("getServiceStateForSubscriber",
                                            new Class[]{int.class});
                            method.setAccessible(true);
                            ss = (ServiceState) method.invoke(tm, defaultDataSubId);
                        } catch (Throwable t) {
                            FoxBaseCrashUtils.reportErrorData(t);
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                                ss = tm.getServiceState();
                            }
                        }
                    }
                    if (ss != null && isServiceStateFiveGAvailable(ss.toString())) {
                        networkType = NETWORK_TYPE_NR;
                    }
                } catch (Exception e) {
                    FoxBaseCrashUtils.reportErrorData(e);
                }
            }
        }
        return networkType;
    }

    /**
     * get data sub id
     *
     * @return subId
     */
    private static int getSubId() {
        int defaultDataSubId = -1;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            defaultDataSubId = SubscriptionManager.getDefaultDataSubscriptionId();
        }
        return defaultDataSubId;
    }

    /**
     * check the service state str is 5G
     *
     * @param ss services state str
     * @return true if is 5G
     */
    private static boolean isServiceStateFiveGAvailable(String ss) {
        boolean available = false;
        if (!TextUtils.isEmpty(ss)
                && (ss.contains("nrState=NOT_RESTRICTED")
                || ss.contains("nrState=CONNECTED"))) {
            available = true;
        }
        return available;
    }
}
