package com.mediamain.android.base.util.crash;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.text.TextUtils;

import com.mediamain.android.BuildConfig;
import com.mediamain.android.base.FoxBaseSDK;
import com.mediamain.android.base.config.FoxBaseConstants;
import com.mediamain.android.base.config.FoxBaseUrl;
import com.mediamain.android.base.okgo.OkGo;
import com.mediamain.android.base.okgo.callback.StringCallback;
import com.mediamain.android.base.okgo.model.Response;
import com.mediamain.android.base.okgo.request.PostRequest;
import com.mediamain.android.base.util.FoxBaseCommonUtils;
import com.mediamain.android.base.util.FoxBaseGsonUtil;
import com.mediamain.android.base.util.FoxBaseSPUtils;
import com.mediamain.android.base.util.cache.FoxBaseCacheUtils;

import org.json.JSONObject;

import java.lang.Thread.UncaughtExceptionHandler;
import java.text.Format;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;

/**
 * <pre>
 *     author: lzy
 *     desc  : 崩溃信息工具类
 *     time  : 20191201
 * </pre>
 */
public final class FoxBaseCrashUtils {

    private static String versionName;
    private static int versionCode;
    private static String mAppSecret = null;
    private static String mAppKey = null;

    private static String mPackageName;
    private static String mConfigData;
    private static int mDataFrom;

    @SuppressLint("SimpleDateFormat")
    private static final Format FORMAT = new SimpleDateFormat("MM-dd_HH-mm-ss");

    private static final UncaughtExceptionHandler DEFAULT_UNCAUGHT_EXCEPTION_HANDLER;
    private static final UncaughtExceptionHandler UNCAUGHT_EXCEPTION_HANDLER;
    private static OnCrashListener sOnCrashListener;

    static {
        try {
            PackageInfo pi = FoxBaseSDK.getContext().getPackageManager().getPackageInfo(FoxBaseSDK.getContext().getPackageName(), 0);
            if (pi != null) {
                versionName = pi.versionName;
                versionCode = pi.versionCode;
            }
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        DEFAULT_UNCAUGHT_EXCEPTION_HANDLER = Thread.getDefaultUncaughtExceptionHandler();

        UNCAUGHT_EXCEPTION_HANDLER = new UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(final Thread t, final Throwable e) {
                try {
                    if (e != null) {
                        String crashInfo = addCommonParams(e, 1);
                        FoxBaseCacheUtils.commitString(FoxBaseConstants.KEY_TUIA_SDK_APP_CRASH+mAppKey, FoxBaseCommonUtils.getCrashData(FoxBaseSDK.getContext(), mDataFrom, mConfigData, mAppKey, crashInfo, "2", false));
                        if (sOnCrashListener != null) {
                            sOnCrashListener.onCrash(crashInfo, e);
                        }
                        if (DEFAULT_UNCAUGHT_EXCEPTION_HANDLER != null) {
                            DEFAULT_UNCAUGHT_EXCEPTION_HANDLER.uncaughtException(t, e);
                        }
                    } else {
                        if (DEFAULT_UNCAUGHT_EXCEPTION_HANDLER != null) {
                            DEFAULT_UNCAUGHT_EXCEPTION_HANDLER.uncaughtException(t, null);
                        } else {
                            try {
                                Thread.sleep(2000);
                            } catch (Exception s) {
                            }
                            android.os.Process.killProcess(android.os.Process.myPid());
                            System.exit(1);
                        }
                        return;
                    }
                } catch (Exception e1) {
                    e1.printStackTrace();
                }
            }
        };
    }

    /**
     * Initialization.
     * <p>Must hold {@code <uses-permission android:name="android.permission.WRITE_EXTERNAL_STORAGE" />}</p>
     */
    @SuppressLint("MissingPermission")
    public static void init(Context context, String packageName, int dataFrom, String configData) {
        mPackageName = packageName;
        mDataFrom = dataFrom;
        mConfigData = configData;
        try {
            init("");
            reportCrashData("", "1", false);

            // 自定义uncaughtHandler会捕捉全局crash，以KEY_TUIA_SDK_CRASH_INFO_TYPE类型做判断：1->sdk崩溃，2->oaid崩溃，-1->app崩溃（且不属于sdk崩溃）
            String type = FoxBaseCacheUtils.readString(FoxBaseConstants.KEY_TUIA_SDK_CRASH_INFO_TYPE);
            if (TextUtils.isEmpty(type) || type.equals("-1")){
                return;
            }

            final String md = FoxBaseCacheUtils.readString(FoxBaseConstants.KEY_TUIA_SDK_APP_CRASH+mAppKey);
            if (!FoxBaseCommonUtils.isEmpty(md)) {
                reportCrashData(md, "2", false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Initialization
     * <p>Must hold {@code <uses-permission android:name="android.permission.WRITE_EXTERNAL_STORAGE" />}</p>
     *
     * @param crashDirPath The directory's path of saving crash information.
     */
    public static void init(final String crashDirPath) {
        init(crashDirPath, null);
    }

    /**
     * Initialization
     * <p>Must hold {@code <uses-permission android:name="android.permission.WRITE_EXTERNAL_STORAGE" />}</p>
     *
     * @param onCrashListener The crash listener.
     */
    @SuppressLint("MissingPermission")
    public static void init(final OnCrashListener onCrashListener) {
        init("", onCrashListener);
    }

    /**
     * Initialization
     * <p>Must hold {@code <uses-permission android:name="android.permission.WRITE_EXTERNAL_STORAGE" />}</p>
     *
     * @param crashDirPath    The directory's path of saving crash information.
     * @param onCrashListener The crash listener.
     */
    public static void init(final String crashDirPath, final OnCrashListener onCrashListener) {
        sOnCrashListener = onCrashListener;
        Thread.setDefaultUncaughtExceptionHandler(UNCAUGHT_EXCEPTION_HANDLER);
    }

    public static void reportErrorData(Throwable throwable){
        try {
            // config接口请求失败
            if (!FoxBaseSPUtils.getInstance().containsKey(FoxBaseConstants.KEY_TUIA_SDK_ERROR_REPORT_SWITCH)){
                return;
            }

            int reportSwitch = FoxBaseSPUtils.getInstance().getInt(FoxBaseConstants.KEY_TUIA_SDK_ERROR_REPORT_SWITCH, 0);
            //  开关状态
            if (reportSwitch <= 0){
                return;
            }

            String currentErrorInfo = addCommonParams(throwable, 2);
            String[] currentErrorArray = currentErrorInfo.split("\n");

            String lastErrorInfo = FoxBaseCacheUtils.readString(currentErrorArray[10]);
            // 第一次保存异常信息
            if (TextUtils.isEmpty(lastErrorInfo)){
                // 将当前异常信息更新到cache
                currentErrorInfo = System.currentTimeMillis() + "\n" + currentErrorInfo;
                FoxBaseCacheUtils.commitString(currentErrorArray[10], currentErrorInfo);
                return;
            }

            String[] lastErrorArray = lastErrorInfo.split("\n");
            try {
                long time = Long.parseLong(lastErrorArray[0]);
                if (time > 0){
                    return;
                }

                // 将异常信息头部保留的时间戳删除
                lastErrorInfo = "";
                for(int i=2; i<lastErrorArray.length; i++){
                    lastErrorInfo = lastErrorInfo + lastErrorArray[i] + "\n";
                }
                // 上报上一次异常信息
                lastErrorInfo = FoxBaseCommonUtils.getCrashData(FoxBaseSDK.getContext(), mDataFrom, mConfigData, mAppKey, lastErrorInfo, "2", true);
                reportCrashData(lastErrorInfo, "2", true);
            }catch (NumberFormatException e){

            }

            // 将当前异常信息更新到cache
            currentErrorInfo = System.currentTimeMillis() + "\n" + currentErrorInfo;
            FoxBaseCacheUtils.commitString(currentErrorArray[10], currentErrorInfo);
        }catch (Exception e){

        }
    }

    public static String getCommonParams(){
        final String time = FORMAT.format(new Date(System.currentTimeMillis()));
        return "************* Log Head ****************" +
                "\nTime Of Crash      : " + time +
                "\nDevice Manufacturer: " + Build.MANUFACTURER +
                "\nDevice Model       : " + Build.MODEL +
                "\nAndroid Version    : " + Build.VERSION.RELEASE +
                "\nAndroid SDK        : " + Build.VERSION.SDK_INT +
                "\nApp VersionName    : " + versionName +
                "\nApp VersionCode    : " + versionCode +
                "\nSDK VersionCode    : " + BuildConfig.VERSION_NAME +
                "\n************* Log Head ****************\n";
    }

    /**
     * 添加公共参数
     * @param t：跑出异常
     * @param type：异常分类
     *            1 -> CRASH
     *            2 -> try/catch
     *            3 -> 其他
     * @return
     */
    private static String addCommonParams(Throwable t, int type){
        if (t == null){
            return "";
        }

        final StringBuilder sb = new StringBuilder();
        sb.append(getCommonParams()).append(FoxBaseThrowableUtils.getFullStackTrace(t, type));
        return sb.toString();
    }

    // interface
    public interface OnCrashListener {
        void onCrash(String crashInfo, Throwable e);
    }

    /**
     * 激活崩溃日志上报
     */
    private static void reportCrashData(String md, final String dataType, final boolean isError) {
        try {
            if (FoxBaseSDK.getContext() == null) {
                return;
            }

            if (FoxBaseCommonUtils.isEmpty(mAppKey) || FoxBaseCommonUtils.isEmpty(mAppSecret)) {
                mAppKey = FoxBaseCommonUtils.getAppKey();
                mAppSecret = FoxBaseCommonUtils.getAppSecret();
            }
            if (FoxBaseCommonUtils.isEmpty(mAppKey) || FoxBaseCommonUtils.isEmpty(mAppSecret)) {
                return;
            }
            long time = System.currentTimeMillis();
            if (FoxBaseCommonUtils.isEmpty(md)){
                md = FoxBaseCommonUtils.getCrashData(FoxBaseSDK.getContext(), mDataFrom, mConfigData, mAppKey, "", dataType, false);
            }
            int nonce = (int) ((Math.random() * 9 + 1) * 100000);
            String signature = FoxBaseCommonUtils.sha1("appSecret=" + mAppSecret + "&md=" + md + "&nonce=" + nonce + "&timestamp=" + time);
            HashMap<String, Object> hashMap = new HashMap<>();
            hashMap.put("time", time);
            hashMap.put("appKey", mAppKey);
            hashMap.put("nonce", nonce);
            hashMap.put("signature", signature);
            hashMap.put("md", md);
            final String sign = FoxBaseCommonUtils.encrypt(FoxBaseGsonUtil.GsonString(hashMap), FoxBaseConstants.KEY_SECRET);
            JSONObject mJSONObject = new JSONObject();
            mJSONObject.put("sign", sign);
            hashMap.put("sign", sign);
            final PostRequest<String> stringGetRequest = OkGo.<String>post(FoxBaseUrl.BASE_SDK_REPORTCRASH);
            stringGetRequest
                    .upJson(mJSONObject.toString())
                    .execute(new StringCallback() {
                        @Override
                        public void onSuccess(Response<String> response) {
                            if (isError){
                                return;
                            }

                            try {
                                if (response != null && response.body() != null) {
                                    if (!FoxBaseCommonUtils.isEmpty(dataType) && dataType.equals("2")) {
                                        FoxBaseCacheUtils.commitString(FoxBaseConstants.KEY_TUIA_SDK_APP_CRASH+mAppKey, "");
                                    }
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }

                        @Override
                        public void onError(Response<String> response) {
                            super.onError(response);
                        }
                    });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}

