package com.mediamain.android.base.util.crash;

import android.text.TextUtils;

import com.mediamain.android.base.config.FoxBaseConstants;
import com.mediamain.android.base.util.FoxBaseSPUtils;
import com.mediamain.android.base.util.cache.FoxBaseCacheUtils;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * <pre>
 *     author: Blankj
 *     blog  : http://blankj.com
 *     time  : 2019/02/12
 *     desc  : utils about exception
 * </pre>
 */
public class FoxBaseThrowableUtils {

    private static final String LINE_SEP = System.getProperty("line.separator");

    private FoxBaseThrowableUtils() {
        throw new UnsupportedOperationException("u can't instantiate me...");
    }

    public static String getFullStackTrace(Throwable throwable, int type) {
        final List<Throwable> throwableList = new ArrayList<>();
        while (throwable != null && !throwableList.contains(throwable)) {
            throwableList.add(throwable);
            throwable = throwable.getCause();
        }
        final int size = throwableList.size();
        final List<String> frames = new ArrayList<>();
        List<String> nextTrace = getStackFrameList(throwableList.get(size - 1));
        for (int i = size; --i >= 0; ) {
            final List<String> trace = nextTrace;
            if (i != 0) {
                nextTrace = getStackFrameList(throwableList.get(i - 1));
                removeCommonFrames(trace, nextTrace);
            }
            if (i == size - 1) {
                frames.add(throwableList.get(i).toString());
            } else {
                frames.add(" Caused by: " + throwableList.get(i).toString());
            }
            frames.addAll(trace);
        }
        StringBuilder sb = new StringBuilder();
        for (final String element : frames) {
            sb.append(element).append(LINE_SEP);
        }

        indexCrashInfo(sb.toString(), type);
        return sb.toString();
    }

    private static List<String> getStackFrameList(final Throwable throwable) {
        final StringWriter sw = new StringWriter();
        final PrintWriter pw = new PrintWriter(sw, true);
        throwable.printStackTrace(pw);
        final String stackTrace = sw.toString();
        final StringTokenizer frames = new StringTokenizer(stackTrace, LINE_SEP);
        final List<String> list = new ArrayList<>();
        boolean traceStarted = false;
        while (frames.hasMoreTokens()) {
            final String token = frames.nextToken();
            // Determine if the line starts with <whitespace>at
            final int at = token.indexOf("at");
            if (at != -1 && token.substring(0, at).trim().isEmpty()) {
                traceStarted = true;
                list.add(token);
            } else if (traceStarted) {
                break;
            }
        }
        return list;
    }

    private static void removeCommonFrames(final List<String> causeFrames, final List<String> wrapperFrames) {
        int causeFrameIndex = causeFrames.size() - 1;
        int wrapperFrameIndex = wrapperFrames.size() - 1;
        while (causeFrameIndex >= 0 && wrapperFrameIndex >= 0) {
            // Remove the frame from the cause trace if it is the same
            // as in the wrapper trace
            final String causeFrame = causeFrames.get(causeFrameIndex);
            final String wrapperFrame = wrapperFrames.get(wrapperFrameIndex);
            if (causeFrame.equals(wrapperFrame)) {
                causeFrames.remove(causeFrameIndex);
            }
            causeFrameIndex--;
            wrapperFrameIndex--;
        }
    }

    private static void indexCrashInfo(String reportInfo, int type){
        if (TextUtils.isEmpty(reportInfo)){
            return;
        }

        switch (type){
            case 1:
                if (reportInfo.contains(FoxBaseConstants.KEY_TUIA_SDK_PACKAGE_NAME)){
                    FoxBaseCacheUtils.commitString(FoxBaseConstants.KEY_TUIA_SDK_CRASH_INFO_TYPE, "1");
                }else if (reportInfo.contains(FoxBaseConstants.KEY_TUIA_SDK_OAID_PACKAGE_NAME)){
                    FoxBaseCacheUtils.commitString(FoxBaseConstants.KEY_TUIA_SDK_CRASH_INFO_TYPE, "2");
                }else {
                    FoxBaseCacheUtils.commitString(FoxBaseConstants.KEY_TUIA_SDK_CRASH_INFO_TYPE, "-1");
                }
                break;
            case 2:
                String[] strArray = reportInfo.split("\n");
                updateErrorInfo(strArray);
                break;
            default:
                break;
        }
    }

    private static void updateErrorInfo(String[] reportInfoArray) {
        StringBuilder reportInfo = new StringBuilder();
        String errorValue = FoxBaseCacheUtils.readString(reportInfoArray[0]);
        // 第一次存储异常信息
        if (TextUtils.isEmpty(errorValue)){

            reportInfo.append(System.currentTimeMillis()).append(LINE_SEP);
            reportInfo.append(FoxBaseCrashUtils.getCommonParams());
            for (String s : reportInfoArray){
                reportInfo.append(s).append(LINE_SEP);
            }
            FoxBaseCacheUtils.commitString(reportInfoArray[0], reportInfo.toString());
        } else {
            int reportInterval = FoxBaseSPUtils.getInstance().getInt(FoxBaseConstants.KEY_TUIA_SDK_ERROR_REPORT_TIME_INTERVAL, -1);
            String[] lastErrorInfoArray = errorValue.split("\n");
            try {
                long time = Long.parseLong(lastErrorInfoArray[0]);
                // 上一次的异常信息超过设定周期，可以上报
                if (time + reportInterval * 60 * 1000 < System.currentTimeMillis()){
                    // 将cache里异常数据中时间设为-1，表示可以上报
                    reportInfo.append(-1).append(LINE_SEP);
                    for (String s : lastErrorInfoArray) {
                        reportInfo.append(s).append(LINE_SEP);
                    }

                    // 更新上一次异常在cache的时间戳
                    FoxBaseCacheUtils.commitString(lastErrorInfoArray[11], reportInfo.toString());
                }
            }catch (NumberFormatException e){

            }
        }
    }
}
