package com.mediamain.android.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.support.annotation.DrawableRes;
import android.support.v4.util.ArrayMap;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.RelativeLayout;

import com.mediamain.android.R;
import com.mediamain.android.base.FoxBaseSDK;
import com.mediamain.android.base.config.Constants;
import com.mediamain.android.base.config.FoxSDKError;
import com.mediamain.android.base.config.FoxSDKType;
import com.mediamain.android.base.util.FoxBaseAnimationUtil;
import com.mediamain.android.base.util.FoxBaseCommonUtils;
import com.mediamain.android.base.util.FoxBaseDensityUtils;
import com.mediamain.android.base.util.FoxBaseLogUtils;
import com.mediamain.android.base.util.FoxBaseSPUtils;
import com.mediamain.android.base.util.crash.FoxBaseCrashUtils;
import com.mediamain.android.view.base.AdBase;
import com.mediamain.android.view.base.FoxSize;
import com.mediamain.android.view.bean.FoxResponseBean;
import com.mediamain.android.view.bean.MessageData;
import com.mediamain.android.view.imageloader.FoxImageView;
import com.mediamain.android.view.interfaces.FoxImageLoaderCalback;
import com.mediamain.android.view.interfaces.ServingCallback;
import com.mediamain.android.view.util.FoxListenerManager;
import com.mediamain.android.view.util.FoxListenerObserver;
import com.mediamain.android.view.util.FoxStringUtil;
import com.mediamain.android.view.util.FoxViewCommonUtils;
import com.mediamain.android.view.util.FoxViewMaidianUtils;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * @Author: Zhougaofeng
 * Email: 365315018@qq.com
 * Time: 2016/10/24
 * Description: 横幅/Banner
 */
public class FoxStreamerView extends AdBase implements View.OnClickListener, FoxListenerObserver {

    private static final String TAG = "FoxStreamerView";

    private FoxImageView mImage;
    private ImageView mCloseView;
    private ImageView mAdView;
    private FoxSize mFoxSize;
    private boolean is_clicked = false;
    private Context mContext;
    private int mAdslotId;
    private String mUserId;
    private FoxResponseBean mTmResponse;
    private FoxResponseBean.DataBean mData;
    private String mkey;
    private WeakReference<FoxActivity> activityWeakReference;

    public FoxStreamerView(Context context, FoxSize foxSize) {
        super(context);
        mContext = context.getApplicationContext();
        mFoxSize = foxSize;
        initView(context);
    }

    public FoxStreamerView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public FoxStreamerView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mContext = context.getApplicationContext();
        initAttributes(context, attrs, defStyleAttr);
        initView(context);
    }

    /**
     * 信息回传
     */
    public void sendMessage(int type, String data) {
        try {
            if (activityWeakReference != null) {
                ((FoxActivity) (activityWeakReference.get())).sendMessage(type, data);
            }
        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    /**
     * onMeasure是measure方法引起的回调,而measure方法是父VIew在测量子View会调用子的View的measure方法
     * 所以widthMeasureSpec(宽度测量规则)和heightMeasureSpec(高度测量规则)是父VIew在调用子View的measure方法时计算好的
     * MeasureSpec： 测量规则类，由size和mode2个因素组成:
     * size: 就是指定的大小值
     * mode: MeasureSpec.AT_MOST : 对应的是warp_content;
     * MeasureSpec.EXACTLY : 对应的是具体的dp值，match_parent
     * MeasureSpec.UNSPECIFIED: 未定义的，一般用adapter的view的测量中
     */
    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = MeasureSpec.getSize(widthMeasureSpec);
        float height = width * mFoxSize.getHeight() / mFoxSize.getWidth();
        heightMeasureSpec = MeasureSpec.makeMeasureSpec((int) height, MeasureSpec.EXACTLY);
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
    }

    public void loadAd(int adslot_id) {
        mAdslotId = adslot_id;
        loadAdRequest(adslot_id, "");
    }

    public void loadAd(int adslot_id, String userId) {
        loadAdRequest(adslot_id, userId);
    }

    public void destroy() {
        try {
            FoxBaseLogUtils.d(TAG + "——>destroy");
            FoxListenerManager.getInstance().unregistrationObserver(mkey, this);
            if (mImage != null) {
                FoxBaseAnimationUtil.clearAnimation(mImage);
                mImage.stopCurrentFuture(true);
                mImage = null;
            }
            removeAllViews();
        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    @Override
    public void onClick(View v) {
        try {
            if (mData != null && getVisibility() == View.VISIBLE) {
                if (mFoxListener != null) {
                    mFoxListener.onAdClick();
                }
                if (!FoxBaseCommonUtils.isEmpty(mkey)) {
                    FoxBaseSPUtils.getInstance().setString(mkey, mAdslotId + "");
                }
                FoxBaseLogUtils.d(TAG + "——>onAdClick" + mData.getActivityUrl());
                FoxActivity.starActivity(getContext(), mkey, FoxStringUtil.appandUrl(mData.getActivityUrl()), FoxSDKType.FOX_STREAMER_VIEW.getCode());
                if (!is_clicked) {
                    doResponse(1);
                    is_clicked = true;
                }
            }
        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    @Override
    public void update(String event, Object msg) {
        try {
            if (FoxBaseCommonUtils.isEmpty(event)) {
                return;
            }
            if (event.contains(Constants.KEY_AD_CLOSE)) {
                if (mFoxListener != null && msg instanceof String) {
                    mFoxListener.onAdActivityClose((String) msg);
                }
            }
            if (event.contains(Constants.KEY_AD_MESSAGE)) {
                if (mFoxListener != null && msg instanceof MessageData) {
                    mFoxListener.onAdMessage((MessageData) msg);
                }
                return;
            }
            if (event.contains(Constants.KEY_ACTIVITY_EXAMPLE)) {
                activityWeakReference = new WeakReference<>((FoxActivity) msg);
                return;
            }


        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    private boolean useImage = true;
    private Bitmap customBitmap;
    private String customPath;

    public void loadCustomImage(int slotId, String userId, String path) {
        useImage = false;
        if (FoxBaseCommonUtils.isEmpty(path)) {
            return;
        }
        if (path.startsWith("http") || path.startsWith("https")) {
            customPath = path;
        } else {
            customBitmap = BitmapFactory.decodeFile(path);
        }
        loadAdRequest(slotId, userId);
    }

    public void loadCustomImage(int slotId, String userId, Bitmap bitmap) {
        useImage = false;
        customBitmap = bitmap;
        loadAdRequest(slotId, userId);
    }

    public void loadCustomImage(int slotId, String userId, @DrawableRes int resId) {
        Bitmap bitmap = BitmapFactory.decodeResource(FoxBaseSDK.getContext().getResources(), resId);
        loadCustomImage(slotId, userId, bitmap);
    }

//-------------------------------------------------Private--------------------------------------------------------------

    /**
     * 请求GG
     *
     * @param adslot_id
     * @param userId
     */
    private void loadAdRequest(int adslot_id, String userId) {
        try {
            mAdslotId = adslot_id;
            mUserId = userId;
            FoxBaseLogUtils.d(TAG + "——>loadAdRequest->start");
            setVisibility(View.GONE);

            Map<String, String> params = new HashMap<>();
            params.put("slotAccessType", "0");
            params.put("slotSceneType", Integer.toString(FoxSDKType.FOX_STREAMER_VIEW.getCode())); //Banner/横幅接入类型

            loadAdRequest(adslot_id, userId, false, params, new ServingCallback() {
                @Override
                public void onServingDataCorrect(FoxResponseBean.DataBean data) {
                    super.onServingDataCorrect(data);
                    servingSuccessResponse(data);
                }

                @Override
                public void onServingDataError(int code, String message) {
                    super.onServingDataError(code, message);
                }
            });

        } catch (Exception ignored) {
            FoxBaseCrashUtils.reportErrorData(ignored);
        }
    }

    /**
     * Serving请求成功
     *
     * @param data
     */
    private void servingSuccessResponse(FoxResponseBean.DataBean data) {
        mData = data;
        is_clicked = false;
        if (!FoxBaseCommonUtils.isEmpty(mUserId)) {
            if (data.getActivityUrl().contains("?")) {
                mData.setActivityUrl(data.getActivityUrl() + "&userId=" + mUserId);
            } else {
                mData.setActivityUrl(data.getActivityUrl() + "?userId=" + mUserId);
            }
        }
        if (mImage != null) {
            setVisibility(View.VISIBLE);
            mImage.setVisibility(VISIBLE);
            if (data.getSckId() == 0) {
                if (customBitmap != null) {
                    mImage.setImageSrc(customBitmap);
                } else if (!FoxBaseCommonUtils.isEmpty(customPath)) {
                    mImage.setImageUrl(customPath, R.drawable.default_image_background);
                } else {
                    if (mFoxListener != null) {
                        mFoxListener.onFailedToReceiveAd(FoxSDKError.INVALID_CUSTOM_IMAGE_URI.getCode(), FoxSDKError.INVALID_CUSTOM_IMAGE_URI.getMessage());
                    }
                }
            } else {
                mImage.setImageUrl(FoxStringUtil.appandUrl(data.getImageUrl()), R.drawable.default_image_background);
            }
            FoxViewCommonUtils.startPreLoad(String.valueOf(mAdslotId), mData.getActivityUrl(), FoxSDKType.FOX_STREAMER_VIEW.getCode());
        }
        if (mCloseView != null) {
            mCloseView.setVisibility(data.isVisibleOfCloseButton() ? View.VISIBLE : View.GONE);
        }
        if (mAdView != null) {
            mAdView.setVisibility(data.isVisibleOfIcon() ? View.VISIBLE : View.GONE);
        }
    }

    private void initAttributes(Context context, AttributeSet attrs, int defStyle) {
        try {
            TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.FoxStreamerView, defStyle, 0);
            if (typedArray != null) {
                int size = typedArray.getInt(R.styleable.FoxStreamerView_fox_size, 1);
                switch (size) {
                    case 0:
                        mFoxSize = FoxSize.TMBr;
                        break;
                    case 1:
                        mFoxSize = FoxSize.LANDER_TMBr;
                        break;
                    default:
                        mFoxSize = FoxSize.TMBr;
                        break;
                }
                final int width = typedArray.getInt(R.styleable.FoxStreamerView_fox_width, 0);
                final int height = typedArray.getInt(R.styleable.FoxStreamerView_fox_height, 0);
                if (width > 0 && height > 0) {
                    mFoxSize = new FoxSize(width, height, width + "x" + height + "_mb");
                }
                typedArray.recycle();
            }
        } catch (Exception e) {
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    private void initView(Context context) {
        if (getChildCount() == 0) {
            mkey = UUID.randomUUID().toString();
            FoxListenerManager.getInstance().registrationObserver(mkey, this);
            //创建里面的控件
            mCloseView = new ImageView(context);
            mAdView = new ImageView(context);
            mImage = new FoxImageView(context);
            //设置读取出来的控件属性
            mImage.setScaleType(ImageView.ScaleType.FIT_XY);

            DisplayMetrics metric = new DisplayMetrics();
            WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            wm.getDefaultDisplay().getMetrics(metric);
            int height = metric.widthPixels * mFoxSize.getHeight() / mFoxSize.getWidth();
            setLayoutParams(new LayoutParams(metric.widthPixels, height));

            LayoutParams lp1 = new LayoutParams(LayoutParams.MATCH_PARENT, height);
            addView(mImage, lp1);

            LayoutParams lp2 = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
            lp2.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            lp2.addRule(RelativeLayout.ALIGN_PARENT_RIGHT, RelativeLayout.TRUE);
            addView(mCloseView, lp2);
            int closePadding = FoxBaseDensityUtils.dp2px(mContext, 5);
            mCloseView.setPadding(closePadding, closePadding, closePadding, closePadding);
            mCloseView.setImageResource(R.drawable.fox_close);

            LayoutParams lp3 = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
            lp3.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            lp3.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            int adMargin = FoxBaseDensityUtils.dp2px(mContext, 5);
            lp3.bottomMargin = adMargin;
            lp3.leftMargin = adMargin;
            addView(mAdView, lp3);
            mAdView.setImageResource(R.drawable.fox_ad_icon);

            mImage.setOnClickListener(this);
            mCloseView.setOnClickListener(new OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mFoxListener != null) {
                        mFoxListener.onCloseClick();
                        FoxBaseLogUtils.d(TAG + "——>onCloseClick");
                    }
                    setVisibility(View.GONE);
                }
            });

            mImage.setLoadCallback(new FoxImageLoaderCalback() {
                @Override
                public void finish() {
                    if (mImage != null) {
                        setVisibility(View.VISIBLE);
                    }
                    if (mData != null) {
                        doResponse(0);
                        if (mFoxListener != null) {
                            mFoxListener.onReceiveAd();
                            mFoxListener.onAdExposure();
                            FoxBaseLogUtils.d(TAG + "——>onReceiveAd");
                            FoxBaseLogUtils.d(TAG + "——>onAdExposure");
                        }
                    }
                }

                @Override
                public void failed() {
                    if (mFoxListener != null) {
                        mFoxListener.onLoadFailed();
                        FoxBaseLogUtils.d(TAG + "——>onLoadFailed");
                    }
                }
            });

        }

        setVisibility(View.GONE);
    }

    /**
     * 上报接口
     *
     * @param logType 0 曝光
     *                1 点击
     */
    private void doResponse(int logType) {
        ArrayMap<String, String> params = new ArrayMap<>();
        params.put("slotAccessType", "0");
//        String size = mData.getSize();
//        if (logType == 0 && !FoxBaseCommonUtils.isEmpty(size) && customBitmap != null && mData.getSckId() == 0) {
//            // 本地素材尺寸
//            int localWidth = customBitmap.getWidth();
//            int localHeight = customBitmap.getHeight();
//            params.put("sck_spec_custom", localWidth + "*" + localHeight);
//            params.put("sck_spec_tuia", size);
//        }
        FoxViewMaidianUtils.doResponse(mAdslotId, logType, mData, params);
    }
}
