package com.mediamain.android.view.imageloader;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Movie;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Handler;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;

import com.mediamain.android.base.util.FoxBaseCommonUtils;
import com.mediamain.android.R;
import com.mediamain.android.base.util.crash.FoxBaseCrashUtils;
import com.mediamain.android.view.interfaces.FoxImageLoaderCalback;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.Future;


@SuppressLint("AppCompatCustomView")
public class FoxImageView extends ImageView implements FoxImageWorker.LoadImgCallable {

    private FoxImageLoaderCalback imageLoaderCalback;
    private static final int DEFAULT_MOVIE_VIEW_DURATION = 1000;
    private int mMovieResourceId;
    private Movie movie;
    private long mMovieStart;
    private int mCurrentAnimationTime;
    private float mLeft;
    private float mTop;
    private float mScale;
    private int mMeasuredMovieWidth;
    private int mMeasuredMovieHeight;
    private volatile boolean mPaused;
    private boolean mVisible = true;

    private String imageUrl;
    private Future<Bitmap> future;
    private Handler handler = new Handler();
    int defaultImageResourceId;//默认图片资源id
    Runnable r;
    private boolean isGif = false;

    public FoxImageView(Context context) {
        this(context, null);
    }

    public FoxImageView(Context context, AttributeSet attrs) {
        this(context, attrs, R.styleable.FoxImageView_gif);
    }

    public FoxImageView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        setViewAttributes(context, attrs, defStyle);
    }

    @SuppressLint("NewApi")
    private void setViewAttributes(Context context, AttributeSet attrs, int defStyle) {
        /**
         * Starting from HONEYCOMB(Api Level:11) have to turn off HW acceleration to draw
         * Movie on Canvas.
         */

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
            setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        }
        final TypedArray array = context.obtainStyledAttributes(attrs,
                R.styleable.FoxImageView, defStyle, R.style.Widget_GifView);
        //-1 is default value
        mMovieResourceId = array.getResourceId(R.styleable.FoxImageView_gif, -1);
        mPaused = array.getBoolean(R.styleable.FoxImageView_paused, false);
        array.recycle();
        if (mMovieResourceId != -1) {
            movie = Movie.decodeStream(getResources().openRawResource(mMovieResourceId));
        }
    }

    public void setGifResource(int movieResourceId) {
        isGif = true;
        this.mMovieResourceId = movieResourceId;
        movie = Movie.decodeStream(getResources().openRawResource(mMovieResourceId));
        requestLayout();
    }

    public void setImageUrl(String url, int defaultImageResourceId) {
        if(FoxBaseCommonUtils.isEmpty(url)){
            return;
        }
        isGif = url.endsWith(".gif");
        if(isGif){
            GifTask gifTask = new GifTask();
            gifTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, url);
        }else {
            this.defaultImageResourceId = defaultImageResourceId;
            /*
             * 可以通过setImageUrl(nll)来停止该图片的下载
             */

            if (TextUtils.isEmpty(url)) {
                imageUrl = null;
                stopCurrentFuture(false);
                return;
            } else {
                imageUrl = url;
                stopCurrentFuture(false);//缓存中已经存在，则尽可能关掉正在运行的线程
                try {
                    if (FoxImageWorker.getInstance().isBitmapExist(url)) {
                        setImageBitmap(FoxImageWorker.getInstance().getBitmapFromMemCache(imageUrl));
                        if (imageLoaderCalback != null) {
                            imageLoaderCalback.finish();
                        }
                    } else {
                        future = FoxImageWorker.getInstance().getBitmapFromUrl(getContext(), 0, imageUrl, this);
                    }
                } catch (Exception e) {
                    FoxBaseCrashUtils.reportErrorData(e);
                    return;
                }
            }
        }
    }

    public void setImageSrc(Bitmap bitmap) {
        if (bitmap != null) {
            setImageBitmap(bitmap);
            if (imageLoaderCalback != null) {
                imageLoaderCalback.finish();
            }
        } else {
            if (imageLoaderCalback != null) {
                imageLoaderCalback.failed();
            }
        }
    }

    public void setLoadCallback(FoxImageLoaderCalback callback) {
        imageLoaderCalback = callback;
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        if(!isGif){
            super.onMeasure(widthMeasureSpec,heightMeasureSpec);
            return;
        }

        if (movie != null) {
            int movieWidth = movie.width();
            int movieHeight = movie.height();
            int maximumWidth = MeasureSpec.getSize(widthMeasureSpec);
            float scaleW = (float) movieWidth / (float) maximumWidth;
            mScale = 1f / scaleW;
            mMeasuredMovieWidth = maximumWidth;
            mMeasuredMovieHeight = (int) (movieHeight * mScale);
            setMeasuredDimension(mMeasuredMovieWidth, mMeasuredMovieHeight);
        } else {
            setMeasuredDimension(getSuggestedMinimumWidth(),getSuggestedMinimumHeight());
        }
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        if(!isGif) {
            super.onLayout(changed, l, t, r, b);
            return;
        }
        /*
		 * Calculate mLeft / mTop for drawing in center
		 */
        if (movie != null) {
            mLeft = (getWidth() - mMeasuredMovieWidth) / 2f;
            mTop = (getHeight() - mMeasuredMovieHeight) / 2f;
            mVisible = getVisibility() == View.VISIBLE;
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if(!isGif) {
            super.onDraw(canvas);
            return;
        }
        if (movie != null) {
            if (!mPaused) {
                updateAnimationTime();
                drawMovieFrame(canvas);
                invalidateView();
            } else {
                drawMovieFrame(canvas);
            }
        } else {
            super.onDraw(canvas);
        }
    }

    /**
     * Invalidates view only if it is mVisible.
     * <br>
     * {@link #postInvalidateOnAnimation()} is used for Jelly Bean and higher.
     */
    @SuppressLint("NewApi")
    private void invalidateView() {
        if (mVisible && isGif) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                postInvalidateOnAnimation();
            } else {
                invalidate();
            }
        }
    }

    @SuppressLint("NewApi")
    @Override
    public void onScreenStateChanged(int screenState) {
        super.onScreenStateChanged(screenState);
        if (isGif) {
            mVisible = screenState == SCREEN_STATE_ON;
            invalidateView();
        }
    }

    @SuppressLint("NewApi")
    @Override
    protected void onVisibilityChanged(View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        if (isGif) {
            mVisible = visibility == View.VISIBLE;
            invalidateView();
        }
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);
        if (isGif) {
            mVisible = visibility == View.VISIBLE;
            invalidateView();
        }
    }

    @Override
    public void setImage(final Bitmap bitmap, String url) {
        if (isGif) {
            if (bitmap != null && !bitmap.isRecycled()) {
                if (imageLoaderCalback != null)
                    imageLoaderCalback.finish();
            }
        }else {
            if (url.equals(imageUrl) && bitmap != null && !bitmap.isRecycled()) {
                handler.post(r = new Runnable() {
                    @Override
                    public void run() {
                        setImageBitmap(bitmap);
                        if (imageLoaderCalback != null)
                            imageLoaderCalback.finish();
                    }
                });
            }
            future = null;
        }
    }

    @Override
    public void loadFaild() {
        if (imageLoaderCalback != null)
            imageLoaderCalback.failed();
    }


    public class GifTask extends AsyncTask<String, Void, Object> {
        @Override
        protected Object doInBackground(String... strings) {
            InputStream input = null;
            OutputStream output = null;
            HttpURLConnection connection = null;
            try {
                URL url = new URL(strings[0]);
                connection = (HttpURLConnection) url.openConnection();
                connection.connect();
                // 避免因为接收到非HTTP 200 OK状态，而导致只或者错误代码，而不是要下载的文件
                if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
                    return "Server returned HTTP " + connection.getResponseCode()
                            + " " + connection.getResponseMessage();
                }
                // 这对显示下载百分比有帮助  当服务器没有返回文件的大小时，数字可能为-1
                int fileLength = connection.getContentLength();
                // 下载文件
                input = connection.getInputStream();
                byte[] array = streamToBytes(input);
                if (output != null)
                    output.close();
                if (input != null)
                    input.close();
                return array;
            } catch (Exception e) {
                FoxBaseCrashUtils.reportErrorData(e);
                return e.toString();
            } finally {
                if (connection != null)
                    connection.disconnect();
            }
        }

        @Override
        protected void onPostExecute(Object o) {
            if (o instanceof byte[]) {
                if (imageLoaderCalback != null) {
                    imageLoaderCalback.finish();
                }
                byte[] bytes = (byte[]) o;
                invalidateView();
                movie = Movie.decodeByteArray(bytes, 0, bytes.length);
                requestLayout();
            }
        }
    }
    /*--------------------------------------------------Private---------------------------------------------------------*/

    /**
     * Calculate current animation time
     */
    private void updateAnimationTime() {
        long now = android.os.SystemClock.uptimeMillis();
        if (mMovieStart == 0) {
            mMovieStart = now;
        }
        int dur = movie.duration();
        if (dur == 0) {
            dur = DEFAULT_MOVIE_VIEW_DURATION;
        }
        mCurrentAnimationTime = (int) ((now - mMovieStart) % dur);
    }

    /**
     * Draw current GIF frame
     */
    private void drawMovieFrame(Canvas canvas) {
        try {
            movie.setTime(mCurrentAnimationTime);
            canvas.save();
            canvas.scale(mScale, mScale);
//        canvas.scale(1f/scaleW, 1f/scaleH);
            movie.draw(canvas, mLeft / mScale, mTop / mScale);
            canvas.restore();
        }catch (Exception e){
            FoxBaseCrashUtils.reportErrorData(e);
            e.printStackTrace();
        }
    }

    private static byte[] streamToBytes(InputStream is) {
        ByteArrayOutputStream os = new ByteArrayOutputStream(1024);
        byte[] buffer = new byte[1024];
        int len;
        try {
            while ((len = is.read(buffer)) >= 0) {
                os.write(buffer, 0, len);
            }
        } catch (java.io.IOException e) {
            FoxBaseCrashUtils.reportErrorData(e);
        }
        return os.toByteArray();
    }

    /*--------------------------------------------------Image---------------------------------------------------------*/

    /**
     * 停止当前图片加载线程
     *
     * @param revomel 是否清空回调
     */
    public void stopCurrentFuture(boolean revomel) {
        /*
         * 停掉当前handler任务
         */
        if (r != null) {
            handler.removeCallbacks(r);
            r = null;
        }
        if (future != null) {
            try {
                future.cancel(true);
            } catch (Exception e) {
                FoxBaseCrashUtils.reportErrorData(e);
            }
        }
        if (revomel)
            imageLoaderCalback = null;
    }
}
