package com.youqian.uid.plugin;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Predicate;
import com.youqian.uid.annotation.Redundant;
import com.youqian.uid.annotation.RedundantId;
import com.youqian.uid.config.MqConfig;
import com.youqian.uid.event.RedundantEvent;
import com.youqian.uid.mq.product.MqProductComponent;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Signature;
import org.reflections.ReflectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * 冗余代码拦截器
 * Created by xia
 * time 2020/3/31.
 */
@Intercepts({
        @Signature(type = Executor.class, method = "update", args = {
                MappedStatement.class, Object.class
        }),
})
@Component
@Slf4j
public class RedundantInterceptor extends AbstractInterceptor {

    @Autowired
    private MqProductComponent mqProductComponent;

    @Autowired
    private MqConfig mqConfig;

    private Map<Class, List<RedundantEvent>> handlerMap = new ConcurrentHashMap<>();

    private String suffix = "_entity";

    private String filter_suffix = "redundant";

    @Override
    boolean check(MappedStatement mappedStatement,Object entity) {
        if(mappedStatement.getId().endsWith(filter_suffix)){
            return false;
        }
        if("UPDATE".equals(mappedStatement.getSqlCommandType().name())){
            Redundant redundantClass = entity.getClass().getAnnotation(Redundant.class);
            if(redundantClass == null){
                return false;
            }
            return true;
        }
        return false;
    }

    @Override
    public void process(Object object) throws Throwable {
        Class         handlerKey  = object.getClass();
        List<RedundantEvent> handlerList = new ArrayList<>();

        //优化点: 只有更新有注解的字段时才发送消息 否则不做其他处理
        synchronized (this) {
            Set<Field> allFields = ReflectionUtils.getAllFields(object.getClass());
            //主表的key-value
            Map resource = new HashMap(20);
            for (Field field : allFields) {
                field.setAccessible(true);
                Object value = field.get(object);
                resource.put(field.getName(),value);
            }

            //目标类查找对应的字段
            Redundant redundantClass = (Redundant) handlerKey.getAnnotation(Redundant.class);
            Class[] targetClasses = redundantClass.name();
            for(Class targetClass:targetClasses){
                Set<Field> targetFields = ReflectionUtils.getAllFields(
                        targetClass,
                        (Predicate<Field>) input -> input != null && input.getAnnotation(RedundantId.class) != null
                                && input.getAnnotation(RedundantId.class).resource().getSimpleName().equals(handlerKey.getSimpleName())
                );
                if(targetFields.size() == 0){
                    continue;
                }
                //获取表名
                String tableName = getTransformFieldName(targetClass.getSimpleName());
                if(tableName.endsWith(suffix)){
                    tableName = tableName.substring(1,tableName.length()-suffix.length());
                }
                RedundantEvent.KeyValue primaryField = null;
                RedundantEvent.KeyValue redundantField= null;
                for(Field targetField:targetFields){
                    RedundantId redundantId = targetField.getAnnotation(RedundantId.class);
                    Object value = resource.get(redundantId.name());
                    if(value != null){
                        //冗余的字段
                        redundantField = new RedundantEvent.KeyValue();
                        redundantField.setKey(targetField.getName());
                        redundantField.setValue(value);

                        Object rely = resource.get(redundantId.rely());
                        if(rely != null){
                            primaryField = new RedundantEvent.KeyValue();
                            primaryField.setKey(redundantId.rely());
                            primaryField.setValue(rely);
                        }
                    }
                    if(primaryField != null && redundantField != null){
                        handlerList.add(new RedundantEvent(tableName,primaryField,redundantField));
                    }
                }
            }
        }
        //1、发送消息mq
        if(handlerList.size()>0){
            Boolean send = mqProductComponent.send(getSmsHandle(handlerList), mqConfig.getYouqianRedundantTopic(), mqConfig.getYouqianRedundantTag());
            if(!send){
                log.warn("冗余更新消息发送失败:[{}]",object);
            }
        }
    }

    private static String getSmsHandle(List<RedundantEvent> events){
        JSONObject request = new JSONObject();
        request.put("request_id", System.currentTimeMillis());
        JSONArray d = new JSONArray();
        for(RedundantEvent event:events){
            d.add(event);
        }
        request.put("data",d);
        return request.toJSONString();
    }

    /**
     * 转换风格 驼峰转下划线
     * @param fieldName 属性名称
     * @return
     */
    private static String getTransformFieldName(String fieldName) {
        String compile = "[A-Z]";
        Pattern humpPattern = Pattern.compile(compile);
        Matcher matcher = humpPattern.matcher(fieldName);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }
}
