package cn.codeboxes.core.threadpool;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * created by liugq in 2024/09/13
 **/
@Slf4j
@Configuration
@EnableConfigurationProperties(ExecutorServiceProperties.class)
public class ExecutorServiceConfig {

    @Bean
    @ConditionalOnMissingBean
    public ExecutorService executorService(ExecutorServiceProperties properties) {

        int corePoolSize = properties.getCorePoolSize();
        int maximumPoolSize = properties.getMaximumPoolSize();
        long keepAliveTime = properties.getKeepAliveTime();
        int queueSize = properties.getQueueSize();

        NamedThreadFactory threadFactory = new NamedThreadFactory("credits-executor");

        ThreadPoolExecutor executor = new ThreadPoolExecutor(
                corePoolSize,
                maximumPoolSize,
                keepAliveTime,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(queueSize), threadFactory); // 工作队列


        log.info("start thread pool coreSize:{} maxSize:{} queueSize:{}", corePoolSize, maximumPoolSize, queueSize);
        executor.setRejectedExecutionHandler(
                new ThreadPoolExecutor.AbortPolicy()
                );

        // Add lifecycle methods to ensure proper shutdown
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            if (!executor.isShutdown()) {
                executor.shutdown();
                try {
                    if (!executor.awaitTermination(10, TimeUnit.SECONDS)) {
                        executor.shutdownNow();
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    executor.shutdownNow();
                }
            }
        }));

        return executor;
    }


}

