package io.easy.cache.autoconfigure;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.ITopic;
import com.hazelcast.core.Message;
import com.hazelcast.core.MessageListener;
import io.easy.cache.anno.CacheConsts;
import io.easy.cache.event.CacheBroadcastConsumeEventBus;
import io.easy.cache.event.CacheBroadcastMsg;
import io.easy.cache.event.CacheBroadcastPublishEventBus;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Configuration;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-11-04 14:30
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@ConditionalOnClass(HazelcastInstance.class)
public class HazelcastConfiguration implements ApplicationContextAware, InitializingBean {

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() {

        HazelcastInstance hazelcastInstance = applicationContext.getBean(HazelcastInstance.class);
        if (hazelcastInstance == null) {
            log.warn("没有发现HazelcastInstance的实例，集群内缓存不会同步更新");
            return;
        }

        ITopic<CacheBroadcastMsg> topic = hazelcastInstance.getTopic(CacheConsts.DEFAULT_BROADCAST_TOPIC);
        topic.addMessageListener(new MessageListenerImpl());

        CacheBroadcastPublishEventBus.register(cacheBroadcastMsg -> {
             topic.publish(cacheBroadcastMsg);
             return Boolean.TRUE;
        }, applicationContext);
        CacheBroadcastConsumeEventBus.register(applicationContext);
    }

    private static class MessageListenerImpl implements MessageListener<CacheBroadcastMsg> {

        @Override
        public void onMessage(Message<CacheBroadcastMsg> message) {
            CacheBroadcastMsg cacheBroadcastMsg = message.getMessageObject();
            CacheBroadcastConsumeEventBus.post(cacheBroadcastMsg);
        }
    }

}
