package cn.com.duiba.activity.center.api.remoteservice.activity;

import java.util.List;

import cn.com.duiba.activity.center.api.dto.activity.ReservationProductOrderDto;
import cn.com.duiba.activity.center.api.dto.activity.ReservationProductOrderPageQueryParam;
import cn.com.duiba.activity.center.api.tool.Page;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;


/**
 * 预约商品订单表RemoteService接口
 * 
 * @author cgq
 * @date 2025-01-19
 */
@AdvancedFeignClient
public interface RemoteReservationProductOrderService {
    
    /**
     * 插入订单记录
     * 
     * @param dto 订单DTO
     * @return 影响行数
     * @throws BizException 业务异常
     */
    int insert(ReservationProductOrderDto dto) throws BizException;
    
    /**
     * 批量插入订单记录
     * 
     * @param dtoList 订单DTO列表
     * @return 影响行数
     * @throws BizException 业务异常
     */
    int batchInsert(List<ReservationProductOrderDto> dtoList) throws BizException;
    
    /**
     * 根据ID更新订单记录
     * 
     * @param dto 订单DTO
     * @return 影响行数
     * @throws BizException 业务异常
     */
    int updateById(ReservationProductOrderDto dto) throws BizException;
    
    /**
     * 根据ID查询订单记录
     * 
     * @param id 主键ID
     * @return 订单DTO
     */
    ReservationProductOrderDto findById(Long id);
    
    /**
     * 根据订单号查询订单记录
     * 
     * @param orderNo 订单号
     * @return 订单DTO
     */
    ReservationProductOrderDto findByOrderNo(String orderNo);
    
    /**
     * 根据批量订单号查询订单记录列表
     * 
     * @param orderNoList 订单号列表
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findByOrderNoList(List<String> orderNoList);
    
    /**
     * 根据用户ID查询订单列表
     * 
     * @param consumerId 用户ID
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findByConsumerId(Long consumerId);
    
    /**
     * 根据商品ID查询订单列表
     * 
     * @param appItemId 商品ID
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findByAppItemId(Long appItemId);

    /**
     * 根据用户ID和商品ID查询订单列表
     * 
     * @param consumerId 用户ID
     * @param appItemId 商品ID
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findByConsumerIdAndAppItemId(Long consumerId, Long appItemId);

    /**
     * 根据商品ID查询总数
     * 
     * @param appItemId 商品ID
     * @return 总数
     */
    Long findCountByAppItemId(Long appItemId);
    
    /**
     * 根据抽奖号码查询订单记录
     * 
     * @param appItemId 商品ID
     * @param prizeDrawNumber 抽奖号码
     * @return 订单DTO
     */
    ReservationProductOrderDto findByAppItemIdAndPrizeDrawNumber(Long appItemId, Long prizeDrawNumber);

    /**
     * 根据商品ID和抽奖号码列表查询订单列表
     * 
     * @param appItemId 商品ID
     * @param prizeDrawNumberList 抽奖号码列表
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findCountByAppItemIdAndPrizeDrawNumber(Long appItemId, List<Long> prizeDrawNumberList);

    /**
     * 根据商品ID和抽奖状态查询订单列表
     * 
     * @param appItemId 商品ID
     * @param prizeDrawStatus 抽奖状态
     * @return 订单DTO列表
     */
    List<ReservationProductOrderDto> findByAppItemIdAndPrizeDrawStatus(Long appItemId, Integer prizeDrawStatus);
    
    /**
     * 分页查询预约商品订单
     *
     * @param param 分页及筛选参数
     * @return 分页结果
     */
    Page<ReservationProductOrderDto> pageQuery(ReservationProductOrderPageQueryParam param);
    
    /**
     * 根据ID逻辑删除订单记录
     * 
     * @param id 主键ID
     * @return 影响行数
     */
    int deleteById(Long id);
    
    /**
     * 根据商品ID、批量抽奖码、原状态更新抽奖状态
     * 
     * @param appItemId 商品ID
     * @param prizeDrawNumberList 抽奖码列表
     * @param oldPrizeDrawStatus 原抽奖状态
     * @param newPrizeDrawStatus 新抽奖状态
     * @return 影响行数
     * @throws BizException 业务异常
     */
    int updateByAppItemIdAndPrizeDrawNumberListAndStatus(Long appItemId, List<Long> prizeDrawNumberList, Integer oldPrizeDrawStatus, Integer newPrizeDrawStatus) throws BizException;
} 