package cn.com.duiba.anticheat.center.biz.dao;


import cn.com.duiba.wolf.spring.datasource.AutoRoutingDataSource;
import cn.com.duiba.wolf.spring.datasource.TransactionUtils;
import com.google.common.base.Objects;
import org.apache.ibatis.executor.BatchResult;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.jdbc.datasource.DelegatingDataSource;

import javax.sql.DataSource;
import java.util.List;
import java.util.Map;

/**
 * Created by liuyao on 16/7/15.
 */
public class BaseDao implements ApplicationContextAware{

    private static Map<DatabaseSchema, SqlSessionTemplate> sessionTemplateMap;

    private volatile SqlSessionTemplate batchSqlSessionTemplate;

    protected DatabaseSchema databaseSchema;

    public void setDatabaseSchema(DatabaseSchema databaseSchema){
        this.databaseSchema = databaseSchema;
    }

    /**
     * 获取合适的sqlSessionTemplate
     * @param isForceUseMasterDataSource 是否强制使用master库,当为true时会强制使用master库,当为false时会判断当前是否正在事务中,如果是的话底层仍然会使用master库,如果不是的话会根据配置的规则按权重走主库或slave只读库.建议只在执行增删改sql时传入此参数为true.类似select * for update的语句实际上只在事务中有用,故如果不走事务,这样的语句有可能会被分配到只读库.所以写代码时需要注意for update语句一定要放在事务中.
     * @return
     */
    private SqlSessionTemplate getSqlSessionTemplate(boolean isForceUseMasterDataSource){
        DatabaseSchema schema = chooseSchema();
        if(schema == null){
            throw new NullPointerException("please set schema in class:" + this.getClass().getName());
        }
        SqlSessionTemplate sqlSessionTemplate = sessionTemplateMap.get(schema);
        if(Objects.equal(null,sqlSessionTemplate)){
            throw new NullPointerException("当你看到这个报错的时候,你肯定没有配置返回恰当的sqlSessionTemplate,麻烦屈身在spring-datasource.xml配置一下吧");
        }

        DataSource ds = sqlSessionTemplate.getSqlSessionFactory().getConfiguration().getEnvironment().getDataSource();
        if (ds instanceof DelegatingDataSource) {
            ds = ((DelegatingDataSource) ds).getTargetDataSource();
        }

        if(ds instanceof AutoRoutingDataSource){
            AutoRoutingDataSource autoRoutingDataSource = (AutoRoutingDataSource)ds;
            autoRoutingDataSource.determineCurrentLookupKeyByItSelf(isForceUseMasterDataSource);
        }
        return sqlSessionTemplate;
    }

    private SqlSessionTemplate getBatchSqlSessionTemplate(){
        if(batchSqlSessionTemplate==null){
            batchSqlSessionTemplate=new SqlSessionTemplate(getSqlSessionTemplate(true).getSqlSessionFactory(), ExecutorType.BATCH);
        }
        return batchSqlSessionTemplate;
    }

    /**
     * 这个方法延迟到子类实现,由子类决定注入哪个sqlSessionTemplate,以使用不同的库
     * @return
     */
    private DatabaseSchema chooseSchema(){
        if(Objects.equal(null,this.databaseSchema)){
            return DatabaseSchema.DEVELOPER_APP;//默认为主库
        }else{
            return this.databaseSchema;
        }
    }

    public int insert(String statement) {
        return getSqlSessionTemplate(true).insert(addNameSpace(statement));
    }

    public int delete(String statement) {
        return getSqlSessionTemplate(true).delete(addNameSpace(statement));
    }

    public int batchDelete(String statement,Object parameter){
        return getBatchSqlSessionTemplate().delete(addNameSpace(statement),parameter);
    }

    public int update(String statement) {
        return getSqlSessionTemplate(true).update(addNameSpace(statement));
    }

    public int delete(String statement, Object parameter) {
        return getSqlSessionTemplate(true).delete(addNameSpace(statement), parameter);
    }

    public <E> List<E> selectList(String statement, Object parameter, RowBounds rowBounds) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement), parameter, rowBounds);
    }

    public void select(String statement, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), handler);
    }

    public <T> T selectOne(String statement, Object parameter) {
        return getSqlSessionTemplate(false).selectOne(addNameSpace(statement), parameter);
    }

    public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), parameter, mapKey);
    }

    public int insert(String statement, Object parameter) {
        return getSqlSessionTemplate(true).insert(addNameSpace(statement), parameter);
    }

    public <K, V> Map<K, V> selectMap(String statement, Object parameter, String mapKey, RowBounds rowBounds) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), parameter, mapKey, rowBounds);
    }

    public void select(String statement, Object parameter, RowBounds rowBounds, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), parameter, rowBounds, handler);
    }

    public <E> List<E> selectList(String statement) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement));
    }

    public void select(String statement, Object parameter, ResultHandler handler) {
        getSqlSessionTemplate(false).select(addNameSpace(statement), parameter, handler);
    }

    public <K, V> Map<K, V> selectMap(String statement, String mapKey) {
        return getSqlSessionTemplate(false).selectMap(addNameSpace(statement), mapKey);
    }

    public int update(String statement, Object parameter) {
        return getSqlSessionTemplate(true).update(addNameSpace(statement), parameter);
    }

    public <T> T selectOne(String statement) {
        return getSqlSessionTemplate(false).selectOne(addNameSpace(statement));
    }

    public <E> List<E> selectList(String statement, Object parameter) {
        return getSqlSessionTemplate(false).selectList(addNameSpace(statement), parameter);
    }

    /**
     * 获取mybatis命名空间
     * @param method
     * @return
     */
    protected String addNameSpace(String method){
        return getClass().getName()+"."+method;
    }

    public static void setSessionTemplateMap(Map<DatabaseSchema, SqlSessionTemplate> sessionTemplateMap) {
        synchronized (BaseDao.class){
            if(BaseDao.sessionTemplateMap == null){
                BaseDao.sessionTemplateMap = sessionTemplateMap;
            }
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        setSessionTemplateMap(applicationContext.getBean("sessionTemplateMap",Map.class));
    }

    /**
     * 批量更新,注意，此方法一定要在事务中执行才有效，请在上层Service中使用事务，比如加上注解@Transactional
     *
     * @param statement
     * @param paramsList
     */
    protected int batchUpdate(String statement,List<?> paramsList){
        if(!TransactionUtils.isCurrentThreadInTransaction()){
            throw new IllegalStateException("batchInsert must be executed in transaction");
        }
        SqlSessionTemplate batchTemplate = getBatchSqlSessionTemplate();
        statement = addNameSpace(statement);
        for ( Object params : paramsList ) {
           batchTemplate.update(statement, params);
        }

        int lines = 0;
        List<BatchResult> results =  batchTemplate.flushStatements();
        for(BatchResult batchResult : results){
            for(int line : batchResult.getUpdateCounts()) {
                lines += line;
            }
        }
        return lines;
    }

    /**
     * 批量插入,注意，此方法一定要在事务中执行才有效，请在上层Service中使用事务，比如加上注解@Transactional
     * 此方法还是循环调用insert 不要使用，直接使用insert 传参是list即可   --liukai
     * @param statement
     * @param paramsList
     * @deprecated
     */
    @Deprecated
    protected void batchInsert(String statement,List<?> paramsList){
        if(!TransactionUtils.isCurrentThreadInTransaction()){
            throw new IllegalStateException("batchInsert must be executed in transaction");
        }
        SqlSessionTemplate batchTemplate = getBatchSqlSessionTemplate();
        statement = addNameSpace(statement);
        for ( Object params : paramsList ) {
            batchTemplate.insert(statement, params);
        }

        batchTemplate.flushStatements();
    }
}
