package cn.com.duiba.anticheat.center.biz.service.rules.impl;

import cn.com.duiba.anticheat.center.common.exceptions.AnticheatException;
import com.alibaba.fastjson.JSONObject;

import org.apache.commons.lang.StringUtils;
import org.drools.core.impl.KnowledgeBaseImpl;
import org.kie.api.definition.rule.Rule;
import org.kie.api.io.ResourceType;
import org.kie.internal.builder.KnowledgeBuilder;
import org.kie.internal.builder.KnowledgeBuilderFactory;
import org.kie.internal.io.ResourceFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;

import cn.com.duiba.anticheat.center.api.enums.RuleSceneEnum;
import cn.com.duiba.anticheat.center.api.result.rules.RuleChangeDto;
import cn.com.duiba.anticheat.center.biz.dao.rules.RuleConfigDao;
import cn.com.duiba.anticheat.center.biz.entity.rules.RuleConfigEntity;
import cn.com.duiba.anticheat.center.biz.enums.RuleStatusEnum;
import cn.com.duiba.anticheat.center.biz.service.rules.RuleChangeService;
import cn.com.duiba.boot.exception.BizException;


/**
 * 本类只修改数据库
 * Created by sty on 2018/7/25.
 */
@Service
public class RuleChangeServiceImpl implements RuleChangeService {

  private static final Logger LOGGER = LoggerFactory.getLogger(RuleChangeServiceImpl.class);

  @Autowired
  private RuleConfigDao ruleConfigDao;
  @Autowired
  private StringRedisTemplate stringRedisTemplate;

  private static final String REDIS_BORDCAST_TOPIC = "kbaseChange";

  @Override
  public void delete(RuleSceneEnum scene, String ruleName) throws BizException {
    if(scene == null){
      return;
    }
    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if(kbase == null){
      return;
    }
    //知识库查找是否有改规则
    Rule rule = kbase.getRule(scene.getPkg(),ruleName);
    if(rule == null){
      throw  new BizException("无此规则");
    }
    //数据库将规则软删除
    ruleConfigDao.updateRuleByName(ruleName, RuleStatusEnum.CLOSE.getType());
    //发消息
    RuleChangeDto  changeDto = new RuleChangeDto();
    changeDto.setOpType(RuleChangeDto.DELETE);
    changeDto.setRuleName(ruleName);
    changeDto.setScene(scene);
    stringRedisTemplate.convertAndSend(REDIS_BORDCAST_TOPIC, JSONObject.toJSONString(changeDto));
  }

  @Override
  public void add(RuleSceneEnum scene, String ruleContent, String ruleName) throws BizException {
    if(!basicCheck(scene, ruleContent, ruleName)){
      return;
    }

    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if(kbase == null){
      return;
    }
    //查找是否已经存在改规则
    Rule rule = kbase.getRule(scene.getPkg(),ruleName);
    if(rule != null){
      throw  new  BizException("已经存在此规则");
    }
    //知识库增加
    KnowledgeBuilder kb = KnowledgeBuilderFactory.newKnowledgeBuilder();
    //装入规则，可以装入多个,进行编译
    try {
      kb.add(ResourceFactory.newByteArrayResource(ruleContent.getBytes("utf-8")), ResourceType.DRL);
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("规则转为二进制转换出错", e);
      throw  new  BizException("规则转换二进制失败"+e.getMessage());
    }
    if(kb.hasErrors()){
      throw  new  BizException("规则编译失败"+kb.getErrors());
    }
    //数据库将规则增加
    RuleConfigEntity entity = new RuleConfigEntity();
    entity.setRuleName(ruleName);
    entity.setRuleContent(ruleContent);
    entity.setRuleScene(scene.getType());
    entity.setRuleStatus(RuleStatusEnum.OPEN.getType());
    ruleConfigDao.insert(entity);
    //发送消息
    RuleChangeDto  changeDto = new RuleChangeDto();
    changeDto.setOpType(RuleChangeDto.ADD);
    changeDto.setRuleName(ruleName);
    changeDto.setScene(scene);
    changeDto.setRuleContent(ruleContent);
    stringRedisTemplate.convertAndSend(REDIS_BORDCAST_TOPIC, JSONObject.toJSONString(changeDto));
  }

  @Override
  public void update(RuleSceneEnum scene, String ruleContent, String ruleName) throws BizException {
    if(!basicCheck(scene, ruleContent, ruleName)){
      return;
    }

    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if(kbase == null){
      return;
    }
    //查找是否已经存在改规则
    Rule rule = kbase.getRule(scene.getPkg(),ruleName);
    if(rule == null){
      throw  new  BizException("无此规则");
    }
    //知识库增加
    KnowledgeBuilder kb = KnowledgeBuilderFactory.newKnowledgeBuilder();

    //装入规则，可以装入多个
    try {
      kb.add(ResourceFactory.newByteArrayResource(ruleContent.getBytes("utf-8")), ResourceType.DRL);
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("规则转为二进制转换出错", e);
      throw  new  BizException("规则转换二进制失败"+e.getMessage());
    }
    if(kb.hasErrors()){
      throw  new  BizException("规则编译失败"+kb.getErrors());
    }
    //数据库将规则更新
    RuleConfigEntity entity = new RuleConfigEntity();
    entity.setRuleName(ruleName);
    entity.setRuleContent(ruleContent);
    entity.setRuleScene(scene.getType());
    ruleConfigDao.update(entity);

    //发送消息
    RuleChangeDto  changeDto = new RuleChangeDto();
    changeDto.setOpType(RuleChangeDto.UPDATE);
    changeDto.setRuleName(ruleName);
    changeDto.setScene(scene);
    changeDto.setRuleContent(ruleContent);
    stringRedisTemplate.convertAndSend(REDIS_BORDCAST_TOPIC, JSONObject.toJSONString(changeDto));
  }

  @Override
  public Boolean refresh(RuleSceneEnum scene) {
    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if (kbase == null) {
      throw new AnticheatException("找不到场景对应知识库");
    }

    RuleChangeDto  changeDto = new RuleChangeDto();
    changeDto.setOpType(RuleChangeDto.REFRESH);
    changeDto.setScene(scene);
    stringRedisTemplate.convertAndSend(REDIS_BORDCAST_TOPIC, JSONObject.toJSONString(changeDto));
    return Boolean.TRUE;
  }

  @Override
  public Boolean open(RuleSceneEnum scene, String ruleName) {
    KnowledgeBaseImpl kbase = (KnowledgeBaseImpl) KieComponent.getBaseByScene(scene.getType());
    if (kbase == null) {
      throw new AnticheatException("规则知识库不存在");
    }
    Rule rule = kbase.getRule(scene.getPkg(),ruleName);
    // 如果已经存在
    if (rule != null) {
      return Boolean.FALSE;
    }

    RuleConfigEntity entity = new RuleConfigEntity();
    entity.setRuleName(ruleName);
    entity.setRuleScene(scene.getType());
    entity.setRuleStatus(1);
    ruleConfigDao.update(entity);

    RuleChangeDto changeDto = new RuleChangeDto();
    changeDto.setOpType(RuleChangeDto.OPEN);
    changeDto.setRuleName(ruleName);
    changeDto.setScene(scene);
    stringRedisTemplate.convertAndSend(REDIS_BORDCAST_TOPIC, JSONObject.toJSONString(changeDto));

    return Boolean.TRUE;
  }

  /**
   * 基础校验，其中规则文件中应该包含所属类的类名，以及规则名字
   * 规则文件不应该太大，基本一个规则放置在一个文件里面，只要包是同一个，是可以调用func等公共东西
   * @param scene
   * @param ruleContent
   * @param ruleName
   * @return
   */
  private  boolean basicCheck(RuleSceneEnum scene, String ruleContent, String ruleName){
      if(scene == null || StringUtils.isBlank(ruleContent) || StringUtils.isBlank(ruleName)){
        return false;
      }
      return ruleContent.contains(ruleName) && ruleContent.contains(scene.getPkg());
  }
}
