package cn.com.duiba.anticheat.center.biz.remoteservice.goods.impl;

import cn.com.duiba.anticheat.center.api.constant.StaticListConstant;
import cn.com.duiba.anticheat.center.api.domain.goods.BehaviorParams;
import cn.com.duiba.anticheat.center.api.domain.goods.ConsumerParams;
import cn.com.duiba.anticheat.center.api.domain.goods.GoodsParams;
import cn.com.duiba.anticheat.center.api.domain.goods.OrderParams;
import cn.com.duiba.anticheat.center.api.domain.goods.RequestParams;
import cn.com.duiba.anticheat.center.api.remoteservice.goods.RemoteAnticheatCheckService;
import cn.com.duiba.anticheat.center.api.result.goods.ACResultDto;
import cn.com.duiba.anticheat.center.biz.dao.goods.AnticheatDebugLogDao;
import cn.com.duiba.anticheat.center.biz.entity.goods.AnticheatDebugLogEntity;
import cn.com.duiba.anticheat.center.biz.entity.goods.AnticheatStrategyConfigEntity;
import cn.com.duiba.anticheat.center.biz.service.tongdun.FraudApiResponse;
import cn.com.duiba.anticheat.center.biz.service.tongdun.TongdunClient;
import cn.com.duiba.anticheat.center.biz.service.tongdun.TongdunThreadLocal;
import cn.com.duiba.anticheat.center.biz.strategy.goods.AnticheatStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatBlackConsumerStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatBlackIpStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatConsumerDayTimesStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatFirstInTimesStartegy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatFirstNoSwipeStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatFirstSameUaCreditsStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatIpDayTimesStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatSameCreditsAddUpStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatSameCreditsStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatSameDeapStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatSameIpStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatSwipeSkipStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatTongDunHighStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatTongDunMiddleStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatWhiteAppStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatWhiteIpStrategy;
import cn.com.duiba.anticheat.center.biz.strategy.goods.impl.AnticheatWhiteItemStrategy;
import cn.com.duiba.anticheat.center.common.constants.CacheConstants;
import cn.com.duiba.anticheat.center.common.tool.ThreadTool;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.utils.DateUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * 商品兑换防作弊检查
 */
@RestController("remoteAnticheatCheckService")
public class RemoteAnticheatCheckServiceImpl implements InitializingBean, RemoteAnticheatCheckService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteAnticheatCheckServiceImpl.class);

    @Autowired
    private AnticheatBlackIpStrategy anticheatBlackIpStrategy;
    @Autowired
    private AnticheatWhiteIpStrategy anticheatWhiteIpStrategy;
    @Autowired
    private AnticheatSameCreditsStrategy anticheatSameCreditsStrategy;
    @Autowired
    private AnticheatIpDayTimesStrategy anticheatIpDayTimesStrategy;
    @Autowired
    private AnticheatConsumerDayTimesStrategy anticheatConsumerDayTimesStrategy;
    @Autowired
    private AnticheatSwipeSkipStrategy anticheatSwipeSkipStrategy;
    @Autowired
    private AnticheatSameDeapStrategy anticheatSameDeapStrategy;
    @Autowired
    private AnticheatFirstInTimesStartegy anticheatFirstInTimesStartegy;
    @Autowired
    private AnticheatFirstNoSwipeStrategy anticheatFirstNoSwipeStrategy;
    @Autowired
    private AnticheatSameCreditsAddUpStrategy anticheatSameCreditsAddUpStrategy;
    @Autowired
    private AnticheatSameIpStrategy anticheatSameIpStrategy;
    @Autowired
    private AnticheatBlackConsumerStrategy anticheatBlackConsumerStrategy;
    @Autowired
    private AnticheatFirstSameUaCreditsStrategy anticheatFirstSameUaCreditsStrategy;
    @Autowired
    private AnticheatWhiteItemStrategy anticheatWhiteItemStrategy;
    @Autowired
    private AnticheatTongDunMiddleStrategy anticheatTongDunMiddleStrategy;
    @Autowired
    private AnticheatTongDunHighStrategy anticheatTongDunHighStrategy;
    @Autowired
    private AnticheatWhiteAppStrategy anticheatWhiteAppStrategy;
    @Resource
    private ExecutorService executorService;
    @Resource
    private ScheduledExecutorService scheduledExecutorService;
    @Autowired
    private AnticheatDebugLogDao anticheatDebugLogDao;

    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;

    /**
     * 扫描周期5分钟
     */
    private static final int CIRCLE = 5;
    /**
     * 超时时间 150毫秒
     */
    private static final int TIMEOUT_TIME = 150;

    /**
     * 信号量
     */
    private static final Semaphore SEMAPHORE = new Semaphore(100);

    private List<AnticheatStrategy> tongDunStrategies = new ArrayList<>();

    private List<AnticheatStrategy> whiteStrategies = new ArrayList<>();

    private List<AnticheatStrategy> blackStrategies = new ArrayList<>();

    @Override
    public void afterPropertiesSet() throws Exception {
        tongDunStrategies.add(anticheatTongDunMiddleStrategy);
        tongDunStrategies.add(anticheatTongDunHighStrategy);

        blackStrategies.add(anticheatBlackIpStrategy);
        blackStrategies.add(anticheatSameCreditsStrategy);
        blackStrategies.add(anticheatIpDayTimesStrategy);
        blackStrategies.add(anticheatConsumerDayTimesStrategy);
        blackStrategies.add(anticheatSwipeSkipStrategy);
        blackStrategies.add(anticheatSameDeapStrategy);
        blackStrategies.add(anticheatFirstInTimesStartegy);
        blackStrategies.add(anticheatFirstNoSwipeStrategy);
        blackStrategies.add(anticheatSameCreditsAddUpStrategy);
        blackStrategies.add(anticheatSameIpStrategy);
        blackStrategies.add(anticheatBlackConsumerStrategy);
        blackStrategies.add(anticheatFirstSameUaCreditsStrategy);

        whiteStrategies.add(anticheatWhiteAppStrategy);
        whiteStrategies.add(anticheatWhiteIpStrategy);
        whiteStrategies.add(anticheatWhiteItemStrategy);

        // 启动优先级扫描
        sortScheduler();
    }


    private String getIpKey(String ip) {
        return CacheConstants.AC_ANTICHEAT_EXCHANGE + "-ip-" + ip;
    }

    private String getConsumerKey(Long cid) {
        return CacheConstants.AC_ANTICHEAT_EXCHANGE + "-cid-" + cid;
    }

    @Override
    public DubboResult<Void> onOrderCreate(OrderParams order) {
        try {
            String ipKey = getIpKey(order.getIp());
            redisAtomicClient.incrBy(ipKey,1,DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            String cidKey = getConsumerKey(order.getConsumerId());
            redisAtomicClient.incrBy(cidKey,1,DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            return DubboResult.successResult(null);
        } catch (Exception e) {
            LOGGER.error("onOrderCreate, error", e);
            return DubboResult.failResult("onOrderCreate, error");
        }
    }

    @Override
    public DubboResult<Void> onOrderFail(OrderParams order) {
        try {
            String ipKey = getIpKey(order.getIp());
            redisAtomicClient.incrBy(ipKey,-1,DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            String cidKey = getConsumerKey(order.getConsumerId());
            redisAtomicClient.incrBy(cidKey,-1,DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            return DubboResult.successResult(null);
        } catch (Exception e) {
            LOGGER.error("onOrderFail, error", e);
            return DubboResult.failResult("onOrderFail, 异常");
        }
    }

    @Override
    public DubboResult<ACResultDto> checkCouponExchange(final ConsumerParams consumer, final GoodsParams goods, final RequestParams request, final BehaviorParams behavior) {
        return check(consumer, goods, request, behavior, new ACResultDto(true, null, null, ACResultDto.SUB_RESULT_DEFAULT));
    }
    
    private DubboResult<ACResultDto> check(final ConsumerParams consumer, final GoodsParams goods, final RequestParams request, final BehaviorParams behavior, ACResultDto defaultResult) {
        //命中APP白名单直接通过
        if(StaticListConstant.matchWhiteApp(consumer.getAppId())){
            return DubboResult.successResult(new ACResultDto(true, null, null, ACResultDto.SUB_RESULT_DEFAULT));
        }
        Callable<DubboResult<ACResultDto>> callable = new Callable<DubboResult<ACResultDto>>() {
            @Override
            public DubboResult<ACResultDto> call() throws Exception {
                ACResultDto result = innerCheckCouponExchange(consumer, goods, request, behavior);
                return DubboResult.successResult(result);
            }
        };
        if (SEMAPHORE.tryAcquire()) {
            Future<DubboResult<ACResultDto>> future = null;
            try {
                future = executorService.submit(callable);
                DubboResult<ACResultDto> result = future.get(TIMEOUT_TIME, TimeUnit.MILLISECONDS);
                return result != null ? result : DubboResult.successResult(defaultResult);
            } catch(TimeoutException e){
                LOGGER.info("兑换项防作弊降级,业务超时, errMsg={}", e.getMessage());
                LOGGER.debug("兑换项防作弊降级,业务超时", e);
                ThreadTool.cancelTask(future);
                return DubboResult.successResult(defaultResult);
            }catch (Exception e) {
                LOGGER.warn("兑换项防作弊降级,异常信息", e);
                ThreadTool.cancelTask(future);
                return DubboResult.successResult(defaultResult);
            } finally {
                SEMAPHORE.release();
            }
        } else {
            LOGGER.info("兑换项防作弊降级通过,流量超了：consumerId={}, gtype={}, gid={}", consumer.getConsumerId(), goods.getGtype(), goods.getGid());
            return DubboResult.successResult(defaultResult);
        }
    }
    
    /**
     * @see cn.com.duiba.anticheat.center.api.remoteservice.goods.RemoteAnticheatCheckService#checkGoodsWebExchange(cn.com.duiba.anticheat.center.api.domain.goods.ConsumerParams, cn.com.duiba.anticheat.center.api.domain.goods.GoodsParams, cn.com.duiba.anticheat.center.api.domain.goods.RequestParams, cn.com.duiba.anticheat.center.api.domain.goods.BehaviorParams)
     */
    @Override
    public DubboResult<ACResultDto> checkGoodsWebExchange(ConsumerParams consumer, GoodsParams item,
                                                          RequestParams request, BehaviorParams behavior) {
        return check(consumer, item, request, behavior, new ACResultDto(false, null, null, ACResultDto.SUB_RESULT_NEED_IDENTIFYING_CODE));
    }

    @Override
    public DubboResult<Void> fillbackDebugIds(List<Long> debugIds, Long orderId) {
        try {
            AnticheatDebugLogEntity debug;
            for (Long it : debugIds) {
                debug = new AnticheatDebugLogEntity(it);
                debug.setOrderId(orderId);
                anticheatDebugLogDao.update(debug);
            }
            return DubboResult.successResult(null);
        } catch (Exception e) {
            LOGGER.error("回填 dubugIds 异常", e);
            return DubboResult.failResult("回填 debugIds 异常");
        }
    }

    private ACResultDto innerCheckCouponExchange(ConsumerParams consumer, GoodsParams goods, RequestParams request, BehaviorParams behavior) {
        // 白名单，如有有命中，即刻放行
        List<Long> debugIds = new ArrayList<>();
        ACResultDto ret;

        //白名单验证
        ret = whiteStrategyMatch(consumer, goods, request, behavior, debugIds);
        if(ret != null){
            return ret;
        }

        // 黑名单，如果有命中，即刻拒绝
        ret = blackStrategyMatch(consumer, goods, request, behavior, debugIds, blackStrategies);
        if(ret != null){
            return ret;
        }

        //同盾验证
        ret = tongDunStrategyMatch(consumer, goods, request, behavior, debugIds);
        if(ret != null){
            return ret;
        }

        // 如果都没有命中，默认放行
        return new ACResultDto(true, null, debugIds, ACResultDto.SUB_RESULT_DEFAULT);
    }

    private ACResultDto whiteStrategyMatch(ConsumerParams consumer, GoodsParams goods, RequestParams request,
                                           BehaviorParams behavior, List<Long> debugIds){
        for (AnticheatStrategy as : whiteStrategies) {
            Transaction t = Cat.newTransaction("URL", as.getClass().getSimpleName());
            try {
                AnticheatStrategy.AnticheatStrategyResult ret = as.checkCouponExchangeFast(consumer, goods, request, behavior);
                t.setStatus(Transaction.SUCCESS);
                if (ret.isMatch()) {
                    debugIds.add(ret.getDebugId());
                }
                if (judgeMatch(ret, as, consumer)) {
                    return new ACResultDto(true, "", debugIds, ACResultDto.SUB_RESULT_DEFAULT);
                }
            } catch (Exception e) {
                t.setStatus(e);
                throw e;
            } finally {
                t.complete();
            }
        }
        return null;
    }

    private ACResultDto tongDunStrategyMatch(ConsumerParams consumer, GoodsParams goods, RequestParams request,
                                           BehaviorParams behavior, List<Long> debugIds){
        List<AnticheatStrategy> targetStrategies = new ArrayList<>();
        for(AnticheatStrategy strategy : tongDunStrategies){
            if(strategy.needValid(consumer)){
                targetStrategies.add(strategy);
            }
        }

        if(targetStrategies.isEmpty()){
            return null;
        }

        FraudApiResponse response = TongdunClient.checkExchangeTongdun(consumer, goods, request);
        TongdunThreadLocal.get().setApiResponse(response);

        return blackStrategyMatch(consumer, goods, request, behavior, debugIds, targetStrategies);
    }

    private ACResultDto blackStrategyMatch(ConsumerParams consumer, GoodsParams goods, RequestParams request,
                                           BehaviorParams behavior, List<Long> debugIds, List<AnticheatStrategy> strategies){
        for (AnticheatStrategy as : strategies) {
            Transaction t = Cat.newTransaction("URL", as.getClass().getSimpleName());
            try {
                AnticheatStrategy.AnticheatStrategyResult ret = as.checkCouponExchangeFast(consumer, goods, request, behavior);
                t.setStatus(Transaction.SUCCESS);
                if (ret.isMatch()) {
                    debugIds.add(ret.getDebugId());
                }
                if(judgeMatch(ret, as, consumer)){
                    return new ACResultDto(false, "黑名单命中", debugIds, getCheckMode(as.getCheckMode()));
                }
            } catch (Exception e) {
                t.setStatus(e);
                throw e;
            } finally {
                t.complete();
            }
        }
        return null;
    }

    /**
     * 把黑名单策略重新排序
     * 排序策略：启用的排前，直接拒绝的排前面
     */
    public List<AnticheatStrategy> sortBlackStrategies(List<AnticheatStrategy> list) {
        Collections.sort(list, new Comparator<AnticheatStrategy>() {
            @Override
            public int compare(AnticheatStrategy o1, AnticheatStrategy o2) {
                int ib = compareBoolean(o1.isEnable(), o2.isEnable());
                if (ib != 0) {
                    return ib;
                }
                return compareInt(o1.getCheckMode(), o2.getCheckMode());
            }

            private int compareInt(int i1, int i2) {
                if (i1 == i2) {
                    return 0;
                }
                if (i1 == 0) {
                    return -1;
                }
                return 1;
            }

            private int compareBoolean(boolean b1, boolean b2) {
                if (b1 == b2) {
                    return 0;
                }
                if (b1) {
                    return -1;
                }
                return 1;
            }
        });
        return list;
    }

    private void sortScheduler() {
        scheduledExecutorService.scheduleWithFixedDelay(() -> {
            List<AnticheatStrategy> newList = new ArrayList<>(blackStrategies);
            blackStrategies = sortBlackStrategies(newList);
            LOGGER.info("黑名单策略排序扫描执行~");
        }, 0, CIRCLE, TimeUnit.MINUTES);
    }

    /**
     * 策略有效命中判定
     * @param ret
     * @param as
     * @param consumer
     * @return
     */
    private boolean judgeMatch(AnticheatStrategy.AnticheatStrategyResult ret,
                                   AnticheatStrategy as, ConsumerParams consumer){
        if (!ret.isMatch()) {
            return false;
        }
        if (!as.needValid(consumer)) {
            return false;
        }
        return true;
    }

    private Integer getCheckMode(int checkMode){
        if (checkMode == AnticheatStrategyConfigEntity.CHECK_MODE_DEFAULT) {    // 使用默认方式拒绝
            return ACResultDto.SUB_RESULT_DEFAULT;
        } else if (checkMode == AnticheatStrategyConfigEntity.CHECK_MODE_NEED_IDENTIFYING_CODE) {    // 使用需要验证码的方式拒绝
            return ACResultDto.SUB_RESULT_NEED_IDENTIFYING_CODE;
        }
        return null;
    }
}
