package cn.com.duiba.anticheat.center.biz.service.risk.impl;

import cn.com.duiba.anticheat.center.api.dto.RiskRouteDto;
import cn.com.duiba.anticheat.center.api.enums.ActRiskSenceEnum;
import cn.com.duiba.anticheat.center.api.enums.ProjectXLabelEnum;
import cn.com.duiba.anticheat.center.api.enums.RiskRoutBizEnum;
import cn.com.duiba.anticheat.center.api.param.RiskRouteParam;
import cn.com.duiba.anticheat.center.api.param.RouteMatchingParam;
import cn.com.duiba.anticheat.center.biz.constant.RedisKeyFactory;
import cn.com.duiba.anticheat.center.biz.dao.risk.RiskRouteConfigDao;
import cn.com.duiba.anticheat.center.biz.entity.risk.RiskRouteConfigEntity;
import cn.com.duiba.anticheat.center.biz.service.risk.RiskRouteService;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.cache.CacheLoader;
import cn.com.duiba.wolf.utils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * @author JunAngLiu
 * @Title: RiskRoutServiceImpl
 * @Description:
 * @date 2019/9/714:30
 */
@Service
public class RiskRouteServiceImpl implements RiskRouteService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RiskRouteServiceImpl.class);

    @Autowired
    private RiskRouteConfigDao dao;
    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;




    @Override
    public boolean insert(RiskRouteDto dto) throws BizException {
        if(Objects.isNull(dto)
                || RiskRoutBizEnum.getByType(dto.getBizType()) == null
                || ActRiskSenceEnum.getEnumByValue(dto.getScene()) == null
                || StringUtils.isBlank(dto.getSceneIdentity())
                || dto.getPunishWay() == null
                || StringUtils.isBlank(dto.getCopy())){
            LOGGER.warn("新增风控路由配置参数校验不通过,入参={}",dto);
            return false;
        }

        //校验当前类型是否存在
        List<RiskRouteConfigEntity> entities = this.checkExist(dto);
        if(CollectionUtils.isNotEmpty(entities)){
            throw new BizException("当前活动已配置该场景");
        }
        //插入数据
        if(dao.save(BeanUtils.copy(dto, RiskRouteConfigEntity.class)) > 0){
            advancedCacheClient.remove(this.getKey(dto.getBizType()));
            return true;
        }
        return false;

    }



    @Override
    public boolean updateById(RiskRouteDto dto) throws BizException{
        if(Objects.isNull(dto)
                || dto.getId() == null){
            LOGGER.warn("更新风控路由配置参数校验不通过,入参={}",dto);
            return false;
        }
        RiskRouteConfigEntity entity = dao.getById(dto.getId());
        if(entity == null){
            throw new BizException("id不存在");
        }
        //非 删除操作，则需校验 配置是否存在
        if(dto.getDeleted() != null && dto.getDeleted() != 1){
            //校验当前类型是否存在
            List<RiskRouteConfigEntity> entities = this.checkExist(dto);
            if(CollectionUtils.isNotEmpty(entities) && !entities.get(0).getId().equals(dto.getId())){
                throw new BizException("当前配置已存在");
            }
        }

        if(dao.updateById(BeanUtils.copy(dto,RiskRouteConfigEntity.class)) > 0){
            advancedCacheClient.remove(this.getKey(dto.getBizType()));
            if(!Objects.equals(dto.getBizType(),entity.getBizType())){
                advancedCacheClient.remove(this.getKey(entity.getBizType()));
            }
            return true;
        }
        return false;
    }



    @Override
    public List<RiskRouteDto> getListByParam(RiskRouteParam param) {
        return BeanUtils.copyList(dao.listByParams(param),RiskRouteDto.class);
    }



    @Override
    public int countByParam(RiskRouteParam param) {
        return dao.countByParams(param);
    }




    @Override
    public RiskRouteDto routeMatch(RouteMatchingParam param) {

        //查询活动的路由配置列表
        RiskRouteCacheWithBizType actCache = new RiskRouteCacheWithBizType(param.getBizEnum().getType());
        List<RiskRouteConfigEntity> routeConfs = advancedCacheClient.getWithCacheLoader(getKey(param.getBizEnum().getType()), 5, TimeUnit.MINUTES, actCache);
        Integer scene = param.getSenceEnum().getValue();
        //如果查询为空，则走 通用的风控路由
        if(CollectionUtils.isEmpty(routeConfs)){
            return this.getOrtherRoute(scene);
        }

        //非星速台活动
        if(Objects.equals(param.getBizEnum(),RiskRoutBizEnum.COMMON_ACTIVITY)){
            if(param.getActivityType() == null){
                return this.getOrtherRoute(scene);
            }
            Optional<RiskRouteConfigEntity> optional = routeConfs.stream().filter(e -> Objects.equals(e.getActivityType(), param.getActivityType())
                                                                                       && Objects.equals(e.getScene(), scene)).findAny();
            return optional.isPresent() ? BeanUtils.copy(optional.get(),RiskRouteDto.class) : this.getOrtherRoute(scene);
        }

        //星速台活动
        Integer label = param.getLabel().getCode();//标签
        if(Objects.equals(param.getLabel(),ProjectXLabelEnum.CUSTOM)){
            //自定义标签：还需要匹配 projectId
            Optional<RiskRouteConfigEntity> optional = routeConfs.stream().filter(e -> Objects.equals(e.getProjectLabel(),label)
                                                                                        && Objects.equals(e.getScene(), scene)
                                                                                        && StringUtils.equals(e.getProjectId(),param.getProjectId())).findAny();
            return optional.isPresent() ? BeanUtils.copy(optional.get(),RiskRouteDto.class) : this.getOrtherRoute(scene);
        }
        //非自定义标签
        Optional<RiskRouteConfigEntity> optional = routeConfs.stream().filter(e -> Objects.equals(e.getProjectLabel(),label)
                                                                                    && Objects.equals(e.getScene(), scene)).findAny();
        return optional.isPresent() ? BeanUtils.copy(optional.get(),RiskRouteDto.class) : this.getOrtherRoute(scene);

    }


    /**
     * 根据场景获取 其它风控 路由配置
     * @param scene
     * @return
     */
    private RiskRouteDto getOrtherRoute(Integer scene){
        Integer elseBizType = RiskRoutBizEnum.ELSE.getType();
        RiskRouteCacheWithBizType elseCache = new RiskRouteCacheWithBizType(elseBizType);
        List<RiskRouteConfigEntity> routeConfs = advancedCacheClient.getWithCacheLoader(getKey(elseBizType), 5, TimeUnit.MINUTES, elseCache);
        // 匹配对应的场景
        Optional<RiskRouteConfigEntity> optional = routeConfs.stream().filter(e -> Objects.equals(e.getScene(), scene)).findAny();
        return optional.isPresent() ? BeanUtils.copy(optional.get(),RiskRouteDto.class) : null;
    }







    private class RiskRouteCacheWithBizType  implements CacheLoader<List<RiskRouteConfigEntity>> {

        Integer bizType;
        public RiskRouteCacheWithBizType(Integer bizType) {
            this.bizType = bizType;
        }

        @Override
        public List<RiskRouteConfigEntity> load() {
            RiskRouteParam param = new RiskRouteParam();
            param.setBizType(this.bizType);
            List<RiskRouteConfigEntity> riskRouteConfigEntities = dao.listByParams(param);
            return CollectionUtils.isEmpty(riskRouteConfigEntities) ? Collections.emptyList() : riskRouteConfigEntities;
        }
    }




    /**
     * 校验当前活动是否已配置该场景
     */
    private List<RiskRouteConfigEntity> checkExist(RiskRouteDto dto) throws BizException{
        RiskRouteParam queryParam = new RiskRouteParam();
        queryParam.setBizType(dto.getBizType());
        queryParam.setScene(dto.getScene());
        if(Objects.equals(dto.getBizType(),RiskRoutBizEnum.COMMON_ACTIVITY.getType())){
            if(dto.getActivityType() == null){
                throw new BizException("风控路由配置参数校验不通过,入参ActivityType为空");
            }
            queryParam.setActivityType(dto.getActivityType());
        }else if(Objects.equals(dto.getBizType(),RiskRoutBizEnum.PROJECTX_ACTIVITY.getType())){
            if(dto.getProjectLabel() == null ){
                throw new BizException("风控路由配置参数校验不通过,入参ProjectLabel为空");
            }
            queryParam.setProjectLabel(dto.getProjectLabel());
            if(Objects.equals(dto.getProjectLabel(), ProjectXLabelEnum.CUSTOM.getCode())){
                if(StringUtils.isBlank(dto.getProjectId())){
                    throw new BizException("风控路由配置参数校验不通过,入参ProjectId为空");
                }
                queryParam.setProjectId(dto.getProjectId());
            }
        }
        return dao.listByParams(queryParam);

    }



    /**
     *
     * key值
     * @param bizType
     * @return
     */
    private String getKey(Integer bizType){
        return RedisKeyFactory.K031.toString() + bizType;
    }



}
