package cn.com.duiba.apollo.portal.biz.service;

import cn.com.duiba.apollo.portal.biz.domain.AppConfigKey;
import cn.com.duiba.apollo.portal.biz.jpa.apollo.entity.AccessKey;
import cn.com.duiba.apollo.portal.biz.jpa.apollo.entity.AppConfig;
import cn.com.duiba.apollo.portal.biz.jpa.apollo.repository.AccessKeyRepository;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * @author liuyao
 */
public class AccessKeyService {

    private static final int ACCESSKEY_COUNT_LIMIT = 5;

    @Resource
    private AccessKeyRepository accessKeyRepository;
    @Resource
    private AppConfigService appConfigService;

    private String cluster;

    @PostConstruct
    public void init(){
        if(StringUtils.isBlank(cluster)){
            throw new RuntimeException("没有指定集群");
        }
    }

    public void setCluster(String cluster) {
        this.cluster = cluster;
    }

    public Boolean enableAccessKey(String appId, Long adminId){
        boolean enable = isAccessKeyEnable(appId);
        AppConfig params = new AppConfig();
        params.setAppId(appId);
        params.setCluster(cluster);
        params.setKey(AppConfig.ACCESS_KEY_ENABLE);
        params.setValue(Boolean.toString(!enable));
        params.setDataChangeLastModifiedBy(adminId);
        appConfigService.config(params);
        return !enable;
    }

    public boolean isAccessKeyEnable(String appId){
        AppConfigKey key = new AppConfigKey();
        key.setKey(AppConfig.ACCESS_KEY_ENABLE);
        key.setCluster(cluster);
        key.setAppId(appId);
        return appConfigService.getBoolean(key);
    }

    public List<AccessKey> findByAppId(String appId) {
        return accessKeyRepository.findByClusterAndAppId(cluster,appId);
    }

    @Transactional
    public AccessKey create(String appId, Long operator) throws Exception {
        long count = accessKeyRepository.countByClusterAndAppId(cluster,appId);
        if (count >= ACCESSKEY_COUNT_LIMIT) {
            throw new Exception("AccessKeys count limit exceeded");
        }
        String secret = UUID.randomUUID().toString().replaceAll("-", "");
        AccessKey entity = new AccessKey();
        entity.setId(0L);
        entity.setAppId(appId);
        entity.setCluster(cluster);
        entity.setSecret(secret);
        entity.setDataChangeLastModifiedBy(operator);
        entity.setDataChangeCreatedBy(operator);
        return accessKeyRepository.save(entity);
    }

    @Transactional
    public AccessKey update(String appId, AccessKey entity) throws Exception {
        long id = entity.getId();
        Long operator = entity.getDataChangeLastModifiedBy();

        AccessKey accessKey = accessKeyRepository.findOneByClusterAndAppIdAndId(cluster,appId, id);
        if (accessKey == null) {
            throw new Exception("AccessKey not exist");
        }

        accessKey.setEnabled(entity.isEnabled());
        accessKey.setDataChangeLastModifiedBy(operator);
        accessKeyRepository.save(accessKey);

        return accessKey;
    }

    @Transactional
    public void delete(String appId, long id, Long operator) throws Exception {
        AccessKey accessKey = accessKeyRepository.findOneByClusterAndAppIdAndId(cluster,appId, id);
        if (accessKey == null) {
            throw new Exception("AccessKey not exist");
        }

        if (accessKey.isEnabled()) {
            throw new Exception("AccessKey should disable first");
        }
        accessKey.setDeleted(Boolean.TRUE);
        accessKey.setDataChangeLastModifiedBy(operator);
        accessKeyRepository.save(accessKey);
    }

    public boolean examine(String appId,String accessKey) {
        if(StringUtils.isBlank(accessKey)){
            return false;
        }
        if(!isAccessKeyEnable(appId)){
            return false;
        }
        Set<String> accessKeys = Sets.newHashSet();
        List<AccessKey> accessKeyList = accessKeyRepository.findByClusterAndAppId(cluster,appId);
        for(AccessKey key:accessKeyList){
            if(key.isEnabled()){
                accessKeys.add(key.getSecret());
            }
        }
        return accessKeys.contains(accessKey);
    }
}
