package cn.com.duiba.apollo.client.core;

import cn.com.duiba.apollo.client.ApolloClientProperties;
import cn.com.duiba.apollo.client.dto.AccessTokenDTO;
import cn.com.duiba.apollo.client.params.ResourceAccessTokenParams;
import cn.com.duiba.apollo.client.remoteservice.RemoteResourceAccessTokenService;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.NetUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.Resource;
import java.util.concurrent.atomic.AtomicReference;

@Aspect
@Slf4j
public class ApolloAccessTokenService {

    @Resource
    private ApolloClientProperties apolloClientProperties;
    @Resource
    private RemoteResourceAccessTokenService remoteResourceAccessTokenService;

    private AtomicReference<String> reference = new AtomicReference<>();

    @AfterThrowing(value="execution(* cn.com.duiba.apollo.client.remoteservice.RemoteResourceAccessTokenService.verifyAccessToken(..))",throwing="throwable")
    public void afterThrowing(Throwable throwable){
        if(throwable instanceof BizException){
            log.error("accessToken验证失败"+reference.get(),throwable);
            flashAccessToken();
        }
    }

    public String getAccessToken(){
        if(StringUtils.isBlank(reference.get())){
            flashAccessToken();
        }
        if(StringUtils.isBlank(reference.get())){
            throw new RuntimeException("未能获取到accessToken");
        }
        return reference.get();
    }

    @Scheduled(cron = "0 0 0/1 * * ?")
    public void autoFlushAccessToken(){
        if(StringUtils.isBlank(reference.get())){
            return;
        }
        flashAccessToken();
    }

    private synchronized void flashAccessToken(){
        if(StringUtils.isBlank(apolloClientProperties.getAccessKey()) || StringUtils.isBlank(apolloClientProperties.getAccessSecret())){
            return;
        }
        ResourceAccessTokenParams params = new ResourceAccessTokenParams();
        params.setAccessKey(apolloClientProperties.getAccessKey());
        params.setAccessSecret(apolloClientProperties.getAccessSecret());
        params.setIp(NetUtils.getLocalIp());

        try{
            AccessTokenDTO accessTokenDTO = remoteResourceAccessTokenService.acquireAccessToken(params);
            String accessToken = reference.getAndSet(accessTokenDTO.getAccessToken());
            if(StringUtils.isNotBlank(accessToken)){
                remoteResourceAccessTokenService.deleteAccessToken(accessToken);
            }
        }catch (BizException e){
            log.error("获取accessToken失败",e);
            reference.set(null);
        }
    }

}
