package cn.com.duiba.application.boot.stream.binding;

import cn.com.duiba.application.boot.stream.binder.Binder;
import cn.com.duiba.application.boot.stream.binder.BinderMessageHandler;
import cn.com.duiba.application.boot.stream.binder.BinderMessageProducer;
import cn.com.duiba.application.boot.stream.channel.ChannelFactoryBean;
import cn.com.duiba.application.boot.stream.channel.ChannelKey;
import cn.com.duiba.application.boot.stream.channel.ChannelType;
import cn.com.duiba.application.boot.stream.channel.ProducerMessageChannel;
import cn.com.duiba.application.boot.stream.config.BindingProperties;
import cn.com.duiba.application.boot.stream.config.BindingServiceProperties;
import cn.com.duiba.application.boot.stream.config.ConsumerProperties;
import cn.com.duiba.application.boot.stream.config.ProducerProperties;
import cn.com.duiba.application.boot.stream.configuration.BinderTypeRegistry;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.messaging.support.AbstractMessageChannel;

import javax.annotation.Resource;
import java.util.Map;

@Slf4j
public class BindingService implements DisposableBean, ApplicationRunner,Ordered {

    private final Map<ChannelKey,Binding> bindingCache = Maps.newConcurrentMap();

    @Resource
    private BindingServiceProperties bindingServiceProperties;
    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private BinderTypeRegistry binderTypeRegistry;

    private synchronized void createBinding(){

        if(!bindingCache.isEmpty()){
            return;
        }
        Map<String, BindingProperties> bindingPropertiesMap = bindingServiceProperties.getBindings();

        for(String bindingName: bindingPropertiesMap.keySet()){

            BindingProperties bindingProperties = bindingPropertiesMap.get(bindingName);
            ProducerProperties producerProperties = bindingProperties.getProducer();
            if(producerProperties.getEnable()){
                ChannelKey channelKey = new ChannelKey();
                channelKey.setChannelType(ChannelType.OUTPUT);
                channelKey.setBindingName(bindingName);
                bindProducer(channelKey);
            }
            ConsumerProperties consumerProperties = bindingProperties.getConsumer();
            if(consumerProperties.getEnable()){
                ChannelKey channelKey = new ChannelKey();
                channelKey.setChannelType(ChannelType.INPUT);
                channelKey.setBindingName(bindingName);
                bindConsumer(channelKey);
            }
        }
    }

    @EventListener(ChannelBindingEvent.class)
    public void channelBindingSubscribable(ChannelBindingEvent event){
        log.info(event.getChannelKey()+"绑定完成");
    }

    public Binding getBinding(ChannelKey key){
        if (!bindingCache.containsKey(key)){
            throw new NullPointerException(key+"对应的通道不存在");
        }
        return bindingCache.get(key);
    }


    private void bindProducer(ChannelKey key){

        Map<String, BindingProperties> bindings = bindingServiceProperties.getBindings();
        BindingProperties bindingProperties = bindings.get(key.getBindingName());

        bindingProperties.setBindingName(key.getBindingName());

        Binder binder = findBinder(bindingProperties.getBinder());

        BinderMessageHandler messageHandler = binder.createProducerMessageHandler(bindingProperties);

        ProducerBinding binding = new ProducerBinding();
        binding.setMessageHandler(messageHandler);

        String producerChannelBeanName = ChannelFactoryBean.getBeanNameByBindingName(key);
        ProducerMessageChannel channel = applicationContext.getBean(producerChannelBeanName, ProducerMessageChannel.class);
        binding.setMessageChannel(channel);

        bindingCache.put(key,binding);

        ChannelBindingEvent event = new ChannelBindingEvent();
        event.setChannelKey(key);
        applicationContext.publishEvent(event);

        binding.doBinding();

    }

    private void bindConsumer(ChannelKey key){
        Map<String, BindingProperties> bindings = bindingServiceProperties.getBindings();
        BindingProperties bindingProperties = bindings.get(key.getBindingName());
        bindingProperties.setBindingName(key.getBindingName());
        Binder binder = findBinder(bindingProperties.getBinder());
        BinderMessageProducer messageProducer = binder.createConsumerEndpoint(bindingProperties);

        SubscribableBinding subscribableBinding = new SubscribableBinding();
        subscribableBinding.setMessageProducer(messageProducer);

        String consumerChannelBeanName = ChannelFactoryBean.getBeanNameByBindingName(key);

        AbstractMessageChannel channel = applicationContext.getBean(consumerChannelBeanName, AbstractMessageChannel.class);
        subscribableBinding.setMessageChannel(channel);
        bindingCache.put(key,subscribableBinding);

        ChannelBindingEvent event = new ChannelBindingEvent();
        event.setChannelKey(key);
        applicationContext.publishEvent(event);

        subscribableBinding.doBinding();
    }


    @Override
    public void destroy() {
        for(Binding binding:bindingCache.values()){
            binding.stop();
        }
    }

    private Binder findBinder(String binderName){
        String beanName = binderTypeRegistry.findBeanNameByBinderType(binderName);
        return applicationContext.getBean(beanName,Binder.class);
    }

    @Override
    public void run(ApplicationArguments args) {
        createBinding();
    }


    @Override
    public int getOrder() {
        return -10;
    }
}
