package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

/**
 * 合并druid查询出来的指标数据，方便业务开发
 * @author xugf 2019-10-23
 */
public class DruidUtil {

    /**
     * 合并druid查询出的指标
     * @param clazz 实例class
     * @param dimensions 维度组合，用 "," 分隔
     * @param ObjectList 多个指标对应的实例集合数组
     * @param <T> 实例对应的泛型
     * @return List<T>
     */
    @SafeVarargs
    public static <T> List<T> mergeList(Class<T> clazz, String dimensions, List<T>... ObjectList) {
        //维度字段必须要有值
        if(StringUtils.isBlank(dimensions)) {
            return null;
        }

        //维度组合
        String[] dimArray = StringUtils.split(dimensions, ",");

        Map<String, List<Object>> map = new HashMap<>();
        for(List<T> list : ObjectList) {
            setValue(dimArray, map, list);
        }

        List<T> resultList = new ArrayList<>();
        for(Map.Entry<String, List<Object>> entry : map.entrySet()) {
            Object[] objectArray = entry.getValue().toArray();
            resultList.add(merge(clazz, objectArray));
        }

        return resultList;
    }

    /**
     * 合并druid查询出的指标
     * @param clazz 实例class
     * @param objects 多个指标对应的多个实例数组
     * @param <T> 实例对应的泛型
     * @return 实体类
     */
    public static <T> T merge(Class<T> clazz, Object... objects) {
        JSONObject json = new JSONObject();
        for(Object object : objects) {
            mergeFields(object, json);
        }
        return json.toJavaObject(clazz);
    }

    /**
     * 获取维度组合对应的key，用于指标关联
     * @param dimArray 维度对应的数组
     * @param jsonObject 实例转成的jsonObject
     * @return key
     */
    private static String getKey(String[] dimArray, JSONObject jsonObject) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String dimension : dimArray) {
            String value = jsonObject.getString(dimension.trim());
            value = StringUtils.isBlank(value) ? "0" : value;
            stringBuilder.append(value).append("-");
        }
        stringBuilder.deleteCharAt(stringBuilder.length() - 1);
        return stringBuilder.toString();
    }

    /**
     * 合并实体类中的属性字段
     * @param object 实体类
     * @param json json实例
     */
    private static void mergeFields(Object object, JSONObject json) {
        JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(object));
        Set<String> keySet = jsonObject.keySet();
        for(String key : keySet) {
            String value = jsonObject.getString(key);
            setValue(key, value, json);
        }
    }

    /**
     * 根据维度组合对所有的实例进行分组
     * @param dimArray 维度数组
     * @param map 分组map
     * @param list 实例集合
     * @param <T> 实例范型
     */
    private static <T> void setValue(String[] dimArray, Map<String, List<Object>> map, List<T> list) {
        for(Object object : list) {
            String key = getKey(dimArray, JSON.parseObject(JSON.toJSONString(object)));
            if(map.containsKey(key)) {
                map.get(key).add(object);
            } else {
                List<Object> mapValueList = new ArrayList<>();
                mapValueList.add(object);
                map.put(key, mapValueList);
            }
        }
    }

    /**
     * 给json字段赋值
     * @param key json key
     * @param value json value
     * @param json json实例
     */
    private static void setValue(String key, String value, JSONObject json) {
        //排除掉空字符串
        if(StringUtils.isBlank(value)) {
            return;
        }

        if(!json.containsKey(key)) {
            json.put(key, value);
        } else {
            //排除掉指标值=0的数据
            if(!json.getString(key).equalsIgnoreCase(value) && Double.parseDouble(value) != 0) {
                json.put(key, value);
            }
        }
    }

}
