package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.hbase.client.AliHBaseUEConnection;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author xugf 2019-08-22
 */
public class HbaseUtil {
    private static final Logger logger = LoggerFactory.getLogger(HbaseUtil.class);
    //hbase连接
    private Connection connection = null;

    //zookeeper 连接地址 (企业标准版配置)
    private String zkList;

    //集群的连接地址 (性能增强型配置)
    private String endPoint;

    //默认用户名 (性能增强型配置)
    private String username;

    //默认密码 (性能增强型配置)
    private String password;

    //默认的namespace
    private String DEFAULT_NAMESPACE = "default";

    //默认列簇
    private String DEFAULT_COLUMN_FAMILY = "cf";

    //批量写入hbase的记录数
    private static final int saveBatchSize = 100;

    public void setZkList(String zkList) {
        this.zkList = zkList;
    }

    public void setEndPoint(String endPoint) {
        this.endPoint = endPoint;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    private Connection getConnection() {
        if (connection != null && !connection.isClosed()) {
            return connection;
        }
        return initConnection();
    }

    private synchronized Connection initConnection() {
        if (connection != null && !connection.isClosed()) {
            return connection;
        }
        try {
            //获得hbase connection连接池
            Configuration conf = HBaseConfiguration.create();
            if (StringUtils.isNotBlank(zkList)) {
                conf.set("hbase.zookeeper.quorum", zkList);
            } else {
                if (StringUtils.isBlank(endPoint)
                        || StringUtils.isBlank(username)
                        || StringUtils.isBlank(password)) {
                    throw new Exception("endPoint/username/password is not set, please set it.");
                }
                //将HBase底层Connection实现替换成HBase增强版专用的AliHBaseUEConnection
                conf.set("hbase.client.connection.impl", AliHBaseUEConnection.class.getName());
                //集群的连接地址
                conf.set("hbase.client.endpoint", endPoint);
                //设置用户名密码
                conf.set("hbase.client.username", username);
                conf.set("hbase.client.password", password);
            }

            connection = ConnectionFactory.createConnection(conf);
            return connection;
        } catch (Exception e) {
            logger.error("initConnection error", e);
        }
        return null;
    }

    private Table getTable(String nameSpace, String tableName) throws Exception {
        Connection connection = getConnection();
        if (connection == null) {
            logger.error("can't get hbase connection");
            throw new Exception("can't get hbase connection");
        }
        return connection.getTable(TableName.valueOf(nameSpace, tableName));
    }

    private void closeTable(Table table) {
        try {
            if (table != null) {
                table.close();
            }
        } catch (Exception e) {
            logger.error("closeTable error.", e);
        }
    }

    //单行单列
    private Get get(String rowKey, String family, String column) {
        Get get = new Get(rowKey.getBytes());
        get.addColumn(family.getBytes(), column.getBytes());
        return get;
    }

    //多行单列
    private List<Get> getList(List<String> rowKeyList, String family, String column) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family, column));
        }
        return getList;
    }

    //单行所有列
    private Get get(String rowKey, String family) {
        Get get = new Get(rowKey.getBytes());
        get.addFamily(family.getBytes());
        return get;
    }

    //多行所有列
    private List<Get> getList(List<String> rowKeyList, String family) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family));
        }
        return getList;
    }

    //单行部分列
    private Get get(String rowKey, String family, List<String> columns) {
        Get get = new Get(rowKey.getBytes());
        for (String column : columns) {
            get.addColumn(family.getBytes(), column.getBytes());
        }
        return get;
    }

    //多行部分列
    private List<Get> getList(List<String> rowKeyList, String family, List<String> columns) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family, columns));
        }
        return getList;
    }

    //多行不同列
    private List<Get> getList(Map<String, List<String>> rowKeyMap, String family) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            List<String> columns = rowKeyMap.get(rowKey);
            getList.add(get(rowKey, family, columns));
        }
        return getList;
    }

    //插入单行多列
    private Put put(String rowKey, String family, Map<String, String> columns) {
        Put put = new Put(rowKey.getBytes());

        for (String column : columns.keySet()) {
            String value = columns.get(column);
            put.addColumn(family.getBytes(), column.getBytes(), value.getBytes());
        }

        return put;

    }


    //插入多行多列
    private List<Put> putList(Map<String, Map<String, String>> rowKeyMap, String family) {
        List<Put> putList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            Map<String, String> columns = rowKeyMap.get(rowKey);
            if (MapUtils.isEmpty(columns)) {
                continue;
            }
            putList.add(put(rowKey, family, columns));

        }
        return putList;
    }

    /**
     * 判断rowkey是否存在
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String tableName, String rowKey) {
        return exists(DEFAULT_NAMESPACE, tableName, rowKey);
    }

    /**
     * 判断rowkey是否存在
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String nameSpace, String tableName, String rowKey) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey)) {
                logger.error("nameSpace/tableName/rowKey is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            Get get = new Get(rowKey.getBytes());
            return table.exists(get);
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列
     * @return Result
     */
    public Result getRow(String tableName, String rowKey, String column) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family, column)) {
                logger.error("nameSpace/tableName/rowKey/family/column is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family, column));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return Result
     */
    public Result getRow(String tableName, String rowKey) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 单行所有列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 单行部分列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列
     * @return Result
     */
    public Result getRow(String tableName, String rowKey, List<String> columns) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 单行部分列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   列
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(columns)) {
                logger.error("column list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family, columns));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 多行单列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param column     列
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList, String column) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, column);
    }


    /**
     * 多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param column     列
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family, column)) {
                logger.error("nameSpace/tableName/family/column is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList)) {
                logger.error("rowKey list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family, column));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 多行所有列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 多行所有列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList)) {
                logger.error("rowKey list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family));
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }


    /**
     * 查询列族下的多列数据(多条记录)
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return 结果数据
     */
    public Result[] getRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family) {
        Table table = null;
        try {
            table = getTable(nameSpace, tableName);
            if (rowKeyMap != null && rowKeyMap.size() > 0) {
                List<Get> getList = getList(rowKeyMap, family);
                return table.get(getList);
            }
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询列族下的多列数据(多条记录)
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return 结果数据
     */
    public Result[] getRowList(String tableName, Map<String, List<String>> rowKeyMap) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }


    /**
     * 多行部分列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param columns    列
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList, List<String> columns) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 多行部分列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param columns    列
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, List<String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList) || CollectionUtils.isEmpty(columns)) {
                logger.error("rowKeyList/columnList is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family, columns));
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 往hbase 表中插入多条记录（多列）
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     */
    public void insert(String tableName, Map<String, Map<String, String>> rowKeyMap) {
        insert(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 往hbase 表中插入多条记录（多列）
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     */
    public void insert(String nameSpace, String tableName, Map<String, Map<String, String>> rowKeyMap, String family) {
        if (MapUtils.isEmpty(rowKeyMap)) {
            return;
        }
        Table table = null;
        try {
            table = getTable(nameSpace, tableName);
            List<Put> allPutList = putList(rowKeyMap, family);

            batchInsert(allPutList, table);

        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
    }

    /**
     * 批量插入
     *
     * @param table      表
     * @param allPutList putlist
     * @throws IOException
     */
    private void batchInsert(List<Put> allPutList, Table table) throws IOException {
        if (allPutList.size() <= saveBatchSize) {
            insert(allPutList, table);
        } else {
            List<Put> putList = new ArrayList<>();
            for (Put put : allPutList) {
                putList.add(put);
                //批量写入数据到hbase中
                if (putList.size() >= saveBatchSize) {
                    insert(putList, table);
                    putList.clear();
                }
            }
            //将剩余数据写入hbase中
            if (putList.size() > 0) {
                insert(putList, table);
            }
        }
    }

    /**
     * 批量插入
     *
     * @param allPutList putlist
     * @param table      表
     * @throws IOException
     */
    private void insert(List<Put> allPutList, Table table) throws IOException {
        table.put(allPutList);
    }


}
