package cn.com.duiba.bigdata.common.biz.utils;

import cn.com.duiba.bigdata.common.biz.entity.redis.KeyIndex;
import cn.com.duiba.bigdata.common.biz.enums.DimStatisticsIndexEnum;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.util.Bytes;
import org.joda.time.DateTime;

import java.math.BigDecimal;
import java.util.*;

@Slf4j
public class BigdataUtil {

    public static boolean numberCheck(String str) {
        return !StringUtils.isBlank(str) && str.matches("[0-9]*");
    }

    //解析json数据
    public static JSONObject parseJson(Object content) {
        JSONObject jsonObject = null;
        if (content == null) {
            return null;
        }
        if (content instanceof String) {
            String json = (String) content;
            jsonObject = JSON.parseObject(json);
        } else if (content instanceof JSONObject) {
            jsonObject = (JSONObject) content;
        }
        return jsonObject;
    }

    public static String getString(JSONObject object, String key) {
        if (object == null) {
            return null;
        }
        String str = object.getString(key);
        if (StringUtils.isBlank(str)) {
            return null;
        }
        if ("null".equalsIgnoreCase(str)) {
            return null;
        }
        return str;
    }

    //获取hbase rowkey
    public static String getMD5HbaseRowkey(String... strs) {
        String key = getJoinStr(strs);
        return StringUtils.isBlank(key) ? null : MD5Util.computeMD5(key).substring(0, 4) + "-" + key;
    }

    public static String getJoinStr(Object... obj) {
        if (obj.length == 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (Object str : obj) {
            sb.append(str).append("_");
        }
        return sb.toString().substring(0, sb.toString().length() - 1);
    }

    /**
     * 最近多少天/小时/分钟的时间列表
     *
     * @param pattern 时间格式
     * @param before  最近多久
     * @return
     */
    public static List<String> getTimeList(String pattern, int before) {
        DateTime dateTime = new DateTime();
        List<String> timeList = new ArrayList<>();
        for (int i = 0; i < before; i++) {
            if ("yyyyMMdd".equals(pattern)) {
                timeList.add(dateTime.plusDays(-i).toString(pattern));
            } else if ("yyyyMMddHH".equals(pattern)) {
                timeList.add(dateTime.plusHours(-i).toString(pattern));
            } else if ("yyyyMMddHHmm".equals(pattern)) {
                timeList.add(dateTime.plusMinutes(-i).toString(pattern));
            }
        }
        return timeList;
    }

    /**
     * 获取map中后端转化的数据
     *
     * @param data
     * @return
     */
    public static <T> Map<String, Long> getBackendCntMap(Map<String, T> data) {
        Map<String, Long> map = new HashMap<>();
        try {
            for (String key : data.keySet()) {
                if (numberCheck(key)) {
                    map.put(key, Long.valueOf(data.get(key).toString()));
                }
            }
        } catch (NumberFormatException e) {
            log.error("get backend data error", e);
        }
        return map;
    }

    /**
     * 除法-保留n位小数
     *
     * @param dividend 分子
     * @param divisor  分母
     * @param digit    保留小数位数
     * @return 结果字符串
     */
    public static <T extends Number> String division(T dividend, T divisor, int digit) {
        String result = "0";
        try {
            if (dividend != null && divisor != null) {
                double divisorDouble = divisor.doubleValue();
                double dividendDouble = dividend.doubleValue();
                if (divisorDouble != 0) {
                    result = new BigDecimal(dividendDouble / divisorDouble).setScale(digit, BigDecimal.ROUND_HALF_UP).stripTrailingZeros().toPlainString();
                }
            }
        } catch (Exception e) {
            log.error("division error", e);
        }
        return result;
    }

    /**
     * 获取实时数仓rediskey
     *
     * @param prefix 前缀
     * @param suffix 时间后缀
     * @param array  key维度
     * @return key
     */
    public static String getBigdataRedisKey(String prefix, String suffix, Object... array) {
        //redis前缀不为空
        if (StringUtils.isAnyBlank(prefix)) {
            log.error("GetRedisKeyUDF prefix is null");
            return null;
        }

        //参数合法性校验
        if (array == null || array.length == 0) {
            log.error("GetRedisKeyUDF eval, please set dim and value");
            return null;
        }
        if (array.length % 2 != 0) {
            log.error("GetRedisKeyUDF eval, array.length % 2 != 0");
            return null;
        }

        List<KeyIndex> list = new ArrayList<>();
        for (int i = 0; i < array.length; i = i + 2) {
            String key = null;
            String value = null;
            if (array[i] != null) {
                key = array[i].toString();
            }
            if (array[i + 1] != null) {
                value = array[i + 1].toString();
            }
            if (StringUtils.isAnyBlank(key, value)) {
                return null;
            }
            Integer index = DimStatisticsIndexEnum.INSTANCE.getDimIndex(key);

            if (index == null) {
                log.error("key = {} is error, please check it.", key);
                return null;
            }
            KeyIndex keyIndex = new KeyIndex(key, value, index);
            list.add(keyIndex);
        }

        //根据dim index排序(升序)
        Collections.sort(list);

        //组装redis key
        StringBuilder sb = new StringBuilder();
        sb.append(prefix.toUpperCase()).append("_");
        for (KeyIndex info : list) {
            sb.append(info.getValue()).append("_");
        }
        if (StringUtils.isNotBlank(suffix)) {
            sb.append(suffix);
        }

        return sb.toString();
    }

    public static String getHbaseDataToString(Result result, String family, String column) {
        String value = "";
        if (result != null && result.containsColumn(family.getBytes(), column.getBytes())) {
            value = Bytes.toString(result.getValue(family.getBytes(), column.getBytes()));
        }
        return value;
    }

    public static String reverse(String str) {
        return new StringBuilder(str).reverse().toString();
    }

    public static Map<String, Map<String, String>> getHbaseStringDataMap(Map<String, List<String>> hbaseRowkeyMap, Result[] results, String family) {
        Map<String, Map<String, String>> map = new HashMap<>();
        if (MapUtils.isEmpty(hbaseRowkeyMap) || results == null) {
            return map;
        }
        for (Result result : results) {
            if (result != null && !result.isEmpty()) {
                Map<String, String> columnMap = new HashMap<>();
                String rowKey = Bytes.toString(result.getRow());
                List<String> columnList = hbaseRowkeyMap.get(rowKey);
                for (String column : columnList) {
                    putHbaseColumnData(columnMap, column, getHbaseDataToString(result, family, column));
                }
                map.put(rowKey, columnMap);
            }

        }
        return map;
    }


    public static void putHbaseColumnData(Map<String, String> map, String column, String value) {
        if (StringUtils.isNotBlank(value)) {
            map.put(column, value);
        }
    }

    /**
     * 解析url字段
     *
     * @param str
     * @param key
     * @return
     */
    public static String parseUrlParam(String str, String key) {
        if (StringUtils.isNotBlank(str)) {
            if (!str.startsWith(key)) {
                key = "&" + key;
            }
            if (str.contains(key)) {
                str = str.substring(str.indexOf(key) + key.length());
                return str.substring(0, str.contains("&") ? str.indexOf("&") : str.length());
            }
        }
        return null;
    }


    public static Map<String, Map<String, Long>> getHbaseLongDataMap(List<String> rowkeyList, Result[] results) {
        if (rowkeyList != null && rowkeyList.size() > 0) {
            Map<String, Map<String, Long>> map = new HashMap<>();
            for (int i = 0; i < rowkeyList.size(); i++) {
                Map<String, Long> columnMap = new HashMap<>();
                String rowkey = rowkeyList.get(i);
                Result result = results[i];
                if (result != null && !result.isEmpty()) {
                    Cell[] cells = result.rawCells();
                    for (Cell cell : cells) {
                        String column = Bytes.toString(cell.getQualifierArray(), cell.getQualifierOffset(), cell.getQualifierLength());
                        Long value = Bytes.toLong(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
                        columnMap.put(column, value);
                    }
                    map.put(rowkey, columnMap);
                }
            }
            return map;
        }
        return null;
    }


    public static Map<String, Map<String, String>> getHbaseStringDataMap(List<String> rowkeyList, Result[] results) {
        if (rowkeyList != null && rowkeyList.size() > 0) {
            Map<String, Map<String, String>> map = new HashMap<>();
            for (int i = 0; i < rowkeyList.size(); i++) {
                Map<String, String> columnMap = new HashMap<>();
                String rowkey = rowkeyList.get(i);
                Result result = results[i];
                if (result != null && !result.isEmpty()) {
                    Cell[] cells = result.rawCells();
                    for (Cell cell : cells) {
                        String column = Bytes.toString(cell.getQualifierArray(), cell.getQualifierOffset(), cell.getQualifierLength());
                        String value = Bytes.toString(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
                        putHbaseColumnData(columnMap, column, value);
                    }
                    map.put(rowkey, columnMap);
                }
            }
            return map;
        }
        return null;
    }

    /**
     * 求平均数
     *
     * @param set  数据集
     * @param skip 是否跳过0值
     * @param <T>  数据类型
     * @return 平均数
     */
    public static <T extends Number> Double avg(List<T> set, boolean skip) {
        int i = 0;
        double result = 0d;
        for (Number num : set) {
            if (num == null) {
                continue;
            }
            double value = num.doubleValue();

            if (skip && value == 0) {
                continue;
            }
            i = i + 1;
            result += value;
        }
        if (i != 0) {
            return result / i;
        }
        return 0d;

    }
}
