package cn.com.duiba.bigdata.common.biz.utils;

import cn.com.duiba.bigdata.common.biz.entity.redis.KeyIndex;
import cn.com.duiba.bigdata.common.biz.enums.DimStatisticsIndexEnum;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;

import java.math.BigDecimal;
import java.util.*;

@Slf4j
public class BigdataUtil {

    public static boolean numberCheck(String str) {
        return !StringUtils.isBlank(str) && str.matches("[0-9]*");
    }

    /**
     * 组装timekey
     *
     * @param key
     * @param timeList
     * @return
     */
    public static List<String> getTimeKeyList(String key, List<String> timeList) {

        List<String> keyList = new ArrayList<>();
        for (String time : timeList) {
            keyList.add(key + time);
        }
        return keyList;
    }

    //解析json数据
    public static JSONObject parseJson(Object content) {
        JSONObject jsonObject = null;
        if (content == null) {
            return null;
        }
        if (content instanceof String) {
            String json = (String) content;
            jsonObject = JSON.parseObject(json);
        } else if (content instanceof JSONObject) {
            jsonObject = (JSONObject) content;
        }
        return jsonObject;
    }

    public static String getString(JSONObject object, String key) {
        if (object == null) {
            return null;
        }
        String str = object.getString(key);
        if (StringUtils.isBlank(str)) {
            return null;
        }
        if ("null".equalsIgnoreCase(str)) {
            return null;
        }
        return str;
    }

    //获取hbase rowkey
    public static String getMD5HbaseRowkey(Object... obj) {
        String key = getJoinStr(obj);
        return StringUtils.isBlank(key) ? null : MD5Util.computeMD5(key).substring(0, 4) + "-" + key;
    }

    /**
     * 字符串拼接用下划线隔开
     *
     * @param obj
     * @return
     */
    public static String getJoinStr(Object... obj) {
        return getSepJoinStr("_", obj);
    }

    /**
     * 字符串拼接用下划线隔开
     *
     * @param obj
     * @return
     */
    public static String getSepJoinStr(String sep, Object... obj) {
        if (obj.length == 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (Object str : obj) {
            sb.append(str).append(sep);
        }
        return sb.delete(sb.length() - sep.length(), sb.length()).toString();
    }


    /**
     * 字符串拼接，没有下划线
     *
     * @param strs
     * @return
     */
    public static String getStr(String... strs) {
        StringBuilder sb = new StringBuilder();
        for (String str : strs) {
            sb.append(str);
        }
        return sb.toString();
    }

    /**
     * map数据融合 key field value
     *
     * @param map
     * @param rowkeyColumnMap
     */
    public static void mergeRowKeyMap(Map<String, String> map, Map<String, Map<String, Long>> rowkeyColumnMap) {
        for (String rowkey : map.keySet()) {
            JSONObject newJson = JSON.parseObject(map.get(rowkey));
            if (rowkeyColumnMap.containsKey(rowkey)) {
                Map<String, Long> oldColumnMap = rowkeyColumnMap.get(rowkey);
                mergeMap(newJson, oldColumnMap);
            } else {
                Map<String, Long> newColumnMap = new HashMap<>();
                for (String key : newJson.keySet()) {
                    Long value = newJson.getLongValue(key);
                    newColumnMap.put(key, value);
                }
                rowkeyColumnMap.put(rowkey, newColumnMap);
            }
        }
    }

    /**
     * 融合数据
     *
     * @param newJson
     * @param oldColumnMap
     */
    private static void mergeMap(JSONObject newJson, Map<String, Long> oldColumnMap) {
        for (String key : newJson.keySet()) {
            Long value = newJson.getLongValue(key);
            if (oldColumnMap.containsKey(key)) {
                oldColumnMap.put(key, oldColumnMap.get(key) + value);
            } else {
                oldColumnMap.put(key, value);
            }
        }
    }

    public static List<String> getTimeList(String type, String pattern, int number, Date date) {
        List<String> list = new ArrayList<>();
        Calendar calendar = Calendar.getInstance();
        for (int i = 0; i < number; i++) {
            calendar.setTime(date);
            if ("day".equals(type)) {
                calendar.add(Calendar.DAY_OF_MONTH, -i);
            } else if ("hour".equals(type)) {
                calendar.add(Calendar.HOUR_OF_DAY, -i);
            } else if ("minute".equals(type)) {
                calendar.add(Calendar.MINUTE, -i);
            }
            list.add(DateFormatUtil.format(pattern, calendar.getTime()));
        }
        return list;
    }

    /**
     * 最近多少天/小时/分钟的时间列表
     *
     * @param pattern 时间格式
     * @param before  最近多久
     * @return
     */
    public static List<String> getTimeList(String pattern, int before) {
        DateTime dateTime = new DateTime();
        List<String> timeList = new ArrayList<>();
        for (int i = 0; i < before; i++) {
            if ("yyyyMMdd".equals(pattern)) {
                timeList.add(dateTime.plusDays(-i).toString(pattern));
            } else if ("yyyyMMddHH".equals(pattern)) {
                timeList.add(dateTime.plusHours(-i).toString(pattern));
            } else if ("yyyyMMddHHmm".equals(pattern)) {
                timeList.add(dateTime.plusMinutes(-i).toString(pattern));
            }
        }
        return timeList;
    }

    /**
     * 返回n分钟内的所有时间字符串
     *
     * @param before
     * @return
     */
    public static String[] getTimeArray(int before, String pattern) {
        DateTime dateTime = new DateTime();
        String[] times = new String[before];
        for (int i = 1; i <= before; i++) {
            times[i - 1] = dateTime.plusMinutes(-i).toString(pattern);
        }
        return times;
    }

    /**
     * 获取map中后端转化的数据
     *
     * @param data
     * @return
     */
    public static <T> Map<String, Long> getBackendCntMap(Map<String, T> data) {
        Map<String, Long> map = new HashMap<>();
        try {
            for (String key : data.keySet()) {
                if (numberCheck(key)) {
                    map.put(key, Long.valueOf(data.get(key).toString()));
                }
            }
        } catch (NumberFormatException e) {
            log.error("get backend data error", e);
        }
        return map;
    }

    /**
     * 除法-保留n位小数
     *
     * @param dividend 分子
     * @param divisor  分母
     * @param digit    保留小数位数
     * @return 结果字符串
     */
    public static <T extends Number> String division(T dividend, T divisor, int digit) {
        String result = "0";
        try {
            if (dividend != null && divisor != null) {
                double divisorDouble = divisor.doubleValue();
                double dividendDouble = dividend.doubleValue();
                if (divisorDouble != 0) {
                    result = new BigDecimal(dividendDouble / divisorDouble).setScale(digit, BigDecimal.ROUND_HALF_UP).stripTrailingZeros().toPlainString();
                }
            }
        } catch (Exception e) {
            log.error("division error", e);
        }
        return result;
    }

    /**
     * 加法 反射用
     *
     * @param v1
     * @param v2
     * @param clazz
     * @return
     */
    public static Number add(Object v1, Object v2, Class<?> clazz) {
        if (v1 == null && v2 == null) {
            return null;
        } else if (v1 == null) {
            return (Number) v2;
        } else if (v2 == null) {
            return (Number) v1;
        }

        if (Double.class.isAssignableFrom(clazz)) {
            return Double.valueOf(v1.toString()) + Double.valueOf(v2.toString());
        }

        if (Integer.class.isAssignableFrom(clazz)) {
            return Integer.valueOf(v1.toString()) + Integer.valueOf(v2.toString());
        }

        if (Long.class.isAssignableFrom(clazz)) {
            return Long.valueOf(v1.toString()) + Long.valueOf(v2.toString());
        }

        return null;
    }

    /**
     * 获取实时数仓rediskey
     *
     * @param prefix 前缀
     * @param suffix 时间后缀
     * @param array  key维度
     * @return key
     */
    public static String getBigdataRedisKey(String prefix, String suffix, Object... array) {
        //redis前缀和后缀不为空
        if (StringUtils.isBlank(prefix)) {
            log.error("GetRedisKeyUDF prefix is null");
            return null;
        }

        //全局维度
        if (array == null || array.length == 0) {
            return prefix + "_" + suffix;
        }

        //array个数校验
        if (array.length % 2 != 0) {
            log.error("GetRedisKeyUDF eval, array.length % 2 != 0");
            return null;
        }

        List<KeyIndex> list = new ArrayList<>();
        for (int i = 0; i < array.length; i = i + 2) {
            String key = null;
            String value = null;
            if (array[i] != null) {
                key = array[i].toString();
            }
            if (array[i + 1] != null) {
                value = array[i + 1].toString();
            }
            if (StringUtils.isAnyBlank(key, value)) {
                return null;
            }
            Integer index = DimStatisticsIndexEnum.INSTANCE.getDimIndex(key);

            if (index == null) {
                log.error("key = {} is error, please check it.", key);
                return null;
            }
            KeyIndex keyIndex = new KeyIndex(key, value, index);
            list.add(keyIndex);
        }

        //根据dim index排序(升序)
        Collections.sort(list);

        //组装redis key
        StringBuilder sb = new StringBuilder();
        sb.append(prefix.toUpperCase()).append("_");
        for (KeyIndex info : list) {
            sb.append(info.getValue()).append("_");
        }
        if (StringUtils.isNotBlank(suffix)) {
            sb.append(suffix);
        }

        return sb.toString();
    }

    public static String reverse(String str) {
        return new StringBuilder(str).reverse().toString();
    }


    public static void putHbaseColumnData(Map<String, String> map, String column, String value) {
        if (StringUtils.isNotBlank(value)) {
            map.put(column, value);
        }
    }

    /**
     * 解析url字段
     *
     * @param str
     * @param key
     * @return
     */
    public static String parseUrlParam(String str, String key) {
        if (StringUtils.isNotBlank(str)) {
            if (!str.startsWith(key)) {
                key = "&" + key;
            }
            if (str.contains(key)) {
                str = str.substring(str.indexOf(key) + key.length());
                return str.substring(0, str.contains("&") ? str.indexOf("&") : str.length());
            }
        }
        return null;
    }


    /**
     * 求平均数
     *
     * @param set  数据集
     * @param skip 是否跳过0值
     * @param <T>  数据类型
     * @return 平均数
     */
    public static <T extends Number> Double avg(List<T> set, boolean skip) {
        int i = 0;
        double result = 0d;
        for (Number num : set) {
            if (num == null) {
                continue;
            }
            double value = num.doubleValue();

            if (skip && value == 0) {
                continue;
            }
            i = i + 1;
            result += value;
        }
        if (i != 0) {
            return result / i;
        }
        return 0d;

    }

    /**
     * 融合一批指标数据
     *
     * @param keyList
     * @param dataMap
     * @return
     */
    public static Map<String, Long> mergeData(List<String> keyList, Map<String, Map<String, String>> dataMap) {
        Map<String, Long> mergeMap = new HashMap<>();
        for (String key : keyList) {
            Map<String, String> data = dataMap.get(key);
            //取值
            for (String field : data.keySet()) {
                mergeMap.put(field, mergeMap.getOrDefault(field, 0L) + Long.valueOf(data.get(field)));
            }
        }

        return mergeMap;
    }

    /**
     * 合并redis或者hbase的结果数据
     *
     * @param resultMap 结果集合
     * @param <X>       key
     * @param <Y>       field
     * @return 合并后的集合
     */
    public static <X, Y> Map<X, Y> mergeResultMap(Map<X, Y>... resultMap) {
        if (resultMap == null || resultMap.length == 0) {
            return null;
        }
        Map<X, Y> dataMap = new HashMap<>();
        for (Map<X, Y> map : resultMap) {
            dataMap.putAll(map);
        }
        return dataMap;
    }

    /**
     * 将非空值放入map
     *
     * @param map   map
     * @param key   key
     * @param value value
     * @param <K,T> 键值类型
     */
    public static <K, T> void putMapNotNull(Map<K, T> map, K key, T value) {

        if (map == null || key == null || value == null) {
            return;
        }

        //字符串值
        if (value instanceof String && StringUtils.isBlank((String) value)) {
            return;
        }

        //集合值
        if (value instanceof Collection && CollectionUtils.isEmpty((Collection) value)) {
            return;
        }

        //map值
        if (value instanceof Map && MapUtils.isEmpty((Map) value)) {
            return;
        }
        map.put(key, value);
    }

    public static String getHbaseJoinStr(Object... obj) {
        return getSepJoinStr("-", obj);
    }

    /**
     * 校验imei格式是否合法
     *
     * @param imei
     * @return
     */
    public static boolean checkImei(String imei) {
        if (StringUtils.isBlank(imei) || imei.length() != 15 || !StringUtils.isNumeric(imei)) {
            return false;
        }
        String data = imei.substring(0, 14);
        int sum1 = 0;
        int sum2 = 0;
        for (int i = 0; i < data.length(); i++) {
            int num = data.charAt(i) - '0';     // ascii to num
            /*(1)将奇数位数字相加(从1开始计数)*/
            if (i % 2 == 0) {
                sum1 = sum1 + num;
            } else {
                /*(2)将偶数位数字分别乘以2,分别计算个位数和十位数之和(从1开始计数)*/
                int temp = num * 2;
                if (temp < 10) {
                    sum2 = sum2 + temp;
                } else {
                    sum2 = sum2 + temp + 1 - 10;
                }
            }
        }
        int total = sum1 + sum2;
        /*如果得出的数个位是0则校验位为0,否则为10减去个位数 */
        String resultStr;
        if (total % 10 == 0) {
            resultStr = data + "0";
        } else {
            resultStr = data + (10 - (total % 10)) + "";
        }
        if (imei.equals(resultStr)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * type asc升序列 null默认为降序
     *
     * @param map
     * @param n
     * @param type
     * @param <T>
     * @return
     */
    public static <K, T> List<K> getMapTopN(Map<K, T> map, Integer n, String type) {
        // map按value排倒序
        List<K> keyList = new ArrayList<>(map.keySet());
        //降序
        Collections.sort(keyList, new ByValueComparator(map));

        //升序
        if ("asc".equals(type)) {
            Collections.reverse(keyList);
        }
        return searchList(keyList, n);
    }

    private static <K> List<K> searchList(List<K> keyList, Integer n) {
        if (n == null || n == 0) {
            return keyList;
        }

        List<K> valueList = new ArrayList<>();
        if (keyList.size() <= n) {
            valueList.addAll(keyList);
        } else {
            valueList.addAll(keyList.subList(0, n));
        }
        return valueList;
    }

    /**
     * 将json字符串转化成map，可以任意指定map中key和value的类型
     *
     * @param jsonStr    json字符串
     * @param keyClazz   map中的key类型
     * @param valueClazz map中的value类型
     * @param <K>        key泛型
     * @param <V>        value泛型
     * @return map
     */
    public static <K, V> Map<K, V> jsonToMap(String jsonStr, Class<K> keyClazz, Class<V> valueClazz) {
        Map<K, V> map = new HashMap<>();
        JSONObject json = JSON.parseObject(jsonStr);
        for (String key : json.keySet()) {
            String value = json.getString(key);
            if (StringUtils.isBlank(value)) {
                continue;
            }

            map.put(getValue(key, keyClazz), getValue(value, valueClazz));
        }

        return map;
    }

    /**
     * map格式转换，可以任意指定map中key和value的类型
     *
     * @param srcMap     原始map集合
     * @param keyClazz   map中的key类型
     * @param valueClazz map中的value类型
     * @param <K>        key泛型
     * @param <V>        value泛型
     * @return map
     */
    public static <X, Y, K, V> Map<K, V> mapToMap(Map<X, Y> srcMap, Class<K> keyClazz, Class<V> valueClazz) {
        Map<K, V> map = new HashMap<>();
        for (Map.Entry<X, Y> entry : srcMap.entrySet()) {
            Object key = entry.getKey();
            Object value = entry.getValue();
            if (key == null || value == null) {
                continue;
            }

            map.put(getValue(key.toString(), keyClazz), getValue(value.toString(), valueClazz));
        }

        return map;
    }

    private static <T> T getValue(String value, Class<T> clazz) {
        if (String.class.isAssignableFrom(clazz)) {
            return (T) value;
        } else if (Integer.class.isAssignableFrom(clazz)) {
            return (T) Integer.valueOf(value);
        } else if (Long.class.isAssignableFrom(clazz)) {
            return (T) Long.valueOf(value);
        } else if (Double.class.isAssignableFrom(clazz)) {
            return (T) Double.valueOf(value);
        } else if (Float.class.isAssignableFrom(clazz)) {
            return (T) Float.valueOf(value);
        }

        return null;
    }

    /**
     * 获取 Optional 结果值
     *
     * @param opt Optional
     * @param <T> 返回对象
     * @return Optional封装的对象
     */
    public static <T> T getOptionalResult(Optional<T> opt) {
        return opt.isPresent() ? opt.get() : null;
    }


    public static Long strToLong(String t) {
        if (StringUtils.isBlank(t)) {
            return 0L;
        }
        return Long.valueOf(t);
    }

    public static Long getLongValue(Long v) {
        if (v == null) {
            return 0L;
        }
        return v;
    }


}
