package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

/**
 * @author xugf 2020-04-21
 * http连接工具类，短连接
 */
@Slf4j
public class HttpClientUtil {

    /**
     * 获取http返回的结果数据
     *
     * @param url     http 连接地址
     * @param body    post请求时的body体
     * @param timeout 超时时间
     * @return http请求结果数据
     */
    public static String getHttpResponse(String url, StringEntity body, int timeout) {
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {

            //构建http请求
            HttpUriRequest httpUriRequest = getHttpUriRequest(url, body, timeout);

            //执行请求
            CloseableHttpResponse response = httpClient.execute(httpUriRequest);

            return EntityUtils.toString(response.getEntity(), "UTF-8");
        } catch (Exception e) {
            log.error("getHttpResponse error,", e);
        }

        return null;
    }

    /**
     * 获取http返回的结果数据
     *
     * @param url     http 连接地址
     * @param head    head
     * @param body    post请求时的body体
     * @param timeout 超时时间
     * @return http请求结果数据
     */
    public static String getHttpResponse(String url, JSONObject head, HttpEntity body, int timeout) {
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {

            //构建http请求
            HttpUriRequest httpUriRequest = getHttpUriRequest(url, body, timeout);

            if (head != null && head.size() > 0) {
                for (String key : head.keySet()) {
                    httpUriRequest.addHeader(key, head.getString(key));
                }
            }

            //执行请求
            CloseableHttpResponse response = httpClient.execute(httpUriRequest);

            return EntityUtils.toString(response.getEntity(), "UTF-8");
        } catch (Exception e) {
            log.error("getHttpResponse error,", e);
        }

        return null;
    }


    //构建http请求
    private static HttpUriRequest getHttpUriRequest(String url, HttpEntity body, int timeout) {
        if (body == null) {
            //http get请求
            return httpGet(url, timeout);
        } else {
            //http post请求
            return httpPost(url, body, timeout);
        }
    }

    //构建httpGet请求
    private static HttpGet httpGet(String url, int timeout) {
        HttpGet httpGet = new HttpGet(url);
        httpGet.setHeader("Content-Type", "application/json;charset=UTF-8");
        httpGet.setConfig(getRequestConfig(timeout));
        return httpGet;
    }

    //构建httpPost请求
    private static HttpPost httpPost(String url, HttpEntity entity, int timeout) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setHeader("Content-Type", "application/json;charset=UTF-8");
        httpPost.setEntity(entity);
        httpPost.setConfig(getRequestConfig(timeout));
        return httpPost;
    }

    //http请求配置信息
    private static RequestConfig getRequestConfig(int timeout) {
        return RequestConfig.custom()
                .setSocketTimeout(timeout)
                .setConnectTimeout(timeout)
                .setConnectionRequestTimeout(timeout)
                .build();
    }
}
