package cn.com.duiba.bigdata.common.biz.utils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.*;
import org.apache.hadoop.hbase.filter.ColumnPrefixFilter;
import org.apache.hadoop.hbase.filter.Filter;
import org.apache.hadoop.hbase.filter.FilterList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author xugf 2019-08-22
 */
public class HbaseUtil {
    private static final Logger logger = LoggerFactory.getLogger(HbaseUtil.class);
    //hbase连接
    private Connection connection = null;

    //zookeeper 连接地址
    private String zkList;

    //默认用户名 (性能增强型配置)
    private String username;

    //默认密码 (性能增强型配置)
    private String password;

    //默认的namespace
    private String DEFAULT_NAMESPACE = "default";

    //默认列簇
    private String DEFAULT_COLUMN_FAMILY = "cf";

    //批量写入hbase的记录数
    private static final int saveBatchSize = 100;

    public void setZkList(String zkList) {
        this.zkList = zkList;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    protected Connection getConnection() {
        if (connection != null && !connection.isClosed()) {
            return connection;
        }
        return initConnection();
    }

    private synchronized Connection initConnection() {
        if (connection != null && !connection.isClosed()) {
            return connection;
        }
        try {
            //获得hbase connection连接池
            Configuration conf = HBaseConfiguration.create();
            //设置zk连接地址
            conf.set("hbase.zookeeper.quorum", zkList);
            if (StringUtils.isNoneBlank(username, password)) {
                //设置用户名密码
                conf.set("hbase.client.username", username);
                conf.set("hbase.client.password", password);
            }

            conf.setBoolean("lindorm.client.prefetch.routecache", true);
            connection = ConnectionFactory.createConnection(conf);
            return connection;
        } catch (Exception e) {
            logger.error("initConnection error", e);
        }
        return null;
    }

    private Table getTable(String nameSpace, String tableName) throws Exception {
        Connection connection = getConnection();
        if (connection == null) {
            logger.error("can't get hbase connection");
            throw new Exception("can't get hbase connection");
        }
        return connection.getTable(TableName.valueOf(nameSpace, tableName));
    }

    private void closeTable(Table table) {
        try {
            if (table != null) {
                table.close();
            }
        } catch (Exception e) {
            logger.error("closeTable error.", e);
        }
    }

    //查询单行单列
    private Get get(String rowKey, String family, String column) {
        Get get = new Get(rowKey.getBytes());
        get.addColumn(family.getBytes(), column.getBytes());
        return get;
    }

    //查询单行部分列
    private Get get(String rowKey, String family, List<String> columns) {
        Get get = new Get(rowKey.getBytes());
        for (String column : columns) {
            get.addColumn(family.getBytes(), column.getBytes());
        }
        return get;
    }

    //查询单行所有列
    private Get get(String rowKey, String family) {
        Get get = new Get(rowKey.getBytes());
        get.addFamily(family.getBytes());
        return get;
    }

    //查询单行部分列（单条件过滤）
    private Scan scan(String rowKey, String family, String columnPrefix) {
        Scan scan = new Scan(rowKey.getBytes(), rowKey.getBytes());
        scan.addFamily(family.getBytes());
        scan.setFilter(new ColumnPrefixFilter(columnPrefix.getBytes()));
        return scan;
    }

    //查询单行部分列（多条件过滤）
    private Scan scan(String rowKey, String family, List<String> columnPrefixList) {
        Scan scan = new Scan(rowKey.getBytes(), rowKey.getBytes());
        scan.addFamily(family.getBytes());
        List<Filter> filterList = new ArrayList<>();
        for(String columnPrefix : columnPrefixList) {
            filterList.add(new ColumnPrefixFilter(columnPrefix.getBytes()));
        }
        Filter filter = new FilterList(FilterList.Operator.MUST_PASS_ONE, filterList);
        scan.setFilter(filter);
        return scan;
    }

    //查询多行单列
    private List<Get> getList(List<String> rowKeyList, String family, String column) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family, column));
        }
        return getList;
    }

    //查询多行部分列
    private List<Get> getList(List<String> rowKeyList, String family, List<String> columns) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family, columns));
        }
        return getList;
    }

    //查询多行所有列
    private List<Get> getList(List<String> rowKeyList, String family) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(get(rowKey, family));
        }
        return getList;
    }

    //查询多行所有列
    private List<Get> getList(List<String> rowKeyList) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            getList.add(new Get(rowKey.getBytes()));
        }
        return getList;
    }

    //查询多行不同列
    private List<Get> getList(Map<String, List<String>> rowKeyMap, String family) {
        List<Get> getList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            List<String> columns = rowKeyMap.get(rowKey);
            getList.add(get(rowKey, family, columns));
        }
        return getList;
    }

    //插入单行单列
    private Put put(String rowKey, String family, String column, String value) {
        Put put = new Put(rowKey.getBytes());
        put.addColumn(family.getBytes(), column.getBytes(), value.getBytes());
        return put;
    }

    //插入单行多列
    private Put put(String rowKey, String family, Map<String, String> columns) {
        Put put = new Put(rowKey.getBytes());
        for (String column : columns.keySet()) {
            String value = columns.get(column);
            put.addColumn(family.getBytes(), column.getBytes(), value.getBytes());
        }
        return put;
    }

    //插入多行单列
    private List<Put> putList(List<String> rowKeyList, String family, String column, String value) {
        List<Put> putList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            putList.add(put(rowKey, family, column, value));
        }
        return putList;
    }

    //插入多行同列
    private List<Put> putList(Map<String, String> rowKeyMap, String family, String column) {
        List<Put> putList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            putList.add(put(rowKey, family, column, rowKeyMap.get(rowKey)));
        }
        return putList;
    }

    //插入多行多列
    private List<Put> putList(Map<String, Map<String, String>> rowKeyMap, String family) {
        List<Put> putList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            Map<String, String> columns = rowKeyMap.get(rowKey);
            putList.add(put(rowKey, family, columns));
        }
        return putList;
    }

    //删除单行单列
    private Delete delete(String rowKey, String family, String column) {
        Delete delete = new Delete(rowKey.getBytes());
        delete.addColumn(family.getBytes(), column.getBytes());
        return delete;
    }

    //删除单行多列
    private Delete delete(String rowKey, String family, List<String> columns) {
        Delete delete = new Delete(rowKey.getBytes());
        for (String column : columns) {
            delete.addColumn(family.getBytes(), column.getBytes());
        }
        return delete;
    }

    //删除整行
    private Delete delete(String rowKey, String family) {
        Delete delete = new Delete(rowKey.getBytes());
        delete.addFamily(family.getBytes());
        return delete;
    }

    //删除多行单列
    private List<Delete> delete(List<String> rowKeyList, String family, String column) {
        List<Delete> deleteList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            deleteList.add(delete(rowKey, family, column));
        }
        return deleteList;
    }

    //删除多行多列
    private List<Delete> delete(Map<String, List<String>> rowKeyMap, String family) {
        List<Delete> deleteList = new ArrayList<>();
        for (String rowKey : rowKeyMap.keySet()) {
            List<String> columns = rowKeyMap.get(rowKey);
            deleteList.add(delete(rowKey, family, columns));
        }
        return deleteList;
    }

    //删除多行所有列
    private List<Delete> delete(List<String> rowKeyList, String family) {
        List<Delete> deleteList = new ArrayList<>();
        for (String rowKey : rowKeyList) {
            deleteList.add(delete(rowKey, family));
        }
        return deleteList;
    }


    /**
     * 判断rowkey是否存在
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String tableName, String rowKey) {
        return exists(DEFAULT_NAMESPACE, tableName, rowKey);
    }

    /**
     * 判断rowkey是否存在
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String nameSpace, String tableName, String rowKey) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey)) {
                logger.error("nameSpace/tableName/rowKey is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            Get get = new Get(rowKey.getBytes());
            return table.exists(get);
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 判断rowkey集合是否存在
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键
     * @return rowkey是否存在，存在=true
     */
    public boolean[] existsAll(String nameSpace, String tableName, List<String> rowKeyList) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName)) {
                logger.error("nameSpace/tableName is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList)) {
                logger.error("rowKey list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            List<Get> getList = getList(rowKeyList);
            return table.existsAll(getList);
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列
     * @return Result
     */
    public Result getRow(String tableName, String rowKey, String column) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 查询单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family, column)) {
                logger.error("nameSpace/tableName/rowKey/family/column is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family, column));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询单行部分列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列
     * @return Result
     */
    public Result getRow(String tableName, String rowKey, List<String> columns) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 查询单行部分列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   列
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(columns)) {
                logger.error("column list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family, columns));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return Result
     */
    public Result getRow(String tableName, String rowKey) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 查询单行所有列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @return Result
     */
    public Result getRow(String nameSpace, String tableName, String rowKey, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(get(rowKey, family));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询单行部分列（单条件过滤）
     *
     * @param tableName 表名
     * @param rowKey 主键
     * @param columnPrefix 列的前缀
     * @return Result
     */
    public Result getScanner(String tableName, String rowKey, String columnPrefix) {
        return getScanner(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columnPrefix);
    }

    /**
     * 查询单行部分列（单条件过滤）
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey 主键
     * @param family 列族
     * @param columnPrefix 列的前缀
     * @return Result
     */
    public Result getScanner(String nameSpace, String tableName, String rowKey, String family, String columnPrefix) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family, columnPrefix)) {
                logger.error("nameSpace/tableName/rowKey/family/columnPrefix is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.getScanner(scan(rowKey, family, columnPrefix)).next();
        } catch (Exception e) {
            logger.error("getScanner error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询单行部分列（多条件过滤）
     *
     * @param tableName 表名
     * @param rowKey 表名
     * @param columnPrefixList 多个列的前缀
     * @return Result
     */
    public Result getScanner(String tableName, String rowKey, List<String> columnPrefixList) {
        return getScanner(DEFAULT_NAMESPACE, tableName, tableName, DEFAULT_COLUMN_FAMILY, columnPrefixList);
    }

    /**
     * 查询单行部分列（多条件过滤）
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey 主键
     * @param family 列族
     * @param columnPrefixList 多个列的前缀
     * @return Result
     */
    public Result getScanner(String nameSpace, String tableName, String rowKey, String family, List<String> columnPrefixList) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family/columnPrefix is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(columnPrefixList)) {
                logger.error("columnPrefix list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.getScanner(scan(rowKey, family, columnPrefixList)).next();
        } catch (Exception e) {
            logger.error("getScanner error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询多行单列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param column     列
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList, String column) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, column);
    }


    /**
     * 查询多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param column     列
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family, column)) {
                logger.error("nameSpace/tableName/family/column is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList)) {
                logger.error("rowKey list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family, column));
        } catch (Exception e) {
            logger.error("getRow error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询多行部分列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param columns    列
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList, List<String> columns) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 查询多行部分列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param columns    列
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, List<String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList) || CollectionUtils.isEmpty(columns)) {
                logger.error("rowKeyList/columnList is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family, columns));
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询多行所有列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @return Result数组
     */
    public Result[] getRowList(String tableName, List<String> rowKeyList) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 查询多行所有列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @return Result数组
     */
    public Result[] getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return null;
            }
            if (CollectionUtils.isEmpty(rowKeyList)) {
                logger.error("rowKey list is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyList, family));
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 查询多行不同列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return 结果数据
     */
    public Result[] getRowList(String tableName, Map<String, List<String>> rowKeyMap) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 查询多行不同列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return 结果数据
     */
    public Result[] getRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return null;
            }
            if (rowKeyMap == null || rowKeyMap.size() == 0) {
                logger.error("rowKeyMap is null, please check it.");
                return null;
            }
            table = getTable(nameSpace, tableName);
            return table.get(getList(rowKeyMap, family));
        } catch (Exception e) {
            logger.error("getRowList error", e);
        } finally {
            closeTable(table);
        }
        return null;
    }

    /**
     * 插入单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列名
     * @param value     列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, String rowKey, String column, String value) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column, value);
    }

    /**
     * 插入单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列名
     * @param value     列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, String rowKey, String family, String column, String value) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family, column, value)) {
                logger.error("nameSpace/tableName/rowKey/family/column/value is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.put(put(rowKey, family, column, value));
            return true;
        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 插入单行多列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, String rowKey, Map<String, String> columns) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 插入单行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, String rowKey, String family, Map<String, String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return false;
            }
            if (columns == null || columns.size() == 0) {
                logger.error("columnMap is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.put(put(rowKey, family, columns));
            return true;
        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 插入多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param column     列名
     * @param value      列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, List<String> rowkeyList, String column, String value) {
        return insert(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY, column, value);
    }

    /**
     * 插入多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @param column     列名
     * @param value      列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, List<String> rowkeyList, String family, String column, String value) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family, column, value)) {
                logger.error("nameSpace/tableName/family/column/value is null, please check it.");
                return false;
            }
            if (CollectionUtils.isEmpty(rowkeyList)) {
                logger.error("rowkeyList is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            List<Put> allPutList = putList(rowkeyList, family, column, value);
            batchInsert(allPutList, table);
            return true;
        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 插入多行同列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 主键集合 对应的value
     * @param family    列族
     * @param column    列名
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, Map<String, String> rowKeyMap, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family, column)) {
                logger.error("nameSpace/tableName/family/column/value is null, please check it.");
                return false;
            }
            if (MapUtils.isEmpty(rowKeyMap)) {
                logger.error("rowkeyList is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            List<Put> allPutList = putList(rowKeyMap, family, column);
            batchInsert(allPutList, table);
            return true;
        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 插入多行多列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, Map<String, Map<String, String>> rowKeyMap) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 插入多行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, Map<String, Map<String, String>> rowKeyMap, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return false;
            }
            if (rowKeyMap == null || rowKeyMap.size() == 0) {
                logger.error("rowKeyMap is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            List<Put> allPutList = putList(rowKeyMap, family);
            batchInsert(allPutList, table);
            return true;
        } catch (Exception e) {
            logger.error("insert error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 删除单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey, String column) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 删除单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family, column)) {
                logger.error("nameSpace/tableName/rowKey/family/column is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowKey, family, column));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 删除单行多列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey, List<String> columns) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 删除单行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   列集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return false;
            }
            if (CollectionUtils.isEmpty(columns)) {
                logger.error("columnMap is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowKey, family, columns));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 删除单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除单行所有列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, rowKey, family)) {
                logger.error("nameSpace/tableName/rowKey/family is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowKey, family));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 删除多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param column     列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, List<String> rowkeyList, String column) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 删除多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @param column     列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, List<String> rowkeyList, String family, String column) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family, column)) {
                logger.error("nameSpace/tableName/family/column is null, please check it.");
                return false;
            }
            if (CollectionUtils.isEmpty(rowkeyList)) {
                logger.error("rowkeyList is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowkeyList, family, column));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 插入多行多列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, Map<String, List<String>> rowKeyMap) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除多行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return false;
            }
            if (rowKeyMap == null || rowKeyMap.size() == 0) {
                logger.error("rowKeyMap is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowKeyMap, family));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 删除多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, List<String> rowkeyList) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, List<String> rowkeyList, String family) {
        Table table = null;
        try {
            if (StringUtils.isAnyBlank(nameSpace, tableName, family)) {
                logger.error("nameSpace/tableName/family is null, please check it.");
                return false;
            }
            if (CollectionUtils.isEmpty(rowkeyList)) {
                logger.error("rowkeyList is null, please check it.");
                return false;
            }
            table = getTable(nameSpace, tableName);
            table.delete(delete(rowkeyList, family));
            return true;
        } catch (Exception e) {
            logger.error("delete error", e);
        } finally {
            closeTable(table);
        }
        return false;
    }

    /**
     * 批量插入
     *
     * @param table      表
     * @param allPutList putlist
     * @throws IOException
     */
    private void batchInsert(List<Put> allPutList, Table table) throws IOException {
        if (allPutList.size() <= saveBatchSize) {
            insert(allPutList, table);
        } else {
            List<Put> putList = new ArrayList<>();
            for (Put put : allPutList) {
                putList.add(put);
                //批量写入数据到hbase中
                if (putList.size() >= saveBatchSize) {
                    insert(putList, table);
                    putList.clear();
                }
            }
            //将剩余数据写入hbase中
            if (putList.size() > 0) {
                insert(putList, table);
            }
        }
    }

    /**
     * 批量插入
     *
     * @param allPutList putlist
     * @param table      表
     * @throws IOException
     */
    private void insert(List<Put> allPutList, Table table) throws IOException {
        table.put(allPutList);
    }


}
