package cn.com.duiba.bigdata.common.biz.utils;

import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;

import java.util.Properties;
import java.util.concurrent.atomic.AtomicInteger;

public class KafkaProducerUtil {

    //kafka发送客户端
    private Producer<String, String> producer = null;

    //实例引用计数
    private AtomicInteger refCount = new AtomicInteger();

    //构建kafka发送客户端
    public KafkaProducerUtil(String bootstrapServers) {
        Properties properties = getProperties(bootstrapServers);
        producer = new KafkaProducer<>(properties);
    }

    //sink引用数加1
    public void increment() {
        refCount.incrementAndGet();
    }

    //sink引用数减1
    private int decrement() {
        return refCount.decrementAndGet();
    }

    //判断kafka 连接是否已经被关闭
    public boolean isAlive() {
        return producer != null;
    }

    /**
     * 发送消息到kafka集群
     * @param topic topic名称
     * @param message 消息
     */
    public void sendMessage(String topic, String message) {
        producer.send(new ProducerRecord<>(topic, message));
    }

    /**
     * 在关闭连接前，将缓存中的消息发送到kafka集群上，防止数据丢失
     */
    public void flush() {
        producer.flush();
    }

    /**
     * 关闭kafka连接
     */
    public void close() {
        int count = decrement();
        if (count <= 0 && producer != null) {
            producer.close();
            producer = null;
        }
    }

    //kafka 发送客户端配置
    private Properties getProperties(String bootstrapServers) {
        Properties props = new Properties();
        //kafka集群地址
        props.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);

        //ack方式，all，会等所有的副本commit 最慢的方式,安全性最高，本次采用折中方案 "1"
        props.put(ProducerConfig.ACKS_CONFIG, "1");

        //延迟时间，如果buffer没满，比如设为100，即消息发送会多100ms的延迟
        props.put(ProducerConfig.LINGER_MS_CONFIG, 100);

        //buffer 大小为64K
        props.put(ProducerConfig.BATCH_SIZE_CONFIG, 64 * 1024);

        //设置压缩类型为gzip，gzip压缩率高，snappy压缩速度快
        props.put(ProducerConfig.COMPRESSION_TYPE_CONFIG, "gzip");

        //失败重试次数, 允许重试10次
        props.put(ProducerConfig.RETRIES_CONFIG, 10);

        //每次重试之前需要阻塞的时间, 允许阻塞500ms
        props.put(ProducerConfig.RETRY_BACKOFF_MS_CONFIG, 500);

        //key 序列化类型
        props.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringSerializer");

        //value 序列化类型
        props.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringSerializer");

        return props;
    }

}
