package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

/**
 * 合并druid查询出来的指标数据，方便业务开发
 *
 * @author xugf 2019-10-23
 */
public class DruidUtil {

    /**
     * 合并druid查询出的指标
     *
     * @param clazz      实例class
     * @param dimensions 维度组合，用 "," 分隔
     * @param dataList 多个指标对应的实例集合数组
     * @param <T>        实例对应的泛型
     * @return List<T>
     */
    public static <T> List<T> mergeList(Class<T> clazz, String dimensions, List<List<String>> dataList) {
        //返回的结果数组
        List<T> resultList = new ArrayList<>();

        if (StringUtils.isBlank(dimensions)) {
            //全局维度，只有一条数据
            resultList.add(mergeOne(clazz, dataList));
            return resultList;
        }

        //维度组合
        String[] dimArray = StringUtils.split(dimensions, ",");

        //LinkedHashMap 有顺序之分，配合order by 语义
        Map<String, List<String>> map = new LinkedHashMap<>();
        for (List<String> list : dataList) {
            if (CollectionUtils.isNotEmpty(list)) {
                setValue(dimArray, map, list);
            }
        }

        for (Map.Entry<String, List<String>> entry : map.entrySet()) {
            resultList.add(merge(clazz, entry.getValue()));
        }

        return resultList;
    }

    /**
     * 合并druid查询出的指标(单条记录)
     *
     * @param clazz      实例class
     * @param dataList 多个指标对应的实例集合数组
     * @param <T>        实例对应的泛型
     * @return 实例
     */
    private static <T> T mergeOne(Class<T> clazz, List<List<String>> dataList) {
        List<String> list = new ArrayList<>();
        for (List<String> arrayList : dataList) {
            if (CollectionUtils.isNotEmpty(arrayList)) {
                list.add(arrayList.get(0));
            }
        }

        return merge(clazz, list);
    }

    /**
     * 合并druid查询出的指标
     *
     * @param clazz   实例class
     * @param druidMetricList    druid指标数据集
     * @param <T>     实例对应的泛型
     * @return 实体类
     */
    private static <T> T merge(Class<T> clazz, List<String> druidMetricList) {
        JSONObject json = new JSONObject();
        for (String druidMetricValue : druidMetricList) {
            mergeFields(druidMetricValue, json);
        }
        return json.toJavaObject(clazz);
    }

    /**
     * 获取维度组合对应的key，用于指标关联
     *
     * @param dimArray   维度对应的数组
     * @param jsonObject 实例转成的jsonObject
     * @return key
     */
    private static String getKey(String[] dimArray, JSONObject jsonObject) {
        StringBuilder stringBuilder = new StringBuilder();
        for (String dimension : dimArray) {
            String value = jsonObject.getString(dimension.trim());
            value = StringUtils.isBlank(value) ? "0" : value;
            stringBuilder.append(value).append("-");
        }
        stringBuilder.deleteCharAt(stringBuilder.length() - 1);
        return stringBuilder.toString();
    }

    /**
     * 合并实体类中的属性字段
     *
     * @param druidMetricValue druid指标数据
     * @param json   json实例
     */
    private static void mergeFields(String druidMetricValue, JSONObject json) {
        JSONObject jsonObject = JSON.parseObject(druidMetricValue);
        Set<String> keySet = jsonObject.keySet();
        for (String key : keySet) {
            String value = jsonObject.getString(key);
            setValue(key, value, json);
        }
    }

    /**
     * 根据维度组合对所有的实例进行分组
     *
     * @param dimArray 维度数组
     * @param map      分组map
     * @param list     实例集合
     */
    private static void setValue(String[] dimArray, Map<String, List<String>> map, List<String> list) {
        for (String value : list) {
            String key = getKey(dimArray, JSON.parseObject(value));
            if (map.containsKey(key)) {
                map.get(key).add(value);
            } else {
                List<String> mapValueList = new ArrayList<>();
                mapValueList.add(value);
                map.put(key, mapValueList);
            }
        }
    }

    /**
     * 给json字段赋值
     *
     * @param key   json key
     * @param value json value
     * @param json  json实例
     */
    private static void setValue(String key, String value, JSONObject json) {
        //排除掉空字符串
        if (StringUtils.isBlank(value)) {
            return;
        }

        if (!json.containsKey(key)) {
            json.put(key, value);
        } else {
            //排除掉指标值=0的数据
            if (!json.getString(key).equalsIgnoreCase(value) && Double.parseDouble(value) != 0) {
                json.put(key, value);
            }
        }
    }

}
