package cn.com.duiba.bigdata.common.biz.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.concurrent.*;

@Slf4j
public class FutureTaskUtil {

    /**
     * 异步提交需要执行的任务
     *
     * @param executorService 线程池
     * @param callable        具体需要执行的任务
     * @param <T>             返回对象
     * @return FutureTask
     */
    public static <T> FutureTask<T> getFutureTask(ExecutorService executorService, Callable<T> callable) {
        FutureTask<T> task = new FutureTask<>(callable);
        executorService.submit(task);
        return task;
    }

    /**
     * 获取异步任务返回结果
     *
     * @param task    异步任务
     * @param timeout 超时时间
     * @param unit    时间单位
     * @param name    小类
     * @return 异步任务返回结果
     */
    public static <T> T getFutureTaskResult(FutureTask<T> task, long timeout, TimeUnit unit, String name) {
        try {
            if (task == null) {
                return null;
            }
            return task.get(timeout, unit);
        } catch (Exception e) {
            if (e instanceof TimeoutException) {
                if (StringUtils.isNotBlank(name)) {
                    //cat埋点，用于记录超时情况
                    BigdataCatUtil.logEvent("timeout", name);
                }
            } else if (e instanceof ExecutionException) {
                if (StringUtils.isNotBlank(name)) {
                    //cat埋点，用于记录超时情况
                    BigdataCatUtil.logEvent("executionException", name);
                }
            } else {
                log.error("getFutureTaskResult error, name = {} ", name, e);
            }
        }

        return null;
    }

    /**
     * 异步提交需要执行的任务，并等待返回结果
     *
     * @param callable              需要执行的任务
     * @param submitExecutorService 任务提交线程池
     * @param taskExecutorService   任务执行线程池
     * @param timeout               返回结果超时时间
     * @param name                  小类
     * @param <T>                   返回结果对象
     * @return FutureTask
     */
    public static <T> FutureTask<T> getFutureTask(Callable<T> callable, ExecutorService submitExecutorService,
                                                  ExecutorService taskExecutorService, long timeout, String name) {
        return getFutureTask(submitExecutorService,
                () -> getFutureTaskResult(getFutureTask(taskExecutorService, callable), timeout, TimeUnit.MILLISECONDS, name)
        );
    }

    /**
     * 异步提交需要执行的任务，并等待返回结果，ExecutorService.invokeAll实现
     *
     * @param <T>                 返回结果对象
     * @param callables           需要执行的任务
     * @param taskExecutorService 任务执行线程池
     * @param timeout             返回结果超时时间,单位毫秒
     * @return
     */
    public static <T> List<Future<T>> getFutureList(List<? extends Callable<T>> callables, ExecutorService taskExecutorService, long timeout) {

        try {
            if (CollectionUtils.isEmpty(callables)) {
                return null;
            }

            return taskExecutorService.invokeAll(callables, timeout, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            log.error("getFutureList with timeout error = ", e);
        }

        return null;
    }

    /**
     * 从Future获取结果
     *
     * @param future  FutureTask执行结果
     * @param clazz   返回数据类型
     * @param catName cat事件名称
     * @param <T>     返回数据类型
     * @return 指定的数据类型对象
     */
    public static <T> T getFutureValue(Future<Object> future, Class<T> clazz, String catName) {
        try {

            if (future == null) {
                return null;
            }

            //任务被中断，直接返回null
            if (future.isCancelled()) {
                //cat埋点，用于记录超时情况
                if (StringUtils.isNotEmpty(catName)) {
                    BigdataCatUtil.logEvent("callableTimeout", catName);
                }

                return null;
            }

            return (T) future.get();
        } catch (Exception e) {
            log.error("getFutureValue error = ", e);
        }

        return null;
    }


}
