package cn.com.duiba.bigdata.common.biz.utils;

import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.SessionCallback;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author xugf 2019-08-22
 */
public class RedisUtil {

    private final int piplineBatch = 1000;

    private StringRedisTemplate stringRedisTemplate;

    public RedisUtil() {
    }

    public RedisUtil(StringRedisTemplate stringRedisTemplate) {
        this.stringRedisTemplate = stringRedisTemplate;
    }

    public StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }

    public void setStringRedisTemplate(StringRedisTemplate stringRedisTemplate) {
        this.stringRedisTemplate = stringRedisTemplate;
    }

    public boolean exist(String key) {
        return getStringRedisTemplate().hasKey(key);
    }

    /**
     * 删除
     *
     * @param key key
     */
    public void delete(String key) {
        getStringRedisTemplate().delete(key);
    }

    /**
     * 自增自定义数值
     *
     * @param key   key
     * @param delta 增量
     * @return 自增后的值
     */
    public Long increment(String key, long delta) {
        return getStringRedisTemplate().opsForValue().increment(key, delta);
    }

    /**
     * 获取value
     *
     * @param key key
     * @return value
     */
    public String get(String key) {
        return getStringRedisTemplate().opsForValue().get(key);
    }

    /**
     * 设置值
     *
     * @param key     键
     * @param value   值
     * @param timeout 过期时间
     * @param unit    时间单位
     */
    public void set(String key, String value, long timeout, TimeUnit unit) {
        getStringRedisTemplate().opsForValue().set(key, value, timeout, unit);
    }

    /**
     * 设置，并覆盖原字符串部分
     *
     * @param key    键
     * @param value  值
     * @param offset offset
     */
    public void set(String key, String value, long offset) {
        getStringRedisTemplate().opsForValue().set(key, value, offset);
    }

    /**
     * 设置值
     *
     * @param key
     * @param value
     */
    public void set(String key, String value) {
        getStringRedisTemplate().opsForValue().set(key, value);
    }

    /**
     * 设置过期时间--一般配合redis其他操作使用，存在并发风险，慎用！！
     *
     * @param key     key
     * @param timeout 时间
     * @param unit    单位
     * @return 是否成功
     */
    public Boolean expire(String key, long timeout, TimeUnit unit) {
        return getStringRedisTemplate().expire(key, timeout, unit);
    }

    /** hash 相关操作 **/
    /**
     * 散列获取域值
     *
     * @param key     散列key
     * @param hashKey 域key
     * @return 值
     */
    @SuppressWarnings("unchecked")
    public <V> V hGet(String key, Object hashKey) {
        return (V) getStringRedisTemplate().opsForHash().get(key, hashKey);
    }

    /**
     * 散列域赋值
     *
     * @param key     散列key
     * @param hashKey 域key
     * @param value   值
     */
    public void hSet(String key, Object hashKey, Object value) {
        getStringRedisTemplate().opsForHash().put(key, hashKey, value);
    }

    /**
     * 一次性写入hashmap中的所有值
     *
     * @param key     redis key
     * @param hashMap hashKey - hashValue
     */
    public void hSetAll(String key, Map<String, String> hashMap) {
        getStringRedisTemplate().opsForHash().putAll(key, hashMap);
    }


    /**
     * 散列删除域值
     *
     * @param key     散列key
     * @param hashKey 域key
     */
    public void hDel(String key, Object... hashKey) {
        getStringRedisTemplate().opsForHash().delete(key, hashKey);
    }

    /**
     * 注意数据量问题！！！
     *
     * @param key
     * @return
     */
    public List<Object> hValues(String key) {
        return getStringRedisTemplate().opsForHash().values(key);
    }

    /**
     * 根据key 获取 hashMap
     *
     * @param key redis key
     * @return hashMap
     */
    public <HK, HV> Map<HK, HV> entries(String key) {
        return (Map<HK, HV>) getStringRedisTemplate().opsForHash().entries(key);
    }

    /**
     * 散列域值递增固定值
     *
     * @param key     散列key
     * @param hashKey 域key
     * @param delta   递增数目
     * @return 递增后的值
     */
    public Long hIncr(String key, Object hashKey, Long delta) {
        return getStringRedisTemplate().opsForHash().increment(key, hashKey, delta);
    }

    /** set 相关操作 **/
    /**
     * 向set 中添加值
     *
     * @param key   对应key
     * @param value 对应value
     * @return
     */
    public boolean addSet(String key, String value) {
        return getStringRedisTemplate().opsForSet().add(key, value) > 0;
    }

    /**
     * 获取key对应的Set
     *
     * @param key 对应的key
     * @return
     */
    public Set<String> getSet(String key) {
        return getStringRedisTemplate().opsForSet().members(key);
    }

    /**
     * 清空redis 谨慎操作！！！
     */
    public void flushAll() {
        getStringRedisTemplate().execute((RedisCallback<Void>) connection -> {
            connection.flushAll();
            return null;
        });
    }

    /**
     * 以Map的形式返回hash中的存储和值
     *
     * @param key 键值
     * @return map对象
     */
    public Map<String, String> hgetAll(String key) {
        StringRedisTemplate redisTemplate = getStringRedisTemplate();

        //返回object对象
        Object mapValue = getStringRedisTemplate().opsForHash().entries(key);
        Map valMap = (Map) mapValue;
        Map<String, String> hMap = new HashMap<>();
        valMap.forEach((vkey, val) -> {
                    hMap.put(vkey.toString(), val.toString());
                }
        );
        return hMap;
    }

    /**
     * 执行redis任意命令
     *
     * @param command
     * @param args
     * @return
     */
    public Object execute(String command, byte[]... args) {
        return getStringRedisTemplate().execute((RedisCallback<Object>) connection -> connection.execute(command, args));
    }

    /**
     * 批量get
     *
     * @param keys redis key集合
     * @return
     */
    public Map<String, String> pipelineGet(List<String> keys) {
        if (keys.size() <= piplineBatch) {
            return pipelineGetNoBatch(keys);
        } else {
            final Map<String, String> valueMap = new HashMap<>(keys.size());

            //按每50个一组分割
            List<List<String>> parts = Lists.partition(keys, piplineBatch);

            parts.forEach(subKeys -> valueMap.putAll(pipelineGetNoBatch(subKeys)));

            return valueMap;
        }
    }

    /**
     * 批量获取hash中指定field的值
     *
     * @param keys
     * @param field
     * @return
     */
    public Map<String, String> pipelineHget(List<String> keys, String field) {
        if (keys.size() <= piplineBatch) {
            return pipelineHgetNoBatch(keys, field);
        } else {
            final Map<String, String> valueMap = new HashMap<>(keys.size());

            //按每50个一组分割
            List<List<String>> parts = Lists.partition(keys, piplineBatch);

            parts.forEach(subKeys -> valueMap.putAll(pipelineHgetNoBatch(subKeys, field)));

            return valueMap;
        }
    }

    /**
     * 批量插入
     *
     * @param dataMap 数据集合
     * @param expire  过期时间/秒
     */
    public void pipelineHmsetex(Map<String, Map<String, String>> dataMap, int expire) {
        if (MapUtils.isEmpty(dataMap)) {
            return;
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (Map.Entry<String, Map<String, String>> entry : dataMap.entrySet()) {
                    String key = entry.getKey();
                    Map<String, String> value = entry.getValue();
                    template.opsForHash().putAll(key, value);
                    template.expire(key, expire, TimeUnit.SECONDS);
                }
                return null;
            }
        });
    }

    /**
     * 批量插入
     *
     * @param dataMap 数据集合
     * @param expire  过期时间/秒
     */
    public void pipelineMsetex(Map<String, String> dataMap, int expire) {
        if (MapUtils.isEmpty(dataMap)) {
            return;
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (Map.Entry<String, String> entry : dataMap.entrySet()) {
                    template.opsForValue().set(entry.getKey(), entry.getValue(), expire, TimeUnit.SECONDS);
                }
                return null;
            }
        });
    }

    /**
     * 批量set数据
     *
     * @param map    数据集合
     * @param expire 过期时间
     */
    public void pipelineSadd(Map<String, Set<String>> map, int expire) {
        StringRedisTemplate redisTemplate = getStringRedisTemplate();
        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (Map.Entry<String, Set<String>> entry : map.entrySet()) {
                    String key = entry.getKey();
                    Set<String> value = entry.getValue();
                    template.opsForSet().add(key, value.toArray(new String[value.size()]));
                    template.expire(key, expire, TimeUnit.SECONDS);
                }
                return null;
            }
        });
    }


    /**
     * 批量插入
     *
     * @param dataMap key 过期时间/秒 value 数据集合
     */
    public void pipelineMsetex(Map<Integer, Map<String, String>> dataMap) {
        if (MapUtils.isEmpty(dataMap)) {
            return;
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (Map.Entry<Integer, Map<String, String>> expireEntry : dataMap.entrySet()) {
                    for (Map.Entry<String, String> entry : expireEntry.getValue().entrySet()) {
                        template.opsForValue().set(entry.getKey(), entry.getValue(), expireEntry.getKey(), TimeUnit.SECONDS);
                    }
                }
                return null;
            }
        });
    }

    private Map<String, String> pipelineGetNoBatch(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return new HashMap<>();
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();
        final Map<String, String> valueMap = new HashMap<>(keys.size());

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (String key : keys) {
                    template.opsForValue().get(key);
                }
                return null;
            }
        });
        for (int index = 0; index < keys.size(); index++) {
            Object pipResult = pipResults.get(index);
            String key = keys.get(index);
            if (pipResult != null) {
                valueMap.put(key, pipResult.toString());
            }
        }
        return valueMap;
    }

    private Map<String, String> pipelineHgetNoBatch(List<String> keys, String field) {
        if (CollectionUtils.isEmpty(keys)) {
            return new HashMap<>();
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();
        final Map<String, String> valueMap = new HashMap<>(keys.size());

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (String key : keys) {
                    template.opsForHash().get(key, field);
                }
                return null;
            }
        });
        for (int index = 0; index < keys.size(); index++) {
            Object pipResult = pipResults.get(index);
            String key = keys.get(index);
            if (pipResult != null) {
                valueMap.put(key, pipResult.toString());
            }
        }
        return valueMap;
    }


    /**
     * 批量获取hash中指定field的值
     *
     * @param keys
     * @return
     */
    public Map<String, Map<String, String>> pipelineHget(List<String> keys) {
        if (keys.size() <= piplineBatch) {
            return pipelineHgetAllNoBatch(keys);
        } else {
            final Map<String, Map<String, String>> valuesMap = new HashMap<>(keys.size());

            //按每50个一组分割
            List<List<String>> parts = Lists.partition(keys, piplineBatch);

            parts.forEach(subKeys -> valuesMap.putAll(pipelineHgetAllNoBatch(subKeys)));

            return valuesMap;
        }
    }

    private Map<String, Map<String, String>> pipelineHgetAllNoBatch(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return new HashMap<>();
        }
        StringRedisTemplate redisTemplate = getStringRedisTemplate();
        final Map<String, Map<String, String>> valuesMap = new HashMap<>(keys.size());

        List<Object> pipResults = redisTemplate.executePipelined(new SessionCallback<Object>() {
            @Override
            public <K, V> Object execute(RedisOperations<K, V> operations) throws DataAccessException {
                StringRedisTemplate template = (StringRedisTemplate) operations;
                for (String key : keys) {
                    template.opsForHash().entries(key);
                }

                return null;
            }
        });

        for (int index = 0; index < keys.size(); index++) {
            Object pipResult = pipResults.get(index);
            String key = keys.get(index);
            if (pipResult != null) {
                Map valMap = (Map) pipResult;
                Map<String, String> hMap = new HashMap<>();
                valMap.forEach((vkey, val) -> {
                    if (vkey != null && val != null) {
                        hMap.put(vkey.toString(), val.toString());
                    }
                });

                valuesMap.put(key, hMap);
            }
        }

        return valuesMap;
    }
}
