package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.update.UpdateRequestBuilder;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.xpack.client.PreBuiltXPackTransportClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Map;

public class EsUtil {
    private static final Logger logger = LoggerFactory.getLogger(EsUtil.class);
    private TransportClient client;

    //通过 TransportClient 方式连接es
    private String esClusterName;
    private String esHost;
    private String esUserName;
    private String esPwd;

    private EsUtil() throws Exception {
        initClient();
    }

    public EsUtil(String esClusterName, String esHost, String esUserName, String esPwd) throws Exception {
        this.esClusterName = esClusterName;
        this.esHost = esHost;
        this.esUserName = esUserName;
        this.esPwd = esPwd;

        initClient();
    }

    private synchronized void initClient() throws Exception {
        if (client != null) {
            return;
        }
        client = new PreBuiltXPackTransportClient(Settings.builder()
                .put("cluster.name", esClusterName)
                .put("xpack.security.user", String.format("%s:%s", esUserName, esPwd))
                .put("client.transport.sniff", false)
                .build())
                .addTransportAddress(new InetSocketTransportAddress(InetAddress.getByName(esHost), 9300));


    }

    public void close() {
        client.close();
        client = null;
    }

    /**
     * 批量更新es
     *
     * @param index    索引
     * @param batchMap key:type value:data
     * @return
     */
    public void batchUpsert(String index, Map<String, Map<String, JSONObject>> batchMap) {
        for (String type : batchMap.keySet()) {
            Map<String, JSONObject> data = batchMap.get(type);
            batchUpsert(index, type, data);
        }
    }

    /**
     * 批量插入es
     *
     * @param index    索引
     * @param batchMap key:type value:data
     * @return
     */
    public void batchInsert(String index, Map<String, Map<String, JSONObject>> batchMap) {
        for (String type : batchMap.keySet()) {
            Map<String, JSONObject> data = batchMap.get(type);
            batchInsert(index, type, data);
        }
    }

    /**
     * 批量更新es
     *
     * @param index    索引
     * @param type     类型
     * @param batchMap key:type value:data
     * @return
     */
    public void batchUpsert(String index, String type, Map<String, JSONObject> batchMap) {
        BulkRequestBuilder bulkRequest = client.prepareBulk();
        for (String key : batchMap.keySet()) {
            JSONObject source = batchMap.get(key);
            UpdateRequestBuilder urb = getUpdateRequestBuilder(index, type, key, source);
            bulkRequest.add(urb);
        }
        executeRequest(bulkRequest);
    }

    /**
     * 批量插入es
     *
     * @param index    索引
     * @param type     类型
     * @param batchMap key:type value:data
     * @return
     */
    public void batchInsert(String index, String type, Map<String, JSONObject> batchMap) {
        BulkRequestBuilder bulkRequest = client.prepareBulk();
        for (String key : batchMap.keySet()) {
            JSONObject source = batchMap.get(key);
            IndexRequestBuilder irb = getIndexRequestBuilder(index, type, key, source);
            bulkRequest.add(irb);
        }
        executeRequest(bulkRequest);
    }

    /**
     * 执行请求
     *
     * @param bulkRequest
     * @return
     */
    private void executeRequest(BulkRequestBuilder bulkRequest) {
        if (bulkRequest == null || bulkRequest.numberOfActions() == 0) {
            return;
        }
        bulkRequest.execute();
    }


    /**
     * 组装更新对象
     *
     * @param index  索引
     * @param type   类型
     * @param key    id
     * @param source 字段
     * @return UpdateRequestBuilder
     */
    private UpdateRequestBuilder getUpdateRequestBuilder(String index, String type, String key, Map source) {
        UpdateRequestBuilder urb = client.prepareUpdate(index, type, key);
        urb.setDoc(source);
        urb.setDocAsUpsert(true);
        urb.setRetryOnConflict(3);
        return urb;
    }

    /**
     * 组装插入对象
     *
     * @param index  索引
     * @param type   类型
     * @param key    id
     * @param source 字段
     * @return UpdateRequestBuilder
     */
    private IndexRequestBuilder getIndexRequestBuilder(String index, String type, String key, Map source) {
        return client.prepareIndex(index, type, key).setSource(source);
    }

}
