package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSON;
import com.google.common.base.CaseFormat;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class HbaseDataUtil {
    private static final Logger logger = LoggerFactory.getLogger(HbaseDataUtil.class);

    private HbaseUtil hbaseUtil;

    //默认的namespace
    private String DEFAULT_NAMESPACE = "default";

    //默认列簇
    private String DEFAULT_COLUMN_FAMILY = "cf";

    public void setHbaseUtil(HbaseUtil hbaseUtil) {
        this.hbaseUtil = hbaseUtil;
    }

    //java 小驼峰格式
    private String getLowerCamelColumn(boolean lowerCamel, String column) {
        if (lowerCamel && column.contains("_")) {
            return CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, column);
        }
        return column;
    }

    //组装单行数据
    private void putMapData(Map<String, String> map, String column, String value, boolean lowerCamel) {
        if (StringUtils.isAnyBlank(column, value)) {
            return;
        }
        column = getLowerCamelColumn(lowerCamel, column);
        map.put(column, value);
    }

    //组装多行数据
    private void putMapData(Map<String, Map<String, String>> map, String rowKey, Map<String, String> columnMap) {
        if (StringUtils.isBlank(rowKey)) {
            return;
        }
        if (columnMap == null || columnMap.size() == 0) {
            return;
        }
        map.put(rowKey, columnMap);
    }

    //解析单行单列
    private String getResultString(Result result, String family, String column) {
        String value = "";
        if (result != null && result.containsColumn(family.getBytes(), column.getBytes())) {
            value = Bytes.toString(result.getValue(family.getBytes(), column.getBytes()));
        }
        return value;
    }

    //解析单行多列
    private Map<String, String> getResultMap(Result result, String family, List<String> columns, boolean lowerCamel) {
        Map<String, String> map = new HashMap<>();
        for (String column : columns) {
            String value = getResultString(result, family, column);
            putMapData(map, column, value, lowerCamel);
        }
        return map;
    }

    //解析单行多列
    private <T> T getResultMap(Result result, String family, List<String> columns, Class<T> clazz) {
        Map<String, String> map = getResultMap(result, family, columns, true);
        return JSON.parseObject(JSON.toJSONString(map), clazz);
    }

    //解析单行所有列
    private Map<String, String> getResultMap(Result result, String family, boolean lowerCamel) {
        List<Cell> cellList = result.listCells();
        if (CollectionUtils.isEmpty(cellList)) {
            return null;
        }
        Map<String, String> map = new HashMap<>();
        for (Cell cell : cellList) {
            if (!Bytes.toString(CellUtil.cloneFamily(cell)).equalsIgnoreCase(family)) {
                continue;
            }
            String column = Bytes.toString(CellUtil.cloneQualifier(cell));
            String value = Bytes.toString(CellUtil.cloneValue(cell));
            putMapData(map, column, value, lowerCamel);
        }
        return map;
    }

    //解析单行所有列
    private <T> T getResultMap(Result result, String family, Class<T> clazz) {
        Map<String, String> map = getResultMap(result, family, true);
        return JSON.parseObject(JSON.toJSONString(map), clazz);
    }

    //解析多行单列
    private Map<String, String> getResultMap(Result[] results, String family, String column) {
        Map<String, String> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            String value = getResultString(result, family, column);
            if (StringUtils.isNotBlank(value)) {
                map.put(rowKey, value);
            }
        }
        return map;
    }

    //解析多行多列
    private Map<String, Map<String, String>> getResultMap(Result[] results, String family, List<String> columns, boolean lowerCamel) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family, columns, lowerCamel);
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行多列
    private <T> Map<String, T> getResultMap(Result[] results, String family, List<String> columns, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, columns, clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

    //解析多行多列
    private Map<String, Map<String, String>> getResultMap(Result[] results, String family, Map<String, List<String>> rowkeyMap, boolean lowerCamel) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family, rowkeyMap.get(rowKey), lowerCamel);
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行多列
    private <T> Map<String, T> getResultMap(Result[] results, String family, Map<String, List<String>> rowkeyMap, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, rowkeyMap.get(rowKey), clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

    //解析多行所有列
    private Map<String, Map<String, String>> getResultMap(Result[] results, String family, boolean lowerCamel) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family, lowerCamel);
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行所有列
    private <T> Map<String, T> getResultMap(Result[] results, String family, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

    /**
     * 判断rowkey是否存在
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String tableName, String rowKey) {
        return exists(DEFAULT_NAMESPACE, tableName, rowKey);
    }

    /**
     * 判断rowkey是否存在
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @return rowkey是否存在，存在=true
     */
    public Boolean exists(String nameSpace, String tableName, String rowKey) {
        return hbaseUtil.exists(nameSpace, tableName, rowKey);
    }

    /**
     * 查询单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列
     * @return column_value
     */
    public String getRow(String tableName, String rowKey, String column) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 查询单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列
     * @return column_value
     */
    public String getRow(String nameSpace, String tableName, String rowKey, String family, String column) {
        Result result = hbaseUtil.getRow(nameSpace, tableName, rowKey, family, column);
        return getResultString(result, family, column);
    }

    /**
     * 查询单行部分列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列
     * @return Map<String ,   String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String tableName, String rowKey, List<String> columns) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns, false);
    }

    /**
     * 查询单行部分列
     *
     * @param tableName  表名
     * @param rowKey     主键
     * @param columns    列
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String tableName, String rowKey, List<String> columns, boolean lowerCamel) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns, lowerCamel);
    }

    /**
     * 查询单行部分列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public <T> T getRow(String tableName, String rowKey, List<String> columns, Class<T> clazz) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns, clazz);
    }

    /**
     * 查询单行部分列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKey     主键
     * @param family     列族
     * @param columns    列
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns, boolean lowerCamel) {
        Result result = hbaseUtil.getRow(nameSpace, tableName, rowKey, family, columns);
        return getResultMap(result, family, columns, lowerCamel);
    }

    /**
     * 查询单行部分列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   列
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public <T> T getRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns, Class<T> clazz) {
        Result result = hbaseUtil.getRow(nameSpace, tableName, rowKey, family, columns);
        return getResultMap(result, family, columns, clazz);
    }

    /**
     * 查询单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return Map<String ,   String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String tableName, String rowKey) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, false);
    }

    /**
     * 查询单行所有列
     *
     * @param tableName  表名
     * @param rowKey     主键
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String tableName, String rowKey, boolean lowerCamel) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, lowerCamel);
    }

    /**
     * 查询单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public <T> T getRow(String tableName, String rowKey, Class<T> clazz) {
        return getRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, clazz);
    }

    /**
     * 查询单行所有列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKey     主键
     * @param family     列族
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public Map<String, String> getRow(String nameSpace, String tableName, String rowKey, String family, boolean lowerCamel) {
        Result result = hbaseUtil.getRow(nameSpace, tableName, rowKey, family);
        return getResultMap(result, family, lowerCamel);
    }

    /**
     * 查询单行所有列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       String> key:column_name value:column_value
     */
    public <T> T getRow(String nameSpace, String tableName, String rowKey, String family, Class<T> clazz) {
        Result result = hbaseUtil.getRow(nameSpace, tableName, rowKey, family);
        return getResultMap(result, family, clazz);
    }

    /**
     * 查询多行单列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param column     列
     * @return Map<String ,   String> key:rowkey value:column_value
     */
    public Map<String, String> getRowList(String tableName, List<String> rowKeyList, String column) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, column);
    }


    /**
     * 查询多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param column     列
     * @return Map<String ,   String> key:rowkey value:column_value
     */
    public Map<String, String> getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, String column) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyList, family, column);
        return getResultMap(results, family, column);
    }

    /**
     * 查询多行部分列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param columns    列
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, List<String> rowKeyList, List<String> columns) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, columns, false);
    }

    /**
     * 查询多行部分列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param columns    列
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, List<String> rowKeyList, List<String> columns, boolean lowerCamel) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, columns, lowerCamel);
    }

    /**
     * 查询多行部分列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param columns    列
     * @param clazz      将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String tableName, List<String> rowKeyList, List<String> columns, Class<T> clazz) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, columns, clazz);
    }

    /**
     * 查询多行部分列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param columns    列
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, List<String> columns, boolean lowerCamel) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyList, family, columns);
        return getResultMap(results, family, columns, lowerCamel);
    }

    /**
     * 查询多行部分列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param columns    列
     * @param clazz      将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, List<String> columns, Class<T> clazz) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyList, family, columns);
        return getResultMap(results, family, columns, clazz);
    }

    /**
     * 查询多行所有列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, List<String> rowKeyList) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, false);
    }

    /**
     * 查询多行所有列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, List<String> rowKeyList, boolean lowerCamel) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, lowerCamel);
    }

    /**
     * 查询多行所有列
     *
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param clazz      将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String tableName, List<String> rowKeyList, Class<T> clazz) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyList, DEFAULT_COLUMN_FAMILY, clazz);
    }

    /**
     * 查询多行所有列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, boolean lowerCamel) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyList, family);
        return getResultMap(results, family, lowerCamel);
    }

    /**
     * 查询多行所有列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyList 主键列表
     * @param family     列族
     * @param clazz      将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String nameSpace, String tableName, List<String> rowKeyList, String family, Class<T> clazz) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyList, family);
        return getResultMap(results, family, clazz);
    }

    /**
     * 查询多行不同列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return Map<String ,   Map < String ,   String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, Map<String, List<String>> rowKeyMap) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY, false);
    }

    /**
     * 查询多行不同列
     *
     * @param tableName  表名
     * @param rowKeyMap  键值对
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String tableName, Map<String, List<String>> rowKeyMap, boolean lowerCamel) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY, lowerCamel);
    }

    /**
     * 查询多行不同列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String tableName, Map<String, List<String>> rowKeyMap, Class<T> clazz) {
        return getRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY, clazz);
    }

    /**
     * 查询多行不同列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowKeyMap  键值对
     * @param family     列族
     * @param lowerCamel column_name 是否以java小驼峰格式返回 true：是 false：否
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public Map<String, Map<String, String>> getRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family, boolean lowerCamel) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyMap, family);
        return getResultMap(results, family, rowKeyMap, lowerCamel);
    }

    /**
     * 查询多行不同列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @param clazz     将hbase查出来的数据转化成实体对象（注意点：column_name转成小驼峰后与实体对象中的字段名称一致）
     * @return Map<String   ,       Map   <   String   ,       String>> key:rowkey hashkey:column_name hashvalue:column_value
     */
    public <T> Map<String, T> getRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family, Class<T> clazz) {
        Result[] results = hbaseUtil.getRowList(nameSpace, tableName, rowKeyMap, family);
        return getResultMap(results, family, rowKeyMap, clazz);
    }

    /**
     * 插入单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列名
     * @param value     列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, String rowKey, String column, String value) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column, value);
    }

    /**
     * 插入单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列名
     * @param value     列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, String rowKey, String family, String column, String value) {
        return hbaseUtil.insert(nameSpace, tableName, rowKey, family, column, value);
    }

    /**
     * 插入单行多列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, String rowKey, Map<String, String> columns) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 插入单行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, String rowKey, String family, Map<String, String> columns) {
        return hbaseUtil.insert(nameSpace, tableName, rowKey, family, columns);
    }

    /**
     * 插入多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param column     列名
     * @param value      列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, List<String> rowkeyList, String column, String value) {
        return insert(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY, column, value);
    }

    /**
     * 插入多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @param column     列名
     * @param value      列值
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, List<String> rowkeyList, String family, String column, String value) {
        return hbaseUtil.insert(nameSpace, tableName, rowkeyList, family, column, value);
    }

    /**
     * 插入多行多列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String tableName, Map<String, Map<String, String>> rowKeyMap) {
        return insert(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 插入多行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return true:插入成功 false:插入失败
     */
    public boolean insert(String nameSpace, String tableName, Map<String, Map<String, String>> rowKeyMap, String family) {
        return hbaseUtil.insert(nameSpace, tableName, rowKeyMap, family);
    }

    /**
     * 删除单行单列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param column    列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey, String column) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 删除单行单列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param column    列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family, String column) {
        return hbaseUtil.deleteRow(nameSpace, tableName, rowKey, family, column);
    }

    /**
     * 删除单行多列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @param columns   列集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey, List<String> columns) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY, columns);
    }

    /**
     * 删除单行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @param columns   列集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family, List<String> columns) {
        return hbaseUtil.deleteRow(nameSpace, tableName, rowKey, family, columns);
    }

    /**
     * 删除单行所有列
     *
     * @param tableName 表名
     * @param rowKey    主键
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String tableName, String rowKey) {
        return deleteRow(DEFAULT_NAMESPACE, tableName, rowKey, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除单行所有列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKey    主键
     * @param family    列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRow(String nameSpace, String tableName, String rowKey, String family) {
        return hbaseUtil.deleteRow(nameSpace, tableName, rowKey, family);
    }

    /**
     * 删除多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param column     列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, List<String> rowkeyList, String column) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY, column);
    }

    /**
     * 删除多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @param column     列名
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, List<String> rowkeyList, String family, String column) {
        return hbaseUtil.deleteRowList(nameSpace, tableName, rowkeyList, family, column);
    }

    /**
     * 插入多行多列
     *
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, Map<String, List<String>> rowKeyMap) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowKeyMap, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除多行多列
     *
     * @param nameSpace hbase命名空间
     * @param tableName 表名
     * @param rowKeyMap 键值对
     * @param family    列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, Map<String, List<String>> rowKeyMap, String family) {
        return hbaseUtil.deleteRowList(nameSpace, tableName, rowKeyMap, family);
    }

    /**
     * 删除多行单列
     *
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String tableName, List<String> rowkeyList) {
        return deleteRowList(DEFAULT_NAMESPACE, tableName, rowkeyList, DEFAULT_COLUMN_FAMILY);
    }

    /**
     * 删除多行单列
     *
     * @param nameSpace  hbase命名空间
     * @param tableName  表名
     * @param rowkeyList 主键集合
     * @param family     列族
     * @return true:删除成功 false:删除失败
     */
    public boolean deleteRowList(String nameSpace, String tableName, List<String> rowkeyList, String family) {
        return hbaseUtil.deleteRowList(nameSpace, tableName, rowkeyList, family);
    }
}
