package cn.com.duiba.bigdata.common.biz.utils;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.Tuple;

import java.util.Map;
import java.util.Set;

/**
 * @author xugf 2022-12-19
 * 通过jedis连接池访问redis，不接入spring
 */
public class JedisUtil {
    //jedis连接池配置
    private JedisPoolConfig poolConfig;

    //jedis连接池
    private JedisPool jedisPool;

    //连接地址
    private String host;

    //密码
    private String auth;

    //redis默认连接端口
    public static final int DEFAULT_PORT = 6379;

    //默认连接超时时间
    public static final int DEFAULT_TIMEOUT = 2000;


    public void setPoolConfig(JedisPoolConfig poolConfig) {
        this.poolConfig = poolConfig;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public void setAuth(String auth) {
        this.auth = auth;
    }

    //获取jedis连接
    private Jedis getJedis() {
        if (jedisPool == null || jedisPool.isClosed()) {
            initJedisPool();
        }

        return jedisPool.getResource();
    }

    //将jedis连接归还到连接池
    private void closeJedis(Jedis jedis) {
        if (jedis != null) {
            jedis.close();
        }
    }

    private synchronized void initJedisPool() {
        if (jedisPool != null && !jedisPool.isClosed()) {
            return;
        }

        //初始化jedis连接池
        jedisPool = new JedisPool(poolConfig, host, DEFAULT_PORT, DEFAULT_TIMEOUT, auth);
    }

    /**
     * 往zset中添加一个数据
     *
     * @param key    redis key
     * @param member 数据
     * @param score  得分
     * @return 成功的个数
     */
    public Long zadd(String key, String member, double score) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zadd(key, score, member);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 往zset中批量添加数据
     *
     * @param key          redis key
     * @param scoreMembers 数据和得分
     * @return 成功的个数
     */
    public Long zadd(String key, Map<String, Double> scoreMembers) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zadd(key, scoreMembers);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 获取zset key的有序集元素个数
     *
     * @param key redis key
     * @return 有序集元素个数
     */
    public Long zcard(String key) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zcard(key);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除并返回有序集合key中最小得分的成员（单个值）
     *
     * @param key redis key
     * @return 一个元素和分数
     */
    public Tuple zpopmin(String key) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zpopmin(key);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除并返回有序集合key中最小得分的成员（多个值）
     *
     * @param key   redis key
     * @param count 需要弹出的元素个数
     * @return 元素和分数列表
     */
    public Set<Tuple> zpopmin(String key, int count) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zpopmin(key, count);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除并返回有序集合key中最大得分的成员（单个值）
     *
     * @param key redis key
     * @return 一个元素和分数
     */
    public Tuple zpopmax(String key) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zpopmax(key);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除并返回有序集合key中最大得分的成员（多个值）
     *
     * @param key   redis key
     * @param count 需要弹出的元素个数
     * @return 元素和分数列表
     */
    public Set<Tuple> zpopmax(String key, int count) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zpopmax(key, count);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除zset中的成员
     *
     * @param key     redis key
     * @param members 元素集合
     * @return 从有序集合中删除的成员个数，不包括不存在的成员
     */
    public Long zrem(String key, String... members) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.zrem(key, members);
        } finally {
            closeJedis(jedis);
        }
    }

    /**
     * 删除redis中的key
     *
     * @param key redis key
     * @return 删除的个数
     */
    public Long del(String key) {
        Jedis jedis = null;
        try {
            jedis = getJedis();
            return jedis.del(key);
        } finally {
            closeJedis(jedis);
        }
    }


}
