package cn.com.duiba.bigdata.common.biz.utils;

import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.jena.sparql.expr.E_LogicalAnd;
import org.apache.jena.sparql.expr.E_LogicalOr;
import org.apache.jena.sparql.expr.Expr;
import org.apache.jena.sparql.expr.ExprVar;
import org.apache.jena.sparql.util.ExprUtils;

import java.util.List;

/**
 * 标签过滤组合拆解工具类
 */
@Slf4j
public class LabelDepartUtil {

    public static void main(String[] args) {

        String s ="(((((?010102||?010101)||(?010401))||(?01050202||?01050201))||(?020405||?020401))||(?030413||?030406))";
        List<String> labelFilterRule = distributiveLabelFilterRule(s);
        System.out.println(labelFilterRule);
    }

    public static final String QUESTION_MARK = "?";

    /**
     * 目前给DMP后端使用
     * 将标签组合过滤规则拆解，拆成若干个只有交集的并的组合
     * 需要拆解的标签过滤规则，如：labelFilterRule ：(?0701 || ?0702 || ?0703 ) && ( ?010201 || ?010202 || ?010205)
     * 拆解后的结果：[(?0701 && ?010201), (?0701 && ?010202), (?0701 && ?010205), (?0702 && ?010201), (?0702 && ?010202), (?0702 && ?010205), (?0703 && ?010201), (?0703 && ?010202), (?0703 && ?010205)]
     *
     * @param labelFilterRule
     * @return
     */
    public static List<String> distributiveLabelFilterRule(String labelFilterRule) {

        if (StringUtils.isEmpty(labelFilterRule) || !labelFilterRule.contains(QUESTION_MARK)) {
            return null;
        }

        List<String> labelSet = Lists.newArrayList();
        try {
            Expr expr = ExprUtils.parse(labelFilterRule);

            if (expr instanceof E_LogicalAnd) {
                E_LogicalAnd e_logicalAnd = (E_LogicalAnd) expr;
                String opName = e_logicalAnd.getOpName();

                List<String> left = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalAnd.getArg1(), left);

                List<String> right = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalAnd.getArg2(), right);

                for (String l : left) {
                    for (String r : right) {
                        StringBuilder sb = new StringBuilder();
                        labelSet.add(sb.append("(").append(l).append(" ").append(opName).append(" ").append(r).append(")").toString());
                    }
                }

            } else if (expr instanceof E_LogicalOr) {
                E_LogicalOr e_logicalOr = (E_LogicalOr) expr;

                List<String> left = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalOr.getArg1(), left);

                List<String> right = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalOr.getArg2(), right);

                labelSet.addAll(left);
                labelSet.addAll(right);
            }else if(expr instanceof ExprVar){
                ExprVar exprVar = (ExprVar) expr;
                labelSet.add(exprVar.toString());
            }
        } catch (Exception e) {
            log.warn("LabelDepartUtil.distributiveLabelFilterRule labelFilterRule = {} error = {}", labelFilterRule, e);
        }

        return labelSet;
    }

    /**
     * 目前给大数据过滤服务使用
     * 获取标签组合的所有标签元素
     *
     * @param labelCompose 标签组合
     * @return
     */
    public static List<String> fetchLabelComposeElements(String labelCompose) {

        if (StringUtils.isEmpty(labelCompose) || !labelCompose.contains(QUESTION_MARK)) {
            return null;
        }

        try {
            List<String> elements = Lists.newArrayList();
            Expr expr = ExprUtils.parse(labelCompose);
            fetchElementFromLogical(expr, elements);
            return elements;
        } catch (Exception e) {
            log.warn("LabelSetTransformUtil.fetchLabelComposeElements labelCompose = {} error ={} ", labelCompose, e);
        }

        return null;
    }

    /**
     * 针对逻辑或的并操作进行解析标签组合过滤规则，递归处理
     *
     * @param expr
     * @param param
     * @return
     */
    public static String concatStrForLogicalOr(Expr expr, List<String> param) {
        if (expr instanceof ExprVar) {
            ExprVar ev = (ExprVar) expr;
            param.add(ev.toString());
            return null;
        } else if (expr instanceof E_LogicalOr) {
            E_LogicalOr ssub1 = (E_LogicalOr) expr;
            String s1 = concatStrForLogicalOr(ssub1.getArg1(), param);
            if (StringUtils.isNotEmpty(s1)) {
                param.add(s1);
            }

            String s2 = concatStrForLogicalOr(ssub1.getArg2(), param);
            if (StringUtils.isNotEmpty(s2)) {
                param.add(s2);
            }

            return null;
        } else if (expr instanceof E_LogicalAnd) {
            param.add(expr.toString());
            return null;
        }

        return null;
    }

    /**
     * 针对逻辑或的并操作提取元素，递归处理
     *
     * @param expr
     * @param elements
     * @return
     */
    public static String fetchElementFromLogical(Expr expr, List<String> elements) {
        if (expr instanceof ExprVar) {
            ExprVar ev = (ExprVar) expr;
            elements.add(ev.getVarName());
            return null;
        } else if (expr instanceof E_LogicalOr) {
            E_LogicalOr e_logicalOr = (E_LogicalOr) expr;
            String s1 = fetchElementFromLogical(e_logicalOr.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(s1);
            }

            String s2 = fetchElementFromLogical(e_logicalOr.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(s2);
            }

            return null;
        } else if (expr instanceof E_LogicalAnd) {
            E_LogicalAnd e_logicalAnd = (E_LogicalAnd) expr;
            String s1 = fetchElementFromLogical(e_logicalAnd.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(s1);
            }

            String s2 = fetchElementFromLogical(e_logicalAnd.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(s2);
            }
            return null;
        }

        return null;
    }

}

