package cn.com.duiba.bigdata.common.biz.service;

import cn.com.duiba.bigdata.common.biz.dto.KeyValueDto;
import cn.com.duiba.bigdata.common.biz.dto.SortDto;
import cn.com.duiba.bigdata.common.biz.enums.HologresShowTypeEnum;
import cn.com.duiba.bigdata.common.biz.enums.OperatorsEnum;
import cn.com.duiba.bigdata.common.biz.enums.SortEnum;
import cn.com.duiba.bigdata.common.biz.form.HoloQueryForm;
import cn.com.duiba.bigdata.common.biz.interfaces.*;
import cn.com.duiba.bigdata.common.biz.utils.BusinessEnumUtil;
import cn.com.duiba.bigdata.common.biz.utils.MD5Util;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

/**
 * @author xugf 2023-11-28
 * 通过查询条件，自动组装查询sql
 */
@Slf4j
public class HoloSqlService {

    /**
     * 组装查询sql语句
     *
     * @param form         查询条件
     * @param dimensionList    本次查询的原始维度列表
     * @param businessType 业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return sql
     * @throws Exception 异常
     */
    public static String getQuerySql(HoloQueryForm form, List<String> dimensionList, int businessType, int resultType) throws Exception {
        //添加排序指标
        addOrderMetricList(form, businessType);

        //添加计算指标对应的基础指标
        addCalculateMetricList(form, businessType);

        //添加组合指标对应的基础指标
        addCombineMetricList(form, businessType);

        //添加默认的排序指标
        addOrderByMetric(form);

        //对查询指标进行分组合并
        List<Object> metricList = getMetricList(form.getMetricList(), businessType);

        //获取指标条件
        List<KeyValueDto> metricConditionList = getMetricConditionList(form);

        //组装多层嵌套sql语句
        StringBuilder sql = new StringBuilder("select ");

        //查询维度
        String dimensionSumSql = getDimensionSumSql(form.getDataShowType(), dimensionList, businessType, resultType);
        if (StringUtils.isNotBlank(dimensionSumSql)) {
            sql.append(dimensionSumSql);
        }

        //查询指标
        String metricSumSql = getMetricSumSql(metricList, form.getCalculateMetricList(), form.getCombineMetricList(), businessType, resultType);
        //指标一定会有值
        sql.append(metricSumSql);

        //子查询
        String querySql = getQueryUnionAllSql(form, metricList, businessType);
        if (StringUtils.isBlank(querySql)) {
            return null;
        }
        sql.append(" from (").append(querySql).append(") t ");

        //维表关联 left join
        String leftJoinSql = getDictLeftJoinSql(dimensionList, businessType);
        if (StringUtils.isNotBlank(leftJoinSql)) {
            sql.append(leftJoinSql);
        }

        //group by
        String groupBySumSql = getGroupBySumSql(form.getDataShowType(), dimensionList, businessType);
        if (StringUtils.isNotBlank(groupBySumSql)) {
            sql.append(" group by ").append(groupBySumSql);
        }

        //having 条件
        String havingSql = getHavingSumSql(metricConditionList, businessType);
        if (StringUtils.isNotBlank(havingSql)) {
            sql.append(" having ").append(havingSql);
        }

        //order by
        String orderBySql = getOrderBySql(form.getDataShowType(), form.getOrderByMetric(), businessType, resultType);
        if (StringUtils.isNotBlank(orderBySql)) {
            sql.append(orderBySql);
        }

        //分页展示 limit 5 offset 10 ;
        if (form.getPageSize() != null && form.getPageNumber() != null) {
            sql.append(" limit ").append(form.getPageSize()).append(" offset ").append((form.getPageNumber() - 1) * form.getPageSize());
        }

        return sql.toString();
    }

    /**
     * 添加排序指标
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addOrderMetricList(HoloQueryForm form, int businessType) {
        if (form.getOrderByMetric() == null) {
            return;
        }

        String metric = form.getOrderByMetric().getKey();
        if (Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setMetricList(metricList);
            }
            metricList.add(metric);
        } else if (Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getCalculateMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setCalculateMetricList(metricList);
            }
            metricList.add(metric);
        } else if (Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getCombineMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setCombineMetricList(metricList);
            }
            metricList.add(metric);
        }

    }

    /**
     * 将计算类指标对应的分子和分母指标加入到基础指标列表中
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addCalculateMetricList(HoloQueryForm form, int businessType) {
        if (CollectionUtils.isEmpty(form.getCalculateMetricList())) {
            return;
        }

        if (CollectionUtils.isEmpty(form.getMetricList())) {
            //初始化基础指标列表
            form.setMetricList(new ArrayList<>());
        }

        //将计算类指标对应的分子和分母指标加入到基础指标列表中
        for (String metric : form.getCalculateMetricList()) {
            CalculateMetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, metric));
            form.getMetricList().add(metricEnum.getNumerator());
            form.getMetricList().add(metricEnum.getDenominator());
        }

    }

    /**
     * 将组合类指标加入到基础指标列表中
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addCombineMetricList(HoloQueryForm form, int businessType) {
        if (CollectionUtils.isEmpty(form.getCombineMetricList())) {
            return;
        }

        if (CollectionUtils.isEmpty(form.getMetricList())) {
            //初始化基础指标列表
            form.setMetricList(new ArrayList<>());
        }

        //将组合类指标加入到基础指标列表中
        for (String combineMetric : form.getCombineMetricList()) {
            CombineMetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, combineMetric));
            String[] array = StringUtils.split(metricEnum.getMetricList(), ",");
            for (String metric : array) {
                form.getMetricList().add(metric);
            }
        }

    }

    /**
     * 添加默认的排序指标
     * @param form 查询条件
     */
    private static void addOrderByMetric(HoloQueryForm form) {
        if (form.getOrderByMetric() != null) {
            return;
        }

        //没有设置排序指标时，默认取第一个基础指标，降序排序
        String metric = form.getMetricList().get(0);
        SortDto sortDto = new SortDto(metric, SortEnum.DESC.toString());
        form.setOrderByMetric(sortDto);
    }

    /**
     * 对查询指标进行分组合并
     *
     * @param metricList   查询的指标列表
     * @param businessType 业务线类型
     * @return 分组后的指标列表
     */
    public static List<Object> getMetricList(List<String> metricList, int businessType) {
        //去除重复的指标
        Set<String> metricSet = new HashSet<>(metricList);

        //分组指标
        List<Object> list = new ArrayList<>();

        //指标聚合map
        Map<String, List<String>> map = new HashMap<>();
        for (String metric : metricSet) {
            MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric));
            StringBuilder sb = new StringBuilder();
            //指标对应的 hologres 表
            sb.append(metricEnum.getTableName());
            //指标特定的查询条件
            sb.append(metricEnum.getCondition());

            String md5 = MD5Util.computeMD5(sb.toString());

            //map指标分组初始化
            if (!map.containsKey(md5)) {
                map.put(md5, new ArrayList<>());
            }

            //将指标放入map分组中
            map.get(md5).add(metric);
        }

        //将分组后的指标进行返回
        for (List<String> metricGroupList : map.values()) {
            if (metricGroupList.size() == 1) {
                list.add(metricGroupList.get(0));
            } else {
                list.add(metricGroupList);
            }
        }

        return list;
    }

    /**
     * 获取指标条件
     *
     * @param form 查询条件
     * @return 指标条件列表
     */
    private static List<KeyValueDto> getMetricConditionList(HoloQueryForm form) {
        if (CollectionUtils.isEmpty(form.getConditionList())) {
            return null;
        }

        //维度条件
        List<KeyValueDto> dimConditionList = new ArrayList<>();

        //指标条件
        List<KeyValueDto> metricConditionList = new ArrayList<>();

        for (KeyValueDto dto : form.getConditionList()) {
            int type = OperatorsEnum.valueOf(dto.getSymbol()).getType();
            if (type == 1) {
                dimConditionList.add(dto);
            } else if (type == 2) {
                metricConditionList.add(dto);
            }
        }

        if (metricConditionList.size() > 0) {
            //重置维度条件，去除指标条件
            form.setConditionList(dimConditionList);
        }

        return metricConditionList;
    }

    /**
     * 维度列表对应的sql查询片段
     *
     * @param dimensionList 查询的维度列表
     * @param businessType  业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return sql查询片段
     */
    private static String getDimensionSumSql(String dataShowType, List<String> dimensionList, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            HologresShowTypeEnum hologresShowTypeEnum = HologresShowTypeEnum.valueOf(dataShowType);
            if (resultType == 1) {
                sb.append(hologresShowTypeEnum.getFormatSegment()).append(" as ").append("\"").append(hologresShowTypeEnum.getFieldName()).append("\"").append(", ");
            } else if (resultType == 2) {
                sb.append(hologresShowTypeEnum.getFormatSegment()).append(" as ").append(hologresShowTypeEnum.getFieldName()).append(", ");
            }
        }

        sb.append(getDimensionSumSql(dimensionList, businessType, resultType));

        return sb.toString();
    }

    /**
     * 维度列表对应的sql查询片段
     *
     * @param dimensionList 查询的维度列表
     * @param businessType  业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return sql查询片段
     */
    private static String getDimensionSumSql(List<String> dimensionList, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        if (CollectionUtils.isEmpty(dimensionList)) {
            return sb.toString();
        }

        for (String dim : dimensionList) {
            if (Objects.requireNonNull(BusinessEnumUtil.getDictEnumList(businessType)).contains(dim)) {
                //属性字段
                DictEnum dictEnum = Objects.requireNonNull(BusinessEnumUtil.getDictEnum(businessType, dim));

                if (resultType == 1) {
                    sb.append(dictEnum.getAliasName()).append(".").append(dictEnum.getTableFieldName()).append(" as ").append("\"").append(dictEnum.getTableFieldName()).append("\"").append(", ");
                } else if (resultType == 2) {
                    sb.append(dictEnum.getAliasName()).append(".").append(dictEnum.getTableFieldName()).append(" as ").append(dictEnum.getTableFieldName()).append(", ");
                }
            } else {
                //维度字段
                DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));

                if (resultType == 1) {
                    sb.append(dimensionEnum.getResultFieldName()).append(" as ").append("\"").append(dimensionEnum.getResultFieldName()).append("\"").append(", ");
                } else if (resultType == 2) {
                    sb.append(dimensionEnum.getResultFieldName()).append(" as ").append(dimensionEnum.getResultFieldName()).append(", ");
                }
            }
        }

        return sb.toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metricList          指标列表
     * @param calculateMetricList 计算类指标列表
     * @param combineMetricList   组合类指标列表
     * @param businessType        业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return 指标sql片段
     */
    public static String getMetricSumSql(List<Object> metricList, List<String> calculateMetricList, List<String> combineMetricList, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        //基础指标
        for (Object metric : metricList) {
            sb.append(getObjectMetricSumSql(metric, businessType, resultType));
        }

        //计算指标
        if (CollectionUtils.isNotEmpty(calculateMetricList)) {
            //去除重复的指标
            Set<String> calculateMetricSet = new HashSet<>(calculateMetricList);
            for (String metric : calculateMetricSet) {
                sb.append(getCalculateMetricSumSql(metric, businessType, resultType));
            }
        }

        //组合指标
        if (CollectionUtils.isNotEmpty(combineMetricList)) {
            //去除重复的指标
            Set<String> combineMetricSet = new HashSet<>(combineMetricList);
            for (String metric : combineMetricSet) {
                sb.append(getCombineMetricSumSql(metric, businessType, resultType));
            }
        }

        return sb.deleteCharAt(sb.length() - 1).toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param businessType 业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return 指标sql片段
     */
    private static String getObjectMetricSumSql(Object metric, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        if (metric instanceof String) {
            sb.append(getStringMetricSumSql(metric.toString(), businessType, resultType));
        } else if (metric instanceof List) {
            List<String> metricGroup = (List<String>) metric;
            for (String subMetric : metricGroup) {
                sb.append(getStringMetricSumSql(subMetric, businessType, resultType));
            }
        }

        return sb.toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param businessType 业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return 指标sql片段
     */
    private static String getStringMetricSumSql(String metric, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        String resultFieldName = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric)).getResultFieldName();
        if (resultType == 1) {
            sb.append("sum(").append(resultFieldName).append(")").append(" as ").append("\"").append(resultFieldName).append("\"").append(",");
        } else if (resultType == 2) {
            sb.append("sum(").append(resultFieldName).append(")").append(" as ").append(resultFieldName).append(",");
        }

        return sb.toString();
    }

    /**
     * 组装计算类指标sql片段
     *
     * @param metric       计算类指标
     * @param businessType 业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return 计算类指标sql片段
     */
    private static String getCalculateMetricSumSql(String metric, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        CalculateMetricEnum calculateMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, metric));
        if (resultType == 1) {
            sb.append(calculateMetricEnum.getMetricSql()).append(" as ").append("\"").append(calculateMetricEnum.getResultFieldName()).append("\"").append(",");
        } else if (resultType == 2) {
            sb.append(calculateMetricEnum.getMetricSql()).append(" as ").append(calculateMetricEnum.getResultFieldName()).append(",");
        }

        return sb.toString();
    }

    /**
     * 组装组合指标sql片段
     *
     * @param metric       组合指标
     * @param businessType 业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return 组合指标sql片段
     */
    private static String getCombineMetricSumSql(String metric, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        CombineMetricEnum combineMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, metric));
        if (resultType == 1) {
            sb.append(combineMetricEnum.getMetricSql()).append(" as ").append("\"").append(combineMetricEnum.getResultFieldName()).append("\"").append(",");
        } else if (resultType == 2) {
            sb.append(combineMetricEnum.getMetricSql()).append(" as ").append(combineMetricEnum.getResultFieldName()).append(",");
        }

        return sb.toString();
    }

    /**
     * 获取维表关联的sql语句
     * @param dimensionList 维度列表
     * @param businessType 业务线类型
     * @return sql段
     */
    private static String getDictLeftJoinSql(List<String> dimensionList, int businessType) {
        if (CollectionUtils.isEmpty(dimensionList)) {
            return null;
        }

        //属性字段列表
        List<String> dictList = new ArrayList<>();
        for (String dim : dimensionList) {
            if (Objects.requireNonNull(BusinessEnumUtil.getDictEnumList(businessType)).contains(dim)) {
                dictList.add(dim);
            }
        }

        if (CollectionUtils.isEmpty(dictList)) {
            return null;
        }

        Map<String, String> map = new HashMap<>();
        for (String dict : dictList) {
            DictEnum dictEnum = Objects.requireNonNull(BusinessEnumUtil.getDictEnum(businessType, dict));
            map.put(dictEnum.getTableName(), dict);
        }

        StringBuilder sb = new StringBuilder();
        for (String dict : map.values()) {
            DictEnum dictEnum = Objects.requireNonNull(BusinessEnumUtil.getDictEnum(businessType, dict));
            DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dictEnum.getDimensionName()));
            sb.append("left join ").append(dictEnum.getTableName()).append(" ").append(dictEnum.getAliasName())
                    .append(" on ").append("t.").append(dimensionEnum.getResultFieldName())
                    .append(" = ").append(dictEnum.getAliasName()).append(".").append(dictEnum.getPrimaryKey()).append(" ");
        }

        return sb.toString();
    }

    /**
     * 获取group by时的维度sql段
     *
     * @param dimensionList 维度列表
     * @param businessType  业务线类型
     * @return sql段
     */
    private static String getGroupBySumSql(String dataShowType, List<String> dimensionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            sb.append(HologresShowTypeEnum.valueOf(dataShowType).getFieldName()).append(", ");
        }

        if (CollectionUtils.isNotEmpty(dimensionList)) {
            for (String dim : dimensionList) {
                if (Objects.requireNonNull(BusinessEnumUtil.getDictEnumList(businessType)).contains(dim)) {
                    //属性字段
                    DictEnum dictEnum = Objects.requireNonNull(BusinessEnumUtil.getDictEnum(businessType, dim));
                    sb.append(dictEnum.getAliasName()).append(".").append(dictEnum.getTableFieldName()).append(", ");
                } else {
                    //维度字段
                    DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));
                    sb.append(dimensionEnum.getResultFieldName()).append(", ");
                }
            }
        }

        String sql = sb.toString();

        if (StringUtils.isNotBlank(sql)) {
            //去除最后的 ", "
            sql = sql.substring(0, sql.length() - 2);
        }

        return sql;
    }

    /**
     * 获取having 过滤条件
     *
     * @param metricConditionList 指标过滤条件
     * @param businessType        业务线类型
     * @return having sql
     */
    private static String getHavingSumSql(List<KeyValueDto> metricConditionList, int businessType) {
        if (CollectionUtils.isEmpty(metricConditionList)) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        for (KeyValueDto dto : metricConditionList) {
            String symbol = OperatorsEnum.valueOf(dto.getSymbol()).getSymbol();
            String value = dto.getValue().toString();

            if (Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, dto.getKey().toString()));
                sb.append("sum(").append(metricEnum.getResultFieldName()).append(")").append(symbol).append(value).append(" and ");
            } else if (Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                CalculateMetricEnum calculateMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, dto.getKey().toString()));
                sb.append(calculateMetricEnum.getMetricSql()).append(symbol).append(value).append(" and ");
            } else if (Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                CombineMetricEnum combineMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, dto.getKey().toString()));
                sb.append(combineMetricEnum.getMetricSql()).append(symbol).append(value).append(" and ");
            }
        }

        //删除最后一个多余的 "and"
        sb.delete(sb.length() - " and ".length(), sb.length());

        return sb.toString();
    }

    /**
     * 获取order by sql段
     *
     * @param orderByMetric 排序指标
     * @param businessType  业务线类型
     * @param resultType   返回字段类型（1：区分大小写 2：不区分大小写）
     * @return sql段
     */
    private static String getOrderBySql(String dataShowType, SortDto orderByMetric, int businessType, int resultType) {
        StringBuilder sb = new StringBuilder();

        if (orderByMetric == null) {
            if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
                if (resultType == 1) {
                    sb.append(" order by ").append("\"").append(HologresShowTypeEnum.valueOf(dataShowType).getFieldName()).append("\"").append(" ").append(SortEnum.ASC.getName());
                } else if (resultType == 2) {
                    sb.append(" order by ").append(HologresShowTypeEnum.valueOf(dataShowType).getFieldName()).append(" ").append(SortEnum.ASC.getName());
                }
            }

            return sb.toString();
        }

        String orderField = getOrderField(orderByMetric.getKey(), businessType);

        if (StringUtils.isBlank(orderField)) {
            return sb.toString();
        }

        if (resultType == 1) {
            //区分大小写
            orderField = "\"" + orderField + "\"";
        }

        //组装order by语句
        sb.append(" order by ")
                .append(orderField)
                .append(" ")
                .append(SortEnum.valueOf(orderByMetric.getValue()).getName());
        return sb.toString();
    }

    /**
     * 获取排序字段
     *
     * @param key          字段名称
     * @param businessType 业务线类型
     * @return 排序字段
     */
    private static String getOrderField(String key, int businessType) {
        //基础指标
        List<String> metricEnumList = Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType));
        if (metricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, key)).getResultFieldName();
        }

        //计算类指标
        List<String> calculateMetricEnumList = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType));
        if (calculateMetricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, key)).getResultFieldName();
        }

        //组合指标
        List<String> combineMetricEnumList = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType));
        if (combineMetricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, key)).getResultFieldName();
        }

        //维度字段
        List<String> dimensionEnumList = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnumList(businessType));
        if (dimensionEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, key)).getResultFieldName();
        }

        //属性字段
        List<String> dictEnumList = Objects.requireNonNull(BusinessEnumUtil.getDictEnumList(businessType));
        if (dictEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getDictEnum(businessType, key)).getTableFieldName();
        }

        return null;
    }

    /**
     * 组装所有指标对应的查询sql
     *
     * @param form         查询条件
     * @param metricList   指标列表
     * @param businessType 业务线类型
     * @return 查询sql
     * @throws Exception Exception
     */
    private static String getQueryUnionAllSql(HoloQueryForm form, List<Object> metricList, int businessType) throws Exception {
        StringBuilder sql = new StringBuilder();

        for (Object metric : metricList) {
            String querySql = SqlService.getQuerySql(form, metricList, metric, businessType);
            if (StringUtils.isNotBlank(querySql)) {
                sql.append(querySql);
                sql.append(" union all ");
            }
        }

        if (StringUtils.isNotBlank(sql.toString())) {
            //删除最后一个多余的 "union all"
            sql.delete(sql.length() - " union all ".length(), sql.length());
        }

        return sql.toString();
    }


}
